import { Trophy, Users, Calendar, Award, ChevronLeft, ChevronRight } from "lucide-react";
import { Button } from "@/components/ui/button";
import { Card, CardContent } from "@/components/ui/card";
import React, { useState, useEffect } from "react";

const HeroSection = ({ heroSlides }: { heroSlides: any }) => {
  const [currentSlide, setCurrentSlide] = useState(0);

  // Handle empty heroSlides
  if (!heroSlides || heroSlides.length === 0) {
    return (
      <section className="relative min-h-screen">
        <div className="relative w-full h-screen">
          <div className="relative flex items-center justify-center h-full bg-gradient-to-br from-primary/90 to-secondary/90 overflow-hidden">
            <div className="relative z-10 max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-20">
              <div className="text-center text-white">
                <h1 className="font-poppins font-bold text-4xl md:text-6xl leading-tight mb-4">
                  Welcome to <span className="text-green-700">The Milimus Tournament</span>
                </h1>
                <p className="text-xl text-white/90 max-w-2xl mx-auto mb-8">
                  The premier football tournament in the region
                </p>
              </div>
            </div>
          </div>
        </div>
      </section>
    );
  }

  const statsWithIcons = [
    { icon: Trophy, label: "Editions", value: "4th" },
    { icon: Users, label: "Teams", value: "32+" },
    { icon: Calendar, label: "Matches", value: "64+" },
    { icon: Award, label: "Champions", value: "4" },
  ];

  // Auto-play functionality
  useEffect(() => {
    if (heroSlides.length <= 1) return; // Don't auto-play if only one slide
    
    const interval = setInterval(() => {
      setCurrentSlide((prev) => (prev + 1) % heroSlides.length);
    }, 5000); // Change slide every 5 seconds

    return () => clearInterval(interval);
  }, [heroSlides.length]);

  const nextSlide = () => {
    setCurrentSlide((prev) => (prev + 1) % heroSlides.length);
  };

  const prevSlide = () => {
    setCurrentSlide((prev) => (prev - 1 + heroSlides.length) % heroSlides.length);
  };

  const goToSlide = (index: number) => {
    setCurrentSlide(index);
  };

  return (
    // marging top of 20 on small devices 
    <section className="relative min-h-screen">
      {/* Slides */}
      <div className="relative w-full h-screen">
        {heroSlides.map((slide: any, index: number) => (
          <div
            key={slide.id}
            className={`absolute inset-0 transition-opacity duration-1000 ease-in-out ${
              index === currentSlide ? 'opacity-100 z-10' : 'opacity-0 z-0 pointer-events-none'
            }`}
          >
            <div className="relative flex items-center justify-center h-full bg-gradient-to-br from-primary/90 to-secondary/90 overflow-hidden">
              <img
                src={slide.bgImage}
                alt={`Slide ${slide.id}`}
                className="absolute inset-0 w-full h-full object-cover opacity-100 pointer-events-none select-none scale-110"
              />
              <div className="absolute inset-0 bg-gradient-to-b from-black/60 to-black/80" />
              
              {/* Hero Content */}
              <div className="relative z-10 max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-20">
                <div className="grid lg:grid-cols-2 gap-12 items-center">
                  <div className="text-center lg:text-left space-y-8 animate-fade-in-up">
                    <div className="space-y-4">
                      <h1 className="font-poppins font-bold text-4xl md:text-6xl leading-tight text-white">
                        {slide.title.split(' ').map((word: string, i: number) => {
                          if (word === 'The' || word === 'Milimus' || word === 'Tournament') {
                            return <span key={i} className="text-green-700">{word} </span>;
                          }
                          if (word === 'Where' || word === 'Legends' || word === 'Born') {
                            return <span key={i} className="text-green-700">{word} </span>;
                          }
                          if (word === 'Championship' || word === 'Glory' || word === 'Awaits') {
                            return <span key={i} className="text-green-700">{word} </span>;
                          }
                          return <span key={i} className="text-white">{word} </span>;
                        })}
                      </h1>
                      <p className="text-xl text-white/90 max-w-2xl">
                        {slide.subtitle}
                      </p>
                    </div>

                    <div className="flex flex-col sm:flex-row gap-4 justify-center lg:justify-start">
                      <Button size="lg" className="gradient-primary text-lg px-8 py-3 rounded-full">
                        {slide.ctaPrimary}
                      </Button>
                      <Button size="lg" variant="outline" className="text-lg px-8 py-3 rounded-full bg-white/20 text-white border-white/30 hover:bg-white/30">
                        {slide.ctaSecondary}
                      </Button>
                    </div>

                    {/* Quick Stats */}
                    <div className="grid grid-cols-2 md:grid-cols-4 gap-4 pt-8">
                      {statsWithIcons.map((stat, index) => (
                        <Card key={index} className="sports-card border-none bg-white/20 backdrop-blur-sm">
                          <CardContent className="p-4 text-center">
                            <stat.icon className="h-8 w-8 text-red-700 mx-auto mb-2" />
                            <div className="font-bold text-2xl text-white">{stat.value}</div>
                            <div className="text-sm text-white/80">{stat.label}</div>
                          </CardContent>
                        </Card>
                      ))}
                    </div>
                  </div>  

                  {/* Hero Image/Illustration */}
                  <div className="relative lg:ml-auto animate-slide-up">
                    <div className="relative">
                      <Trophy className="h-32 w-32 text-red-700 mx-auto animate-bounce-gentle" />
                    </div>
                  </div>
                </div>
              </div>
            </div>
          </div>
        ))}
      </div>

      {/* Navigation Arrows */}
      <button
        onClick={prevSlide}
        className="absolute left-4 top-1/2 -translate-y-1/2 z-20 p-3 rounded-full bg-white/20 hover:bg-white/30 transition-colors text-white backdrop-blur-sm"
        aria-label="Previous slide"
      >
        <ChevronLeft className="h-6 w-6" />
      </button>
      <button
        onClick={nextSlide}
        className="absolute right-4 top-1/2 -translate-y-1/2 z-20 p-3 rounded-full bg-white/20 hover:bg-white/30 transition-colors text-white backdrop-blur-sm"
        aria-label="Next slide"
      >
        <ChevronRight className="h-6 w-6" />
      </button>

      {/* Slide Indicators */}
      <div className="absolute bottom-8 left-1/2 -translate-x-1/2 z-20 flex gap-2">
        {heroSlides.map((_: any, index: number) => (
          <button
            key={index}
            onClick={() => goToSlide(index)}
            className={`w-3 h-3 rounded-full transition-all duration-300 ${
              index === currentSlide 
                ? 'bg-red-400 scale-125' 
                : 'bg-white/50 hover:bg-white/75'
            }`}
            aria-label={`Go to slide ${index + 1}`}
          />
        ))}
      </div>
    </section>
  );
};

export default HeroSection;