import React, { useState } from 'react';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { Badge } from '@/components/ui/badge';
import { 
  FileText, 
  Download, 
  Shield, 
  CheckCircle,
  Verified,
  Eye,
  EyeOff
} from 'lucide-react';

interface Certificate {
  id: number;
  title: string;
  description: string;
  fileUrl: string;
  category: string;
  status: string;
}

const FKFComplianceShowcase: React.FC = () => {
  const [previewStates, setPreviewStates] = useState<{ [key: number]: boolean }>({});

  const certificates: Certificate[] = [
    {
      id: 1,
      title: 'FKF Compliance Certificate - G-MILIMU1',
      description: 'Official FKF compliance certificate demonstrating our adherence to all regulatory requirements and standards',
      fileUrl: '/G-MILIMU1.pdf',
      category: 'Compliance',
      status: 'Verified'
    },
    {
      id: 2,
      title: 'FKF Compliance Certificate - G-MILIMU2',
      description: 'Additional FKF compliance documentation confirming our continued commitment to excellence and regulatory standards',
      fileUrl: '/G-MILIMU2.pdf',
      category: 'Compliance',
      status: 'Verified'
    }
  ];

  const handleDownload = (fileUrl: string, title: string) => {
    const link = document.createElement('a');
    link.href = fileUrl;
    link.download = title;
    link.target = '_blank';
    document.body.appendChild(link);
    link.click();
    document.body.removeChild(link);
  };

  const handleView = (fileUrl: string) => {
    window.open(fileUrl, '_blank');
  };

  const togglePreview = (certificateId: number) => {
    setPreviewStates(prev => ({
      ...prev,
      [certificateId]: !prev[certificateId]
    }));
  };

  return (
    <section className="py-16 bg-gradient-to-br from-blue-50 via-white to-green-50">
      <div className="container mx-auto px-4">
        {/* Header */}
        <div className="text-center mb-12">
          <div className="inline-flex items-center gap-2 bg-gradient-to-r from-blue-600 to-green-600 text-white px-6 py-3 rounded-full text-sm font-semibold mb-6 shadow-lg">
            <Shield className="w-5 h-5" />
            FKF Compliance Status
          </div>
          <h2 className="text-4xl md:text-5xl font-bold text-gray-900 mb-6">
            Our FKF Compliance <span className="text-green-600">Excellence</span>
          </h2>
          <p className="text-xl text-gray-600 max-w-4xl mx-auto leading-relaxed">
            We maintain the highest standards of FKF compliance with verified certificates and continuous monitoring. 
            Our commitment to excellence ensures we meet and exceed all regulatory requirements.
          </p>
        </div>

        {/* Compliance Overview */}
        {/* <div className="mb-12">
          <Card className="border-2 border-green-200 bg-gradient-to-r from-green-50 to-blue-50 shadow-xl">
            <CardContent className="p-8">
              <div className="grid grid-cols-1 md:grid-cols-3 gap-6">
                <div className="text-center">
                  <div className="w-16 h-16 bg-green-100 rounded-full flex items-center justify-center mx-auto mb-4">
                    <CheckCircle className="w-8 h-8 text-green-600" />
                  </div>
                  <h3 className="text-2xl font-bold text-green-700 mb-2">100%</h3>
                  <p className="text-green-800 font-semibold">Compliance Rate</p>
                </div>
                
                <div className="text-center">
                  <div className="w-16 h-16 bg-blue-100 rounded-full flex items-center justify-center mx-auto mb-4">
                    <Verified className="w-8 h-8 text-blue-600" />
                  </div>
                  <h3 className="text-2xl font-bold text-blue-700 mb-2">2</h3>
                  <p className="text-blue-800 font-semibold">Active Certificates</p>
                </div>
                
                <div className="text-center">
                  <div className="w-16 h-16 bg-purple-100 rounded-full flex items-center justify-center mx-auto mb-4">
                    <Shield className="w-8 h-8 text-purple-600" />
                  </div>
                  <h3 className="text-2xl font-bold text-purple-700 mb-2">Current</h3>
                  <p className="text-purple-800 font-semibold">Status</p>
                </div>
              </div>
              
              <div className="mt-8 text-center">
                <Badge className="bg-green-100 text-green-800 border-green-200 text-lg px-6 py-3 flex items-center gap-2 mx-auto w-fit">
                  <Verified className="w-5 h-5" />
                  Fully Compliant
                </Badge>
              </div>
            </CardContent>
          </Card>
        </div> */}

        {/* Certificates Display */}
        <div className="mb-12">
          {/* <div className="text-center mb-8">
            <h3 className="text-2xl font-bold text-gray-900 mb-4">
              Our FKF Compliance Certificates
            </h3>
            <p className="text-gray-600 max-w-2xl mx-auto">
              Preview and download our official FKF compliance certificates to verify our regulatory compliance status.
            </p>
          </div> */}
          
          <div className="grid grid-cols-1 md:grid-cols-2 gap-8">
            {certificates.map((certificate) => (
              <Card key={certificate.id} className="hover:shadow-xl transition-all duration-300 border-2 hover:border-green-300">
                <CardHeader className="pb-4">
                  <div className="flex items-start justify-between mb-3">
                    <Badge className="bg-blue-100 text-blue-800 border-blue-200 text-xs">
                      {certificate.category}
                    </Badge>
                    <Badge className="bg-green-100 text-green-800 border-green-200 text-xs flex items-center gap-1">
                      <CheckCircle className="w-3 h-3" />
                      {certificate.status}
                    </Badge>
                  </div>
                  <CardTitle className="text-lg leading-tight">{certificate.title}</CardTitle>
                  <p className="text-sm text-gray-600 leading-relaxed">
                    {certificate.description}
                  </p>
                </CardHeader>
                
                <CardContent className="space-y-4">
                  {/* PDF Preview Section */}
                  <div className="relative">
                    {previewStates[certificate.id] ? (
                      <div className="w-full h-64 bg-gray-50 rounded-lg border-2 border-gray-200 overflow-hidden">
                        <iframe
                          src={`${certificate.fileUrl}#page=1&toolbar=0&navpanes=0&scrollbar=0&view=FitH`}
                          title={`Preview of ${certificate.title}`}
                          className="w-full h-full"
                          frameBorder="0"
                        />
                      </div>
                    ) : (
                      <div className="w-full h-64 bg-gray-50 rounded-lg border-2 border-dashed border-gray-300 flex items-center justify-center">
                        <div className="text-center">
                          <FileText className="w-16 h-16 mx-auto mb-4 text-gray-400" />
                          <p className="text-gray-600 font-medium">PDF Certificate</p>
                          <p className="text-sm text-gray-500">Click preview to see content</p>
                        </div>
                      </div>
                    )}
                    
                    {/* Preview Toggle Button */}
                    <Button
                      variant="outline"
                      size="sm"
                      className="absolute top-2 right-2 bg-white/90 hover:bg-white shadow-sm"
                      onClick={() => togglePreview(certificate.id)}
                    >
                      {previewStates[certificate.id] ? (
                        <>
                          <EyeOff className="w-4 h-4 mr-2" />
                          Hide
                        </>
                      ) : (
                        <>
                          <Eye className="w-4 h-4 mr-2" />
                          Preview
                        </>
                      )}
                    </Button>
                  </div>
                  
                  <div className="flex gap-3">
                    <Button
                      variant="outline"
                      className="flex-1"
                      onClick={() => handleView(certificate.fileUrl)}
                    >
                      <FileText className="w-4 h-4 mr-2" />
                      View Full PDF
                    </Button>
                    <Button
                      className="flex-1 bg-green-600 hover:bg-green-700"
                      onClick={() => handleDownload(certificate.fileUrl, certificate.title)}
                    >
                      <Download className="w-4 h-4 mr-2" />
                      Download
                    </Button>
                  </div>
                </CardContent>
              </Card>
            ))}
          </div>
        </div>

    
      </div>
    </section>
  );
};

export default FKFComplianceShowcase;
