import React, { useState, useRef, useEffect } from 'react';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { Badge } from '@/components/ui/badge';
import { 
  FileText, 
  Download, 
  Eye, 
  ExternalLink, 
  Shield, 
  CheckCircle,
  AlertTriangle,
  Info
} from 'lucide-react';

interface ComplianceDocument {
  id: string;
  title: string;
  description: string;
  category: 'required' | 'recommended' | 'optional';
  fileUrl: string;
  previewUrl?: string;
  lastUpdated: string;
  status: 'active' | 'pending' | 'expired';
}

const FKFComplianceSection: React.FC = () => {
  const [selectedDocument, setSelectedDocument] = useState<ComplianceDocument | null>(null);
  const [isPreviewOpen, setIsPreviewOpen] = useState(false);
  const previewRef = useRef<HTMLDivElement>(null);
  const [previewDimensions, setPreviewDimensions] = useState({ width: 0, height: 0 });

  // Sample compliance documents
  const complianceDocuments: ComplianceDocument[] = [
    {
      id: '1',
      title: 'FKF Affiliation Certificate',
      description: 'Official certificate proving FKF affiliation status and compliance',
      category: 'required',
      fileUrl: '/documents/fkf-affiliation-certificate.pdf',
      previewUrl: '/documents/previews/fkf-affiliation-certificate.jpg',
      lastUpdated: '2024-01-15',
      status: 'active'
    },
    {
      id: '2',
      title: 'Club Registration Document',
      description: 'Complete club registration and licensing documentation',
      category: 'required',
      fileUrl: '/documents/club-registration.pdf',
      previewUrl: '/documents/previews/club-registration.jpg',
      lastUpdated: '2024-01-10',
      status: 'active'
    },
    {
      id: '3',
      title: 'Player Eligibility Guidelines',
      description: 'Comprehensive guidelines for player eligibility and registration',
      category: 'required',
      fileUrl: '/documents/player-eligibility.pdf',
      previewUrl: '/documents/previews/player-eligibility.jpg',
      lastUpdated: '2024-01-20',
      status: 'active'
    },
    {
      id: '4',
      title: 'Safety and Medical Standards',
      description: 'Safety protocols and medical requirements for tournaments',
      category: 'recommended',
      fileUrl: '/documents/safety-standards.pdf',
      previewUrl: '/documents/previews/safety-standards.jpg',
      lastUpdated: '2024-01-05',
      status: 'active'
    },
    {
      id: '5',
      title: 'Financial Compliance Report',
      description: 'Financial reporting and compliance documentation',
      category: 'required',
      fileUrl: '/documents/financial-compliance.pdf',
      previewUrl: '/documents/previews/financial-compliance.jpg',
      lastUpdated: '2024-01-12',
      status: 'pending'
    }
  ];

  useEffect(() => {
    const updatePreviewDimensions = () => {
      if (previewRef.current) {
        const rect = previewRef.current.getBoundingClientRect();
        setPreviewDimensions({
          width: rect.width,
          height: rect.height
        });
      }
    };

    updatePreviewDimensions();
    window.addEventListener('resize', updatePreviewDimensions);
    
    return () => window.removeEventListener('resize', updatePreviewDimensions);
  }, [isPreviewOpen]);

  const openPreview = (document: ComplianceDocument) => {
    setSelectedDocument(document);
    setIsPreviewOpen(true);
  };

  const closePreview = () => {
    setIsPreviewOpen(false);
    setSelectedDocument(null);
  };

  const getCategoryColor = (category: string) => {
    switch (category) {
      case 'required':
        return 'bg-red-100 text-red-800 border-red-200';
      case 'recommended':
        return 'bg-yellow-100 text-yellow-800 border-yellow-200';
      case 'optional':
        return 'bg-blue-100 text-blue-800 border-blue-200';
      default:
        return 'bg-gray-100 text-gray-800 border-gray-200';
    }
  };

  const getStatusColor = (status: string) => {
    switch (status) {
      case 'active':
        return 'bg-green-100 text-green-800 border-green-200';
      case 'pending':
        return 'bg-yellow-100 text-yellow-800 border-yellow-200';
      case 'expired':
        return 'bg-red-100 text-red-800 border-red-200';
      default:
        return 'bg-gray-100 text-gray-800 border-gray-200';
    }
  };

  const getStatusIcon = (status: string) => {
    switch (status) {
      case 'active':
        return <CheckCircle className="w-4 h-4" />;
      case 'pending':
        return <AlertTriangle className="w-4 h-4" />;
      case 'expired':
        return <AlertTriangle className="w-4 h-4" />;
      default:
        return <Info className="w-4 h-4" />;
    }
  };

  return (
    <section className="py-12 bg-gradient-to-br from-blue-50 via-white to-green-50">
      <div className="container mx-auto px-4">
        {/* Header */}
        <div className="text-center mb-12">
          <div className="inline-flex items-center gap-2 bg-blue-100 text-blue-800 px-4 py-2 rounded-full text-sm font-medium mb-4">
            <Shield className="w-4 h-4" />
            FKF Compliance
          </div>
          <h2 className="text-3xl md:text-4xl font-bold text-gray-900 mb-4">
            FKF Affiliation Compliance
          </h2>
          <p className="text-lg text-gray-600 max-w-3xl mx-auto">
            Ensure your team meets all FKF affiliation requirements with our comprehensive compliance documentation.
            Review, download, and stay updated with the latest standards.
          </p>
        </div>

        {/* Compliance Overview */}
        <div className="grid grid-cols-1 md:grid-cols-3 gap-6 mb-8">
          <Card className="text-center border-blue-200 bg-blue-50">
            <CardContent className="pt-6">
              <div className="w-12 h-12 bg-blue-100 rounded-full flex items-center justify-center mx-auto mb-3">
                <CheckCircle className="w-6 h-6 text-blue-600" />
              </div>
              <h3 className="font-semibold text-blue-900 mb-1">Required Documents</h3>
              <p className="text-2xl font-bold text-blue-700">
                {complianceDocuments.filter(doc => doc.category === 'required').length}
              </p>
            </CardContent>
          </Card>

          <Card className="text-center border-yellow-200 bg-yellow-50">
            <CardContent className="pt-6">
              <div className="w-12 h-12 bg-yellow-100 rounded-full flex items-center justify-center mx-auto mb-3">
                <AlertTriangle className="w-6 h-6 text-yellow-600" />
              </div>
              <h3 className="font-semibold text-yellow-900 mb-1">Pending Review</h3>
              <p className="text-2xl font-bold text-yellow-700">
                {complianceDocuments.filter(doc => doc.status === 'pending').length}
              </p>
            </CardContent>
          </Card>

          <Card className="text-center border-green-200 bg-green-50">
            <CardContent className="pt-6">
              <div className="w-12 h-12 bg-green-100 rounded-full flex items-center justify-center mx-auto mb-3">
                <Shield className="w-6 h-6 text-green-600" />
              </div>
              <h3 className="font-semibold text-green-900 mb-1">Active Status</h3>
              <p className="text-2xl font-bold text-green-700">
                {complianceDocuments.filter(doc => doc.status === 'active').length}
              </p>
            </CardContent>
          </Card>
        </div>

        {/* Documents Grid */}
        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6 mb-8">
          {complianceDocuments.map((document) => (
            <Card key={document.id} className="hover:shadow-lg transition-shadow duration-200">
              <CardHeader className="pb-3">
                <div className="flex items-start justify-between mb-2">
                  <Badge className={`${getCategoryColor(document.category)} text-xs`}>
                    {document.category.charAt(0).toUpperCase() + document.category.slice(1)}
                  </Badge>
                  <Badge className={`${getStatusColor(document.status)} text-xs flex items-center gap-1`}>
                    {getStatusIcon(document.status)}
                    {document.status}
                  </Badge>
                </div>
                <CardTitle className="text-lg">{document.title}</CardTitle>
                <CardDescription className="text-sm">
                  {document.description}
                </CardDescription>
              </CardHeader>
              <CardContent className="pt-0">
                <div className="flex items-center justify-between text-xs text-gray-500 mb-4">
                  <span>Updated: {new Date(document.lastUpdated).toLocaleDateString()}</span>
                </div>
                
                <div className="flex gap-2">
                  <Button
                    variant="outline"
                    size="sm"
                    className="flex-1"
                    onClick={() => openPreview(document)}
                  >
                    <Eye className="w-4 h-4 mr-2" />
                    Preview
                  </Button>
                  <Button
                    variant="outline"
                    size="sm"
                    className="flex-1"
                    onClick={() => window.open(document.fileUrl, '_blank')}
                  >
                    <Download className="w-4 h-4 mr-2" />
                    Download
                  </Button>
                </div>
              </CardContent>
            </Card>
          ))}
        </div>

        {/* Important Notice */}
        <Card className="border-orange-200 bg-orange-50">
          <CardContent className="pt-6">
            <div className="flex items-start gap-3">
              <AlertTriangle className="w-6 h-6 text-orange-600 mt-1" />
              <div>
                <h3 className="font-semibold text-orange-900 mb-2">Important Notice</h3>
                <p className="text-orange-800 text-sm leading-relaxed">
                  All teams must maintain valid FKF affiliation status to participate in official tournaments. 
                  Please ensure all required documents are up-to-date and properly submitted. 
                  Contact our compliance team if you need assistance with any documentation requirements.
                </p>
              </div>
            </div>
          </CardContent>
        </Card>
      </div>

      {/* Floating Document Preview Modal */}
      {isPreviewOpen && selectedDocument && (
        <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50 p-4">
          <div 
            ref={previewRef}
            className="bg-white rounded-lg shadow-2xl max-w-4xl w-full max-h-[90vh] overflow-hidden"
            style={{
              width: Math.min(previewDimensions.width || 800, window.innerWidth - 32),
              height: Math.min(previewDimensions.height || 600, window.innerHeight - 32)
            }}
          >
            {/* Preview Header */}
            <div className="flex items-center justify-between p-4 border-b bg-gray-50">
              <div>
                <h3 className="font-semibold text-lg">{selectedDocument.title}</h3>
                <p className="text-sm text-gray-600">{selectedDocument.description}</p>
              </div>
              <div className="flex gap-2">
                <Button
                  variant="outline"
                  size="sm"
                  onClick={() => window.open(selectedDocument.fileUrl, '_blank')}
                >
                  <Download className="w-4 h-4 mr-2" />
                  Download
                </Button>
                <Button
                  variant="outline"
                  size="sm"
                  onClick={closePreview}
                >
                  Close
                </Button>
              </div>
            </div>

            {/* Preview Content */}
            <div className="p-4 overflow-auto" style={{ height: 'calc(100% - 80px)' }}>
              {selectedDocument.previewUrl ? (
                <div className="w-full h-full flex items-center justify-center">
                  <img
                    src={selectedDocument.previewUrl}
                    alt={`Preview of ${selectedDocument.title}`}
                    className="max-w-full max-h-full object-contain rounded border"
                    style={{
                      width: Math.min(previewDimensions.width - 64, 800),
                      height: Math.min(previewDimensions.height - 120, 600)
                    }}
                  />
                </div>
              ) : (
                <div className="w-full h-full flex items-center justify-center text-gray-500">
                  <div className="text-center">
                    <FileText className="w-16 h-16 mx-auto mb-4 text-gray-300" />
                    <p>Preview not available</p>
                    <p className="text-sm">Click Download to view the full document</p>
                  </div>
                </div>
              )}
            </div>
          </div>
        </div>
      )}
    </section>
  );
};

export default FKFComplianceSection;
