import React from 'react';
import { Card, CardContent } from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { CheckCircle, Circle, ArrowLeft, ArrowRight } from 'lucide-react';

interface Step {
  id: string;
  title: string;
  description: string;
  icon: React.ReactNode;
}

interface RegistrationStepsProps {
  steps: Step[];
  currentStep: string;
  onStepChange: (step: string) => void;
  onNext: () => void;
  onPrevious: () => void;
  onSubmit: () => void;
  isSubmitting: boolean;
  canProceed: boolean;
  isLastStep: boolean;
  children?: React.ReactNode;
}

const RegistrationSteps: React.FC<RegistrationStepsProps> = ({
  steps,
  currentStep,
  onStepChange,
  onNext,
  onPrevious,
  onSubmit,
  isSubmitting,
  canProceed,
  isLastStep,
  children,
}) => {
  const currentStepIndex = steps.findIndex(step => step.id === currentStep);
  const progress = ((currentStepIndex + 1) / steps.length) * 100;

  return (
    <div className="min-h-screen bg-gradient-to-br from-green-50 to-gray-50 py-8">
      <div className="container mx-auto px-4 max-w-5xl">
        {/* Header */}
        <div className="text-center mb-8">
          <div className="w-20 h-20 bg-green-600 rounded-full flex items-center justify-center mx-auto mb-4">
            <span className="text-white text-3xl font-bold">M</span>
          </div>
          <h1 className="text-3xl font-bold text-gray-900">Team Registration</h1>
          <p className="text-gray-600 mt-2">Register your team for the Milimus Tournament</p>
        </div>

        {/* Enhanced Progress Indicator */}
        <Card className="mb-8 p-6">
          <div className="space-y-4">
            {/* Step Indicators */}
            <div className="flex items-center justify-between">
              {steps.map((step, index) => {
                const isCompleted = index < currentStepIndex;
                const isCurrent = index === currentStepIndex;
                const isUpcoming = index > currentStepIndex;
                
                return (
                  <div key={step.id} className="flex items-center">
                    <button
                      onClick={() => onStepChange(step.id)}
                      className={`flex items-center justify-center w-12 h-12 rounded-full border-2 transition-all duration-300 transform hover:scale-105 ${
                        isCompleted
                          ? 'border-green-600 bg-green-600 text-white shadow-lg shadow-green-200'
                          : isCurrent
                          ? 'border-green-500 bg-green-50 text-green-600 shadow-md shadow-green-100 ring-2 ring-green-200'
                          : 'border-gray-300 bg-white text-gray-400 hover:border-gray-400 hover:bg-gray-50'
                      }`}
                    >
                      {isCompleted ? (
                        <CheckCircle className="w-6 h-6" />
                      ) : (
                        <span className={`text-sm font-bold ${
                          isCurrent ? 'text-green-600' : 'text-gray-400'
                        }`}>
                          {index + 1}
                        </span>
                      )}
                    </button>
                    {index < steps.length - 1 && (
                      <div
                        className={`w-16 md:w-24 h-1 mx-2 rounded-full transition-all duration-300 ${
                          isCompleted 
                            ? 'bg-green-600' 
                            : isCurrent
                            ? 'bg-gradient-to-r from-green-600 to-gray-300'
                            : 'bg-gray-300'
                        }`}
                      />
                    )}
                  </div>
                );
              })}
            </div>

            {/* Progress Bar */}
            <div className="w-full bg-gray-200 rounded-full h-2">
              <div
                className="bg-green-600 h-2 rounded-full transition-all duration-300 ease-out"
                style={{ width: `${progress}%` }}
              />
            </div>

            {/* Step Labels */}
            <div className="grid grid-cols-5 gap-2 text-center">
              {steps.map((step, index) => {
                const isCompleted = index < currentStepIndex;
                const isCurrent = index === currentStepIndex;
                const isUpcoming = index > currentStepIndex;
                
                return (
                  <div key={step.id} className="space-y-1">
                    <div className={`text-sm font-medium transition-all duration-200 ${
                      isCompleted
                        ? 'text-green-700 font-semibold'
                        : isCurrent
                        ? 'text-green-600 font-bold'
                        : 'text-gray-500'
                    }`}>
                      {step.title}
                    </div>
                    <div className={`text-xs hidden md:block transition-all duration-200 ${
                      isCompleted
                        ? 'text-green-600'
                        : isCurrent
                        ? 'text-green-500'
                        : 'text-gray-400'
                    }`}>
                      {step.description}
                    </div>
                  </div>
                );
              })}
            </div>
          </div>
        </Card>

        {/* Step Content */}
        <Card className="p-8 shadow-xl">
          <div className="mb-6">
            <div className="flex items-center gap-3 mb-2">
              <div className="p-2 bg-green-100 rounded-lg">
                {steps[currentStepIndex]?.icon}
              </div>
              <div>
                <h2 className="text-2xl font-bold text-gray-900">
                  {steps[currentStepIndex]?.title}
                </h2>
                <div className="flex items-center gap-2 mt-1">
                  <span className="text-sm text-green-600 font-medium">
                    Step {currentStepIndex + 1} of {steps.length}
                  </span>
                  <div className="w-2 h-2 bg-green-500 rounded-full animate-pulse"></div>
                </div>
              </div>
            </div>
            <p className="text-gray-600 ml-14">
              {steps[currentStepIndex]?.description}
            </p>
          </div>

          {/* Form Content */}
          <div className="mb-8">
            {children}
          </div>

          {/* Navigation */}
          <div className="flex justify-between pt-6 border-t border-gray-200">
            <Button
              type="button"
              variant="outline"
              onClick={onPrevious}
              disabled={currentStepIndex === 0}
              className="min-w-[120px]"
            >
              <ArrowLeft className="w-4 h-4 mr-2" />
              Previous
            </Button>

            <div className="flex items-center space-x-3">
              <span className="text-sm text-gray-500">
                Step {currentStepIndex + 1} of {steps.length}
              </span>
              
              {isLastStep ? (
                <Button
                  type="button"
                  onClick={onSubmit}
                  disabled={!canProceed || isSubmitting}
                  className="min-w-[140px] bg-green-600 hover:bg-green-700"
                >
                  {isSubmitting ? 'Submitting...' : 'Complete Registration'}
                </Button>
              ) : (
                <Button
                  type="button"
                  onClick={onNext}
                  disabled={!canProceed}
                  className="min-w-[120px] bg-green-600 hover:bg-green-700"
                >
                  Next
                  <ArrowRight className="w-4 h-4 ml-2" />
                </Button>
              )}
            </div>
          </div>
        </Card>

        {/* Helper Text */}
        <div className="text-center mt-6">
          <p className="text-sm text-gray-500">
            Need help? Contact us at{' '}
            <a href="mailto:support@milimus.com" className="text-green-600 hover:text-green-700">
              support@milimus.com
            </a>
          </p>
        </div>
      </div>
    </div>
  );
};

export default RegistrationSteps;