import React, { useState } from 'react';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Badge } from '@/components/ui/badge';
import { Search, Trophy, Plus, Calendar, DollarSign } from 'lucide-react';

interface Tournament {
  id: number;
  name: string;
  description: string;
  entry_fee: number;
  currency: string;
  start_date?: string;
  end_date?: string;
  registration_end?: string;
  is_free: boolean;
  enable_categories?: boolean;
  category_fees?: Record<string, number>;
  available_categories?: string[];
}

interface TournamentSelectionStepProps {
  tournaments: Tournament[];
  onSelectTournament: (tournament: Tournament) => void;
  onCreateNewTeam: () => void;
  selectedTournament?: Tournament;
  onNext?: () => void;
}

const TournamentSelectionStep: React.FC<TournamentSelectionStepProps> = ({
  tournaments,
  onSelectTournament,
  onCreateNewTeam,
  selectedTournament,
  onNext
}) => {
  const [searchQuery, setSearchQuery] = useState('');

  const filteredTournaments = tournaments.filter(tournament =>
    tournament.name.toLowerCase().includes(searchQuery.toLowerCase()) ||
    tournament.description.toLowerCase().includes(searchQuery.toLowerCase())
  );

  return (
    <div className="space-y-6">
      <div className="text-start flex items-center gap-2">
        <Trophy className="w-5 h-5 text-green-600" />
        <h2 className="text-lg font-bold text-gray-900 mb-2">Tournament Selection</h2>
      </div>

      {/* Search */}
      <div className="relative">
        <Search className="absolute left-3 top-1/2 transform -translate-y-1/2 text-gray-400 h-4 w-4" />
        <Input
          placeholder="Search tournaments..."
          value={searchQuery}
          onChange={(e) => setSearchQuery(e.target.value)}
          className="pl-10"
        />
      </div>

      {/* Available Tournaments */}
      {filteredTournaments.length > 0 && (
        <div className="space-y-4">
          <h3 className="text-lg font-semibold text-gray-900">Available Tournaments</h3>
          <div className="grid gap-4">
            {filteredTournaments.map((tournament) => (
              <Card
                key={tournament.id}
                className={`cursor-pointer transition-all hover:shadow-md ${
                  selectedTournament?.id === tournament.id ? 'ring-2 ring-green-500 bg-green-50' : ''
                }`}
                onClick={() => onSelectTournament(tournament)}
              >
                <CardHeader className="pb-3">
                  <div className="flex items-center justify-between">
                    <CardTitle className="text-lg">{tournament.name}</CardTitle>
                    {/* <div className="flex items-center gap-2">
                      {tournament.is_free ? (
                        <Badge variant="secondary" className="bg-green-100 text-green-800">
                          Free
                        </Badge>
                      ) : (
                        <Badge variant="secondary" className="bg-orange-100 text-orange-800">
                          {tournament.currency} {tournament.entry_fee.toLocaleString()}
                        </Badge>
                      )}
                    </div> */}
                  </div>
                </CardHeader>
                <CardContent>
                  <p className="text-gray-600 text-sm mb-3">{tournament.description}</p>
                  
                  <div className="grid grid-cols-1 md:grid-cols-2 gap-2 text-xs text-gray-500">
                    {tournament.start_date && tournament.end_date && (
                      <div className="flex items-center gap-1">
                        <Calendar className="w-3 h-3" />
                        <span>{tournament.start_date} - {tournament.end_date}</span>
                      </div>
                    )}
                    {tournament.registration_end && (
                      <div className="flex items-center gap-1">
                        <Calendar className="w-3 h-3" />
                        <span>Registration ends: {tournament.registration_end}</span>
                      </div>
                    )}
                  </div>

                  {/* Category Fees Display */}
                  {tournament.enable_categories && tournament.category_fees && (
                    <div className="mt-3 border-t pt-3">
                      <p className="text-xs font-semibold text-gray-700 mb-2">📋 Category Registration Fees:</p>
                      <div className="grid grid-cols-2 gap-2">
                        {tournament.category_fees.men > 0 && (
                          <div className="flex items-center justify-between text-xs bg-blue-50 px-2 py-1 rounded">
                            <span className="flex items-center gap-1">
                              <span>👨</span>
                              <span className="font-medium">Men</span>
                            </span>
                            <span className="font-bold">{tournament.currency} {tournament.category_fees.men.toLocaleString()}</span>
                          </div>
                        )}
                        {tournament.category_fees.women > 0 && (
                          <div className="flex items-center justify-between text-xs bg-pink-50 px-2 py-1 rounded">
                            <span className="flex items-center gap-1">
                              <span>👩</span>
                              <span className="font-medium">Women</span>
                            </span>
                            <span className="font-bold">{tournament.currency} {tournament.category_fees.women.toLocaleString()}</span>
                          </div>
                        )}
                        {tournament.category_fees.youth_male !== undefined && (
                          <div className="flex items-center justify-between text-xs bg-purple-50 px-2 py-1 rounded">
                            <span className="flex items-center gap-1">
                              <span>🧑</span>
                              <span className="font-medium">Youth M</span>
                            </span>
                            <span className="font-bold">{tournament.category_fees.youth_male > 0 ? `${tournament.currency} ${tournament.category_fees.youth_male.toLocaleString()}` : 'FREE'}</span>
                          </div>
                        )}
                        {tournament.category_fees.youth_female !== undefined && (
                          <div className="flex items-center justify-between text-xs bg-purple-100 px-2 py-1 rounded">
                            <span className="flex items-center gap-1">
                              <span>👧</span>
                              <span className="font-medium">Youth F</span>
                            </span>
                            <span className="font-bold">{tournament.category_fees.youth_female > 0 ? `${tournament.currency} ${tournament.category_fees.youth_female.toLocaleString()}` : 'FREE'}</span>
                          </div>
                        )}
                        {tournament.category_fees.children !== undefined && (
                          <div className="flex items-center justify-between text-xs bg-green-50 px-2 py-1 rounded">
                            <span className="flex items-center gap-1">
                              <span>🧒</span>
                              <span className="font-medium">Children</span>
                            </span>
                            <span className="font-bold">{tournament.category_fees.children > 0 ? `${tournament.currency} ${tournament.category_fees.children.toLocaleString()}` : 'FREE'}</span>
                          </div>
                        )}
                      </div>
                      <p className="text-xs text-gray-500 mt-2 italic">
                        Select your team category in the next step
                      </p>
                    </div>
                  )}
                  
                  {selectedTournament?.id === tournament.id && (
                    <div className="mt-3 p-2 bg-green-100 rounded-md">
                      <p className="text-blue-800 text-sm font-medium">
                        ✓ Selected for registration
                      </p>
                    </div>
                  )}
                </CardContent>
              </Card>
            ))}
          </div>
        </div>
      )}

      

      {/* No tournaments found */}
      {filteredTournaments.length === 0 && searchQuery && (
        <div className="text-center py-8">
          <Trophy className="h-12 w-12 text-gray-300 mx-auto mb-4" />
          <p className="text-gray-500">No tournaments found matching your search.</p>
          <p className="text-gray-400 text-sm mt-2">
            Try a different search term.
          </p>
        </div>
      )}

      {/* No tournaments available */}
      {tournaments.length === 0 && (
        <div className="text-center py-8">
          <Trophy className="h-12 w-12 text-gray-300 mx-auto mb-4" />
          <p className="text-gray-500">No tournaments available for registration.</p>
          <p className="text-gray-400 text-sm mt-2">
            Please check back later for available tournaments.
          </p>
        </div>
      )}

      {/* Navigation */}
      {/* {selectedTournament && onNext && (
        <div className="flex justify-end mt-6">
          <Button 
            type="button" 
            onClick={onNext}
            className="bg-green-600 hover:bg-green-700"
          >
            Continue with {selectedTournament.name}
          </Button>
        </div>
      )} */}
    </div>
  );
};

export default TournamentSelectionStep; 