import React from 'react';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card';
import { Users, Building2, MapPin, Calendar } from 'lucide-react';

interface TeamInfoStepProps {
  data: any;
  setData: (key: string, value: any) => void;
  errors: any;
}

const TeamInfoStep: React.FC<TeamInfoStepProps> = ({ data, setData, errors }) => {
  const currentYear = new Date().getFullYear();
  const years = Array.from({ length: 5 }, (_, i) => currentYear + i);

  return (
    <div className="space-y-6">
      <Card>
        <CardHeader>
          <CardTitle className="flex items-center gap-2">
            <Building2 className="w-5 h-5 text-green-600" />
            Team Details
          </CardTitle>
          <CardDescription>
            Provide basic information about your team
          </CardDescription>
        </CardHeader>
        <CardContent className="space-y-4">
          <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
            <div>
              <Label htmlFor="team_name">Team Name *</Label>
              <Input
                id="team_name"
                value={data.team.name}
                onChange={(e) => setData('team', { ...data.team, name: e.target.value })}
                placeholder="Enter your team name"
                className={errors['team.name'] ? 'border-red-500' : ''}
              />
              {errors['team.name'] && (
                <p className="text-red-600 text-sm mt-1">{errors['team.name']}</p>
              )}
            </div>
            
            <div>
              <Label htmlFor="team_email">Team Email *</Label>
              <Input
                id="team_email"
                type="email"
                value={data.team.email}
                onChange={(e) => setData('team', { ...data.team, email: e.target.value })}
                placeholder="team@example.com"
                className={errors['team.email'] ? 'border-red-500' : ''}
              />
              {errors['team.email'] && (
                <p className="text-red-600 text-sm mt-1">{errors['team.email']}</p>
              )}
            </div>
            
            <div>
              <Label htmlFor="team_phone">Team Phone *</Label>
              <Input
                id="team_phone"
                type="tel"
                value={data.team.phone}
                onChange={(e) => setData('team', { ...data.team, phone: e.target.value })}
                placeholder="07XX XXX XXX"
                className={errors['team.phone'] ? 'border-red-500' : ''}
              />
              {errors['team.phone'] && (
                <p className="text-red-600 text-sm mt-1">{errors['team.phone']}</p>
              )}
            </div>
            
            <div>
              <Label htmlFor="team_location">Location *</Label>
              <div className="relative">
                <MapPin className="absolute left-3 top-1/2 transform -translate-y-1/2 text-gray-400 w-4 h-4" />
                <Input
                  id="team_location"
                  value={data.team.location}
                  onChange={(e) => setData('team', { ...data.team, location: e.target.value })}
                  placeholder="City, State/Province, Country"
                  className={`pl-10 ${errors['team.location'] ? 'border-red-500' : ''}`}
                />
              </div>
              {errors['team.location'] && (
                <p className="text-red-600 text-sm mt-1">{errors['team.location']}</p>
              )}
            </div>
            
            <div>
              <Label htmlFor="team_coach">Head Coach *</Label>
              <Input
                id="team_coach"
                value={data.team.coach}
                onChange={(e) => setData('team', { ...data.team, coach: e.target.value })}
                placeholder="Coach's full name"
                className={errors['team.coach'] ? 'border-red-500' : ''}
              />
              {errors['team.coach'] && (
                <p className="text-red-600 text-sm mt-1">{errors['team.coach']}</p>
              )}
            </div>
            
            <div>
              <Label htmlFor="team_year">Tournament Year *</Label>
              <Select
                value={String(data.team.year)}
                onValueChange={(value) => setData('team', { ...data.team, year: parseInt(value) })}
              >
                <SelectTrigger className={errors['team.year'] ? 'border-red-500' : ''}>
                  <SelectValue placeholder="Select year" />
                </SelectTrigger>
                <SelectContent>
                  {years.map(year => (
                    <SelectItem key={year} value={String(year)}>
                      <div className="flex items-center gap-2">
                        <Calendar className="w-4 h-4" />
                        {year}
                      </div>
                    </SelectItem>
                  ))}
                </SelectContent>
              </Select>
              {errors['team.year'] && (
                <p className="text-red-600 text-sm mt-1">{errors['team.year']}</p>
              )}
            </div>
          </div>
        </CardContent>
      </Card>

      {/* Registration Info */}
      <Card className="bg-green-50 border-green-200">
        <CardContent className="p-4">
          <div className="flex items-start gap-3">
            <Users className="w-5 h-5 text-green-600 mt-0.5" />
            <div>
              <h4 className="font-medium text-green-800 mb-1">Registration Requirements</h4>
              <ul className="text-sm text-green-700 space-y-1">
                <li>• Team must have 11-23 registered players</li>
                <li>• All players must be verified before tournament start</li>
                <li>• Team captain will receive all official communications</li>
                <li>• Registration fee must be paid within 48 hours</li>
              </ul>
            </div>
          </div>
        </CardContent>
      </Card>
    </div>
  );
};

export default TeamInfoStep;