import React, { useState } from 'react';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card';
import { Badge } from '@/components/ui/badge';
import { Button } from '@/components/ui/button';
import { Checkbox } from '@/components/ui/checkbox';
import { Label } from '@/components/ui/label';
import { 
  FileCheck, 
  Building2, 
  Crown, 
  UserCheck, 
  Briefcase, 
  Mail, 
  Phone, 
  MapPin,
  Users,
  Award,
  AlertTriangle,
  CheckCircle
} from 'lucide-react';

interface ReviewStepProps {
  data: any;
  setData: (key: string, value: any) => void;
  errors: any;
  termsAccepted: boolean;
  setTermsAccepted: (accepted: boolean) => void;
  registration_fee: number;
}

const ReviewStep: React.FC<ReviewStepProps> = ({ 
  data, 
  setData, 
  errors, 
  termsAccepted, 
  setTermsAccepted,
  registration_fee
}) => {
  const [showTermsModal, setShowTermsModal] = useState(false);

  // Validation checks
  const validationChecks = [
    {
      label: 'Team information completed',
      valid: data.team.name && data.team.email && data.team.phone && data.team.location && data.team.coach,
      step: 'team'
    },
    {
      label: 'Team captain assigned',
      valid: data.captain.name && data.captain.email && data.captain.phone,
      step: 'captain'
    },
    {
      label: 'Minimum 11 players registered',
      valid: data.players.length >= 11 && data.players.every((p: any) => p.name && p.email && p.phone && p.position),
      step: 'players'
    },
    {
      label: 'At least one official assigned',
      valid: data.officials.length >= 1 && data.officials.every((o: any) => o.name && o.email && o.phone && o.role),
      step: 'officials'
    }
  ];

  const allValid = validationChecks.every(check => check.valid);
  const completedPlayers = data.players.filter((p: any) => p.name && p.email && p.phone && p.position);
  const completedOfficials = data.officials.filter((o: any) => o.name && o.email && o.phone && o.role);

  return (
    <div className="space-y-6">
      {/* Validation Status */}
      <Card className={`${allValid ? 'bg-green-50 border-green-200' : 'bg-red-50 border-red-200'}`}>
        <CardContent className="p-4">
          <h4 className={`font-medium mb-3 flex items-center gap-2 ${allValid ? 'text-green-800' : 'text-red-800'}`}>
            {allValid ? (
              <CheckCircle className="w-5 h-5" />
            ) : (
              <AlertTriangle className="w-5 h-5" />
            )}
            Registration Status
          </h4>
          <div className="space-y-2">
            {validationChecks.map((check, index) => (
              <div key={index} className="flex items-center gap-2">
                {check.valid ? (
                  <CheckCircle className="w-4 h-4 text-green-600" />
                ) : (
                  <AlertTriangle className="w-4 h-4 text-red-600" />
                )}
                <span className={`text-sm ${check.valid ? 'text-green-700' : 'text-red-700'}`}>
                  {check.label}
                </span>
              </div>
            ))}
          </div>
        </CardContent>
      </Card>

      {/* Team Summary */}
      <Card>
        <CardHeader>
          <CardTitle className="flex items-center gap-2">
            <Building2 className="w-5 h-5 text-green-600" />
            Team Summary
          </CardTitle>
        </CardHeader>
        <CardContent className="space-y-4">
          <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
            <div>
              <Label className="text-sm text-gray-600">Team Name</Label>
              <p className="font-medium">{data.team.name}</p>
            </div>
            <div>
              <Label className="text-sm text-gray-600">Tournament Year</Label>
              <p className="font-medium">{data.team.year}</p>
            </div>
            <div>
              <Label className="text-sm text-gray-600">Head Coach</Label>
              <p className="font-medium">{data.team.coach}</p>
            </div>
            <div>
              <Label className="text-sm text-gray-600">Location</Label>
              <div className="flex items-center gap-1">
                <MapPin className="w-4 h-4 text-gray-400" />
                <span>{data.team.location}</span>
              </div>
            </div>
            <div>
              <Label className="text-sm text-gray-600">Contact Email</Label>
              <div className="flex items-center gap-1">
                <Mail className="w-4 h-4 text-gray-400" />
                <span>{data.team.email}</span>
              </div>
            </div>
            <div>
              <Label className="text-sm text-gray-600">Contact Phone</Label>
              <div className="flex items-center gap-1">
                <Phone className="w-4 h-4 text-gray-400" />
                <span>{data.team.phone}</span>
              </div>
            </div>
          </div>
        </CardContent>
      </Card>

      {/* Captain Summary */}
      <Card>
        <CardHeader>
          <CardTitle className="flex items-center gap-2">
            <Crown className="w-5 h-5 text-yellow-600" />
            Team Captain
          </CardTitle>
        </CardHeader>
        <CardContent>
          <div className="flex items-center gap-3 p-4 bg-yellow-50 rounded-lg">
            <div className="w-12 h-12 bg-yellow-100 rounded-full flex items-center justify-center">
              <Crown className="w-6 h-6 text-yellow-600" />
            </div>
            <div>
              <h4 className="font-medium">{data.captain.name}</h4>
              <div className="flex items-center gap-4 text-sm text-gray-600">
                <div className="flex items-center gap-1">
                  <Mail className="w-3 h-3" />
                  {data.captain.email}
                </div>
                <div className="flex items-center gap-1">
                  <Phone className="w-3 h-3" />
                  {data.captain.phone}
                </div>
              </div>
              {data.captain.existing_user_id && (
                <Badge variant="secondary" className="mt-1 text-xs">
                  Existing User Account
                </Badge>
              )}
            </div>
          </div>
        </CardContent>
      </Card>

      {/* Players Summary */}
      <Card>
        <CardHeader>
          <CardTitle className="flex items-center gap-2">
            <UserCheck className="w-5 h-5 text-green-600" />
            Players ({completedPlayers.length})
          </CardTitle>
        </CardHeader>
        <CardContent>
          {completedPlayers.length > 0 ? (
            <div className="grid gap-2 max-h-64 overflow-y-auto">
              {completedPlayers.map((player: any, index: number) => (
                <div key={index} className="flex items-center justify-between p-3 bg-gray-50 rounded-lg">
                  <div className="flex items-center gap-3">
                    <div className="w-8 h-8 bg-green-100 rounded-full flex items-center justify-center">
                      <span className="text-sm font-medium">#{index + 1}</span>
                    </div>
                    <div>
                      <h5 className="font-medium">{player.name}</h5>
                      <div className="flex items-center gap-2 text-sm text-gray-600">
                        <span>{player.position}</span>
                        {player.jersey_number && <span>#{player.jersey_number}</span>}
                      </div>
                    </div>
                  </div>
                  {player.existing_user_id && (
                    <Badge variant="secondary" className="text-xs">
                      Existing User
                    </Badge>
                  )}
                </div>
              ))}
            </div>
          ) : (
            <p className="text-gray-500 text-center py-4">No players added yet</p>
          )}
        </CardContent>
      </Card>

      {/* Officials Summary */}
      <Card>
        <CardHeader>
          <CardTitle className="flex items-center gap-2">
            <Briefcase className="w-5 h-5 text-blue-600" />
            Officials ({completedOfficials.length})
          </CardTitle>
        </CardHeader>
        <CardContent>
          {completedOfficials.length > 0 ? (
            <div className="grid gap-2">
              {completedOfficials.map((official: any, index: number) => (
                <div key={index} className="flex items-center justify-between p-3 bg-gray-50 rounded-lg">
                  <div className="flex items-center gap-3">
                    <div className="w-8 h-8 bg-blue-100 rounded-full flex items-center justify-center">
                      <Briefcase className="w-4 h-4 text-blue-600" />
                    </div>
                    <div>
                      <h5 className="font-medium">{official.name}</h5>
                      <div className="flex items-center gap-2 text-sm text-gray-600">
                        <span>{official.role}</span>
                        {official.qualifications && (
                          <Badge variant="outline" className="text-xs">
                            <Award className="w-3 h-3 mr-1" />
                            Qualified
                          </Badge>
                        )}
                      </div>
                    </div>
                  </div>
                  {official.existing_user_id && (
                    <Badge variant="secondary" className="text-xs">
                      Existing User
                    </Badge>
                  )}
                </div>
              ))}
            </div>
          ) : (
            <p className="text-gray-500 text-center py-4">No officials added yet</p>
          )}
        </CardContent>
      </Card>

      {/* Terms and Conditions */}
      <Card>
        <CardContent className="p-6">
          <div className="flex items-start space-x-3">
            <Checkbox
              id="terms"
              checked={termsAccepted}
              onCheckedChange={setTermsAccepted}
            />
            <div className="space-y-2">
              <Label htmlFor="terms" className="text-sm font-medium cursor-pointer">
                I accept the{' '}
                <button
                  type="button"
                  className="text-green-600 hover:text-green-700 underline"
                  onClick={() => setShowTermsModal(true)}
                >
                  Terms and Conditions
                </button>
                {' '}and{' '}
                <button
                  type="button"
                  className="text-green-600 hover:text-green-700 underline"
                  onClick={() => setShowTermsModal(true)}
                >
                  Privacy Policy
                </button>
              </Label>
              <p className="text-xs text-gray-600">
                By registering, you agree to our tournament rules, regulations, and data processing policies.
              </p>
              {!termsAccepted && (
                <p className="text-red-600 text-xs">
                  You must accept the terms and conditions to complete registration.
                </p>
              )}
            </div>
          </div>
        </CardContent>
      </Card>

      {/* Registration Fee Notice */}
      <Card className="bg-blue-50 border-blue-200">
        <CardContent className="p-4">
          <h4 className="font-medium text-blue-800 mb-2">Registration Fee</h4>
          <p className="text-sm text-blue-700">
            A registration fee of <strong>KES {registration_fee.toLocaleString()}</strong> is required to complete your team registration. 
            You will receive payment instructions via email after submitting this form.
          </p>
          <p className="text-xs text-blue-600 mt-2">
            Payment must be completed within 48 hours to secure your team's spot in the tournament.
          </p>
        </CardContent>
      </Card>
    </div>
  );
};

export default ReviewStep;