import React, { useState } from 'react';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { Button } from '@/components/ui/button';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';
import { Tabs, TabsContent, TabsList, TabsTrigger } from '@/components/ui/tabs';
import { Badge } from '@/components/ui/badge';
import { UserCheck, User, UserPlus, Search, Mail, Phone, Plus, Minus, Users } from 'lucide-react';

interface Player {
  name: string;
  email: string;
  phone: string;
  position: string;
  jersey_number: string;
  existing_user_id?: number | null;
}

interface PlayersStepProps {
  data: any;
  setData: (key: string, value: any) => void;
  errors: any;
  existingUsers?: any[];
}

const positions = [
  'Goalkeeper', 'Right Back', 'Left Back', 'Centre Back', 'Defensive Midfielder',
  'Central Midfielder', 'Attacking Midfielder', 'Right Winger', 'Left Winger',
  'Centre Forward', 'Striker'
];

const PlayersStep: React.FC<PlayersStepProps> = ({ 
  data, 
  setData, 
  errors,
  existingUsers = [] 
}) => {
  const [selectedPlayerIndex, setSelectedPlayerIndex] = useState<number | null>(null);
  const [activeTab, setActiveTab] = useState('create');
  const [searchTerm, setSearchTerm] = useState('');

  // Filter existing users based on search
  const filteredUsers = existingUsers.filter(user =>
    user.name.toLowerCase().includes(searchTerm.toLowerCase()) ||
    user.email.toLowerCase().includes(searchTerm.toLowerCase())
  );

  const handlePlayerChange = (index: number, field: keyof Player, value: string) => {
    const updatedPlayers = data.players.map((player: Player, i: number) => 
      i === index ? { ...player, [field]: value } : player
    );
    setData('players', updatedPlayers);
  };

  const addPlayer = () => {
    if (data.players.length < 23) {
      setData('players', [...data.players, {
        name: '',
        email: '',
        phone: '',
        position: 'Midfielder',
        jersey_number: '',
        existing_user_id: null
      }]);
    }
  };

  const removePlayer = (index: number) => {
    if (data.players.length > 11) {
      const updatedPlayers = data.players.filter((_: any, i: number) => i !== index);
      setData('players', updatedPlayers);
      if (selectedPlayerIndex === index) {
        setSelectedPlayerIndex(null);
      }
    }
  };

  const selectExistingUser = (user: any) => {
    if (selectedPlayerIndex !== null) {
      handlePlayerChange(selectedPlayerIndex, 'name', user.name);
      handlePlayerChange(selectedPlayerIndex, 'email', user.email);
      handlePlayerChange(selectedPlayerIndex, 'phone', user.phone || '');
      handlePlayerChange(selectedPlayerIndex, 'existing_user_id' as keyof Player, String(user.id));
      setSelectedPlayerIndex(null);
      setActiveTab('create');
    }
  };

  const openUserSelection = (index: number) => {
    setSelectedPlayerIndex(index);
    setActiveTab('existing');
    setSearchTerm('');
  };

  return (
    <div className="space-y-6">
      <Card>
        <CardHeader>
          <CardTitle className="flex items-center gap-2">
            <UserCheck className="w-5 h-5 text-green-600" />
            Team Players ({data.players.length}/23)
          </CardTitle>
          <CardDescription>
            Add players to your team roster. Minimum 11 players, maximum 23 players allowed.
          </CardDescription>
        </CardHeader>
        <CardContent>
          {selectedPlayerIndex !== null ? (
            <Tabs value={activeTab} onValueChange={setActiveTab} className="w-full">
              <div className="mb-4">
                <h3 className="font-medium">
                  Player #{selectedPlayerIndex + 1} - Select or Create User
                </h3>
                <Button 
                  variant="outline" 
                  size="sm" 
                  onClick={() => setSelectedPlayerIndex(null)}
                  className="mt-2"
                >
                  Back to Player List
                </Button>
              </div>

              <TabsList className="grid w-full grid-cols-2">
                <TabsTrigger value="create" className="flex items-center gap-2">
                  <UserPlus className="w-4 h-4" />
                  Create New
                </TabsTrigger>
                <TabsTrigger value="existing" className="flex items-center gap-2">
                  <Search className="w-4 h-4" />
                  Select Existing
                </TabsTrigger>
              </TabsList>

              <TabsContent value="create" className="space-y-4 mt-6">
                <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                  <div>
                    <Label>Full Name *</Label>
                    <Input
                      value={data.players[selectedPlayerIndex]?.name || ''}
                      onChange={(e) => handlePlayerChange(selectedPlayerIndex, 'name', e.target.value)}
                      placeholder="Player's full name"
                    />
                  </div>

                  <div>
                    <Label>Email Address *</Label>
                    <Input
                      type="email"
                      value={data.players[selectedPlayerIndex]?.email || ''}
                      onChange={(e) => handlePlayerChange(selectedPlayerIndex, 'email', e.target.value)}
                      placeholder="player@example.com"
                    />
                  </div>

                  <div>
                    <Label>Phone Number *</Label>
                    <Input
                      type="tel"
                      value={data.players[selectedPlayerIndex]?.phone || ''}
                      onChange={(e) => handlePlayerChange(selectedPlayerIndex, 'phone', e.target.value)}
                      placeholder="07XX XXX XXX"
                    />
                  </div>

                  <div>
                    <Label>Jersey Number</Label>
                    <Input
                      type="number"
                      min="1"
                      max="99"
                      value={data.players[selectedPlayerIndex]?.jersey_number || ''}
                      onChange={(e) => handlePlayerChange(selectedPlayerIndex, 'jersey_number', e.target.value)}
                      placeholder="1-99"
                    />
                  </div>

                  <div className="md:col-span-2">
                    <Label>Position *</Label>
                    <Select
                      value={data.players[selectedPlayerIndex]?.position || ''}
                      onValueChange={(value) => handlePlayerChange(selectedPlayerIndex, 'position', value)}
                    >
                      <SelectTrigger>
                        <SelectValue placeholder="Select position" />
                      </SelectTrigger>
                      <SelectContent>
                        {positions.map(position => (
                          <SelectItem key={position} value={position}>
                            {position}
                          </SelectItem>
                        ))}
                      </SelectContent>
                    </Select>
                  </div>
                </div>
              </TabsContent>

              <TabsContent value="existing" className="space-y-4 mt-6">
                <div>
                  <Label>Search Existing Users</Label>
                  <div className="relative">
                    <Search className="absolute left-3 top-1/2 transform -translate-y-1/2 text-gray-400 w-4 h-4" />
                    <Input
                      placeholder="Search by name or email..."
                      value={searchTerm}
                      onChange={(e) => setSearchTerm(e.target.value)}
                      className="pl-10"
                    />
                  </div>
                </div>

                <div className="max-h-64 overflow-y-auto space-y-2">
                  {filteredUsers.length > 0 ? (
                    filteredUsers.map(user => (
                      <Card 
                        key={user.id} 
                        className="p-4 hover:bg-gray-50 cursor-pointer transition-colors"
                        onClick={() => selectExistingUser(user)}
                      >
                        <div className="flex items-center justify-between">
                          <div className="flex items-center gap-3">
                            <div className="w-10 h-10 bg-gray-100 rounded-full flex items-center justify-center">
                              <User className="w-5 h-5 text-gray-600" />
                            </div>
                            <div>
                              <h4 className="font-medium">{user.name}</h4>
                              <div className="flex items-center gap-4 text-sm text-gray-500">
                                <div className="flex items-center gap-1">
                                  <Mail className="w-3 h-3" />
                                  {user.email}
                                </div>
                                {user.phone && (
                                  <div className="flex items-center gap-1">
                                    <Phone className="w-3 h-3" />
                                    {user.phone}
                                  </div>
                                )}
                              </div>
                            </div>
                          </div>
                          <Button variant="outline" size="sm">
                            Select
                          </Button>
                        </div>
                      </Card>
                    ))
                  ) : (
                    <Card className="p-6 text-center border-dashed">
                      <User className="w-12 h-12 text-gray-300 mx-auto mb-3" />
                      <p className="text-gray-500 mb-3">
                        {searchTerm ? `No users found matching "${searchTerm}"` : 'No users available'}
                      </p>
                      <Button 
                        variant="outline" 
                        onClick={() => setActiveTab('create')}
                      >
                        <UserPlus className="w-4 h-4 mr-2" />
                        Create New Player Instead
                      </Button>
                    </Card>
                  )}
                </div>
              </TabsContent>
            </Tabs>
          ) : (
            <div className="space-y-4">
              {/* Players List */}
              <div className="grid gap-4">
                {data.players.map((player: Player, index: number) => (
                  <Card key={index} className="p-4">
                    <div className="flex items-center justify-between">
                      <div className="flex items-center gap-3">
                        <div className="w-10 h-10 bg-gray-100 rounded-full flex items-center justify-center">
                          <span className="text-sm font-medium">#{index + 1}</span>
                        </div>
                        <div>
                          <h4 className="font-medium">
                            {player.name || 'Unnamed Player'}
                            {player.existing_user_id && (
                              <Badge variant="secondary" className="ml-2 text-xs">
                                Existing User
                              </Badge>
                            )}
                          </h4>
                          <div className="flex items-center gap-4 text-sm text-gray-500">
                            <span>{player.position || 'No position'}</span>
                            {player.jersey_number && <span>#{player.jersey_number}</span>}
                            {player.email && (
                              <div className="flex items-center gap-1">
                                <Mail className="w-3 h-3" />
                                {player.email}
                              </div>
                            )}
                          </div>
                        </div>
                      </div>
                      <div className="flex items-center gap-2">
                        <Button 
                          variant="outline" 
                          size="sm"
                          onClick={() => openUserSelection(index)}
                        >
                          {player.name ? 'Edit' : 'Add Player'}
                        </Button>
                        {data.players.length > 11 && (
                          <Button 
                            variant="outline" 
                            size="sm"
                            onClick={() => removePlayer(index)}
                          >
                            <Minus className="w-4 h-4" />
                          </Button>
                        )}
                      </div>
                    </div>
                  </Card>
                ))}
              </div>

              {/* Add Player Button */}
              {data.players.length < 23 && (
                <Button 
                  variant="outline" 
                  onClick={addPlayer}
                  className="w-full"
                >
                  <Plus className="w-4 h-4 mr-2" />
                  Add Player ({data.players.length}/23)
                </Button>
              )}
            </div>
          )}
        </CardContent>
      </Card>

      {/* Squad Requirements */}
      <Card className="bg-green-50 border-green-200">
        <CardContent className="p-4">
          <h4 className="font-medium text-green-800 mb-2 flex items-center gap-2">
            <Users className="w-4 h-4" />
            Squad Requirements
          </h4>
          <ul className="text-sm text-green-700 space-y-1">
            <li>• Minimum 11 players required to register</li>
            <li>• Maximum 23 players allowed per team</li>
            <li>• At least one goalkeeper recommended</li>
            <li>• Jersey numbers must be unique (1-99)</li>
            <li>• All players must provide valid contact information</li>
          </ul>
        </CardContent>
      </Card>
    </div>
  );
};

export default PlayersStep;