import React, { useState } from 'react';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { Button } from '@/components/ui/button';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';
import { Tabs, TabsContent, TabsList, TabsTrigger } from '@/components/ui/tabs';
import { Badge } from '@/components/ui/badge';
import { Textarea } from '@/components/ui/textarea';
import { Briefcase, User, UserPlus, Search, Mail, Phone, Plus, Minus, Award } from 'lucide-react';

interface Official {
  name: string;
  email: string;
  phone: string;
  role: string;
  qualifications: string;
  existing_user_id?: number | null;
}

interface OfficialsStepProps {
  data: any;
  setData: (key: string, value: any) => void;
  errors: any;
  existingUsers?: any[];
}

const officialRoles = [
  'Head Coach',
  'Assistant Coach',
  'Goalkeeper Coach',
  'Fitness Coach',
  'Team Manager',
  'Physiotherapist',
  'Team Doctor',
  'Kit Manager',
  'Scout',
  'Analyst'
];

const OfficialsStep: React.FC<OfficialsStepProps> = ({ 
  data, 
  setData, 
  errors,
  existingUsers = [] 
}) => {
  const [selectedOfficialIndex, setSelectedOfficialIndex] = useState<number | null>(null);
  const [activeTab, setActiveTab] = useState('create');
  const [searchTerm, setSearchTerm] = useState('');

  // Filter existing users based on search
  const filteredUsers = existingUsers.filter(user =>
    user.name.toLowerCase().includes(searchTerm.toLowerCase()) ||
    user.email.toLowerCase().includes(searchTerm.toLowerCase())
  );

  const handleOfficialChange = (index: number, field: keyof Official, value: string) => {
    const updatedOfficials = data.officials.map((official: Official, i: number) => 
      i === index ? { ...official, [field]: value } : official
    );
    setData('officials', updatedOfficials);
  };

  const addOfficial = () => {
    if (data.officials.length < 10) {
      setData('officials', [...data.officials, {
        name: '',
        email: '',
        phone: '',
        role: 'Assistant Coach',
        qualifications: '',
        existing_user_id: null
      }]);
    }
  };

  const removeOfficial = (index: number) => {
    if (data.officials.length > 1) {
      const updatedOfficials = data.officials.filter((_: any, i: number) => i !== index);
      setData('officials', updatedOfficials);
      if (selectedOfficialIndex === index) {
        setSelectedOfficialIndex(null);
      }
    }
  };

  const selectExistingUser = (user: any) => {
    if (selectedOfficialIndex !== null) {
      handleOfficialChange(selectedOfficialIndex, 'name', user.name);
      handleOfficialChange(selectedOfficialIndex, 'email', user.email);
      handleOfficialChange(selectedOfficialIndex, 'phone', user.phone || '');
      handleOfficialChange(selectedOfficialIndex, 'existing_user_id' as keyof Official, String(user.id));
      setSelectedOfficialIndex(null);
      setActiveTab('create');
    }
  };

  const openUserSelection = (index: number) => {
    setSelectedOfficialIndex(index);
    setActiveTab('existing');
    setSearchTerm('');
  };

  return (
    <div className="space-y-6">
      <Card>
        <CardHeader>
          <CardTitle className="flex items-center gap-2">
            <Briefcase className="w-5 h-5 text-blue-600" />
            Team Officials ({data.officials.length}/10)
          </CardTitle>
          <CardDescription>
            Add coaching staff and team officials. At least one official is required.
          </CardDescription>
        </CardHeader>
        <CardContent>
          {selectedOfficialIndex !== null ? (
            <Tabs value={activeTab} onValueChange={setActiveTab} className="w-full">
              <div className="mb-4">
                <h3 className="font-medium">
                  Official #{selectedOfficialIndex + 1} - Select or Create User
                </h3>
                <Button 
                  variant="outline" 
                  size="sm" 
                  onClick={() => setSelectedOfficialIndex(null)}
                  className="mt-2"
                >
                  Back to Officials List
                </Button>
              </div>

              <TabsList className="grid w-full grid-cols-2">
                <TabsTrigger value="create" className="flex items-center gap-2">
                  <UserPlus className="w-4 h-4" />
                  Create New
                </TabsTrigger>
                <TabsTrigger value="existing" className="flex items-center gap-2">
                  <Search className="w-4 h-4" />
                  Select Existing
                </TabsTrigger>
              </TabsList>

              <TabsContent value="create" className="space-y-4 mt-6">
                <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                  <div>
                    <Label>Full Name *</Label>
                    <Input
                      value={data.officials[selectedOfficialIndex]?.name || ''}
                      onChange={(e) => handleOfficialChange(selectedOfficialIndex, 'name', e.target.value)}
                      placeholder="Official's full name"
                    />
                  </div>

                  <div>
                    <Label>Email Address *</Label>
                    <Input
                      type="email"
                      value={data.officials[selectedOfficialIndex]?.email || ''}
                      onChange={(e) => handleOfficialChange(selectedOfficialIndex, 'email', e.target.value)}
                      placeholder="official@example.com"
                    />
                  </div>

                  <div>
                    <Label>Phone Number *</Label>
                    <Input
                      type="tel"
                      value={data.officials[selectedOfficialIndex]?.phone || ''}
                      onChange={(e) => handleOfficialChange(selectedOfficialIndex, 'phone', e.target.value)}
                      placeholder="07XX XXX XXX"
                    />
                  </div>

                  <div>
                    <Label>Role *</Label>
                    <Select
                      value={data.officials[selectedOfficialIndex]?.role || ''}
                      onValueChange={(value) => handleOfficialChange(selectedOfficialIndex, 'role', value)}
                    >
                      <SelectTrigger>
                        <SelectValue placeholder="Select role" />
                      </SelectTrigger>
                      <SelectContent>
                        {officialRoles.map(role => (
                          <SelectItem key={role} value={role}>
                            {role}
                          </SelectItem>
                        ))}
                      </SelectContent>
                    </Select>
                  </div>

                  <div className="md:col-span-2">
                    <Label>Qualifications & Experience</Label>
                    <Textarea
                      value={data.officials[selectedOfficialIndex]?.qualifications || ''}
                      onChange={(e) => handleOfficialChange(selectedOfficialIndex, 'qualifications', e.target.value)}
                      placeholder="Describe relevant qualifications, certifications, and experience..."
                      rows={3}
                    />
                  </div>
                </div>
              </TabsContent>

              <TabsContent value="existing" className="space-y-4 mt-6">
                <div>
                  <Label>Search Existing Users</Label>
                  <div className="relative">
                    <Search className="absolute left-3 top-1/2 transform -translate-y-1/2 text-gray-400 w-4 h-4" />
                    <Input
                      placeholder="Search by name or email..."
                      value={searchTerm}
                      onChange={(e) => setSearchTerm(e.target.value)}
                      className="pl-10"
                    />
                  </div>
                </div>

                <div className="max-h-64 overflow-y-auto space-y-2">
                  {filteredUsers.length > 0 ? (
                    filteredUsers.map(user => (
                      <Card 
                        key={user.id} 
                        className="p-4 hover:bg-gray-50 cursor-pointer transition-colors"
                        onClick={() => selectExistingUser(user)}
                      >
                        <div className="flex items-center justify-between">
                          <div className="flex items-center gap-3">
                            <div className="w-10 h-10 bg-gray-100 rounded-full flex items-center justify-center">
                              <User className="w-5 h-5 text-gray-600" />
                            </div>
                            <div>
                              <h4 className="font-medium">{user.name}</h4>
                              <div className="flex items-center gap-4 text-sm text-gray-500">
                                <div className="flex items-center gap-1">
                                  <Mail className="w-3 h-3" />
                                  {user.email}
                                </div>
                                {user.phone && (
                                  <div className="flex items-center gap-1">
                                    <Phone className="w-3 h-3" />
                                    {user.phone}
                                  </div>
                                )}
                              </div>
                            </div>
                          </div>
                          <Button variant="outline" size="sm">
                            Select
                          </Button>
                        </div>
                      </Card>
                    ))
                  ) : (
                    <Card className="p-6 text-center border-dashed">
                      <User className="w-12 h-12 text-gray-300 mx-auto mb-3" />
                      <p className="text-gray-500 mb-3">
                        {searchTerm ? `No users found matching "${searchTerm}"` : 'No users available'}
                      </p>
                      <Button 
                        variant="outline" 
                        onClick={() => setActiveTab('create')}
                      >
                        <UserPlus className="w-4 h-4 mr-2" />
                        Create New Official Instead
                      </Button>
                    </Card>
                  )}
                </div>
              </TabsContent>
            </Tabs>
          ) : (
            <div className="space-y-4">
              {/* Officials List */}
              <div className="grid gap-4">
                {data.officials.map((official: Official, index: number) => (
                  <Card key={index} className="p-4">
                    <div className="flex items-center justify-between">
                      <div className="flex items-center gap-3">
                        <div className="w-10 h-10 bg-blue-100 rounded-full flex items-center justify-center">
                          <Briefcase className="w-5 h-5 text-blue-600" />
                        </div>
                        <div>
                          <h4 className="font-medium">
                            {official.name || 'Unnamed Official'}
                            {official.existing_user_id && (
                              <Badge variant="secondary" className="ml-2 text-xs">
                                Existing User
                              </Badge>
                            )}
                          </h4>
                          <div className="flex items-center gap-4 text-sm text-gray-500">
                            <span>{official.role || 'No role assigned'}</span>
                            {official.email && (
                              <div className="flex items-center gap-1">
                                <Mail className="w-3 h-3" />
                                {official.email}
                              </div>
                            )}
                            {official.qualifications && (
                              <Badge variant="outline" className="text-xs">
                                <Award className="w-3 h-3 mr-1" />
                                Qualified
                              </Badge>
                            )}
                          </div>
                        </div>
                      </div>
                      <div className="flex items-center gap-2">
                        <Button 
                          variant="outline" 
                          size="sm"
                          onClick={() => openUserSelection(index)}
                        >
                          {official.name ? 'Edit' : 'Add Official'}
                        </Button>
                        {data.officials.length > 1 && (
                          <Button 
                            variant="outline" 
                            size="sm"
                            onClick={() => removeOfficial(index)}
                          >
                            <Minus className="w-4 h-4" />
                          </Button>
                        )}
                      </div>
                    </div>
                  </Card>
                ))}
              </div>

              {/* Add Official Button */}
              {data.officials.length < 10 && (
                <Button 
                  variant="outline" 
                  onClick={addOfficial}
                  className="w-full"
                >
                  <Plus className="w-4 h-4 mr-2" />
                  Add Official ({data.officials.length}/10)
                </Button>
              )}
            </div>
          )}
        </CardContent>
      </Card>

      {/* Staff Requirements */}
      <Card className="bg-blue-50 border-blue-200">
        <CardContent className="p-4">
          <h4 className="font-medium text-blue-800 mb-2 flex items-center gap-2">
            <Award className="w-4 h-4" />
            Official Requirements
          </h4>
          <ul className="text-sm text-blue-700 space-y-1">
            <li>• At least one Head Coach is required</li>
            <li>• Maximum 10 officials allowed per team</li>
            <li>• Medical staff recommended for competitive matches</li>
            <li>• All coaching qualifications should be specified</li>
            <li>• Officials will receive tournament communications</li>
          </ul>
        </CardContent>
      </Card>
    </div>
  );
};

export default OfficialsStep;