import React, { useState } from 'react';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { Search, Users, Plus } from 'lucide-react';

interface ExistingTeam {
  id: number;
  name: string;
  description: string;
}

interface ExistingTeamStepProps {
  existingTeams: ExistingTeam[];
  onSelectTeam: (team: ExistingTeam) => void;
  onCreateNewTeam: () => void;
  selectedTeam?: ExistingTeam;
}

const ExistingTeamStep: React.FC<ExistingTeamStepProps> = ({
  existingTeams,
  onSelectTeam,
  onCreateNewTeam,
  selectedTeam
}) => {
  const [searchQuery, setSearchQuery] = useState('');

  const filteredTeams = existingTeams.filter(team =>
    team.name.toLowerCase().includes(searchQuery.toLowerCase()) ||
    team.description.toLowerCase().includes(searchQuery.toLowerCase())
  );

  return (
    <div className="space-y-6">
      <div className="text-center">
        <Users className="h-12 w-12 text-green-600 mx-auto mb-4" />
        <h2 className="text-2xl font-bold text-gray-900 mb-2">Team Selection</h2>
        <p className="text-gray-600">
          Choose an existing team or create a new one for this tournament
        </p>
      </div>

      {/* Search */}
      <div className="relative">
        <Search className="absolute left-3 top-1/2 transform -translate-y-1/2 text-gray-400 h-4 w-4" />
        <Input
          placeholder="Search existing teams..."
          value={searchQuery}
          onChange={(e) => setSearchQuery(e.target.value)}
          className="pl-10"
        />
      </div>

      {/* Existing Teams */}
      {filteredTeams.length > 0 && (
        <div className="space-y-4">
          <h3 className="text-lg font-semibold text-gray-900">Existing Teams</h3>
          <div className="grid gap-4">
            {filteredTeams.map((team) => (
              <Card
                key={team.id}
                className={`cursor-pointer transition-all hover:shadow-md ${
                  selectedTeam?.id === team.id ? 'ring-2 ring-green-500 bg-green-50' : ''
                }`}
                onClick={() => onSelectTeam(team)}
              >
                <CardHeader className="pb-3">
                  <CardTitle className="text-lg">{team.name}</CardTitle>
                </CardHeader>
                <CardContent>
                  <p className="text-gray-600 text-sm">{team.description}</p>
                  {selectedTeam?.id === team.id && (
                    <div className="mt-3 p-2 bg-green-100 rounded-md">
                      <p className="text-green-800 text-sm font-medium">
                        ✓ Selected for registration
                      </p>
                    </div>
                  )}
                </CardContent>
              </Card>
            ))}
          </div>
        </div>
      )}

      {/* Create New Team Option */}
      <div className="border-t pt-6">
        <Card className="border-dashed border-2 border-gray-300 hover:border-green-500 transition-colors">
          <CardContent className="p-6 text-center">
            <Plus className="h-8 w-8 text-gray-400 mx-auto mb-3" />
            <h3 className="text-lg font-semibold text-gray-900 mb-2">
              Create New Team
            </h3>
            <p className="text-gray-600 text-sm mb-4">
              Register a new team for this tournament
            </p>
            <Button
              onClick={onCreateNewTeam}
              variant="outline"
              className="w-full"
            >
              Create New Team
            </Button>
          </CardContent>
        </Card>
      </div>

      {/* No teams found */}
      {filteredTeams.length === 0 && searchQuery && (
        <div className="text-center py-8">
          <Users className="h-12 w-12 text-gray-300 mx-auto mb-4" />
          <p className="text-gray-500">No teams found matching your search.</p>
          <p className="text-gray-400 text-sm mt-2">
            Try a different search term or create a new team.
          </p>
        </div>
      )}
    </div>
  );
};

export default ExistingTeamStep; 