import React, { useState } from 'react';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { Button } from '@/components/ui/button';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card';
import { Tabs, TabsContent, TabsList, TabsTrigger } from '@/components/ui/tabs';
import { Badge } from '@/components/ui/badge';
import { Crown, User, UserPlus, Search, Mail, Phone, Shield } from 'lucide-react';

interface CaptainStepProps {
  data: any;
  setData: (key: string, value: any) => void;
  errors: any;
  existingUsers?: any[];
}

const CaptainStep: React.FC<CaptainStepProps> = ({ 
  data, 
  setData, 
  errors,
  existingUsers = [] 
}) => {
  const [activeTab, setActiveTab] = useState('create');
  const [searchTerm, setSearchTerm] = useState('');

  // Filter existing users based on search
  const filteredUsers = existingUsers.filter(user =>
    user.name.toLowerCase().includes(searchTerm.toLowerCase()) ||
    user.email.toLowerCase().includes(searchTerm.toLowerCase())
  );

  const handleCaptainChange = (field: string, value: string) => {
    setData('captain', { ...data.captain, [field]: value });
  };

  const selectExistingUser = (user: any) => {
    setData('captain', {
      ...data.captain,
      name: user.name,
      email: user.email,
      phone: user.phone || '',
      existing_user_id: user.id
    });
    setActiveTab('selected');
  };

  const clearSelection = () => {
    setData('captain', {
      name: '',
      email: '',
      phone: '',
      password: '',
      password_confirmation: '',
      existing_user_id: null
    });
    setActiveTab('create');
  };

  return (
    <div className="space-y-6">
      <Card>
        <CardHeader>
          <CardTitle className="flex items-center gap-2">
            <Crown className="w-5 h-5 text-yellow-600" />
            Team Captain Setup
          </CardTitle>
          <CardDescription>
            Create a new captain account or select from existing users
          </CardDescription>
        </CardHeader>
        <CardContent>
          <Tabs value={activeTab} onValueChange={setActiveTab} className="w-full">
            <TabsList className="grid w-full grid-cols-3">
              <TabsTrigger value="create" className="flex items-center gap-2">
                <UserPlus className="w-4 h-4" />
                Create New
              </TabsTrigger>
              <TabsTrigger value="existing" className="flex items-center gap-2">
                <Search className="w-4 h-4" />
                Select Existing
              </TabsTrigger>
              <TabsTrigger value="selected" disabled={!data.captain.existing_user_id}>
                <User className="w-4 h-4" />
                Selected
              </TabsTrigger>
            </TabsList>

            <TabsContent value="create" className="space-y-4 mt-6">
              <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                <div>
                  <Label htmlFor="captain_name">Full Name *</Label>
                  <Input
                    id="captain_name"
                    value={data.captain.name}
                    onChange={(e) => handleCaptainChange('name', e.target.value)}
                    placeholder="Captain's full name"
                    className={errors['captain.name'] ? 'border-red-500' : ''}
                  />
                  {errors['captain.name'] && (
                    <p className="text-red-600 text-sm mt-1">{errors['captain.name']}</p>
                  )}
                </div>

                <div>
                  <Label htmlFor="captain_email">Email Address *</Label>
                  <Input
                    id="captain_email"
                    type="email"
                    value={data.captain.email}
                    onChange={(e) => handleCaptainChange('email', e.target.value)}
                    placeholder="captain@example.com"
                    className={errors['captain.email'] ? 'border-red-500' : ''}
                  />
                  {errors['captain.email'] && (
                    <p className="text-red-600 text-sm mt-1">{errors['captain.email']}</p>
                  )}
                </div>

                <div>
                  <Label htmlFor="captain_phone">Phone Number *</Label>
                  <Input
                    id="captain_phone"
                    type="tel"
                    value={data.captain.phone}
                    onChange={(e) => handleCaptainChange('phone', e.target.value)}
                    placeholder="07XX XXX XXX"
                    className={errors['captain.phone'] ? 'border-red-500' : ''}
                  />
                  {errors['captain.phone'] && (
                    <p className="text-red-600 text-sm mt-1">{errors['captain.phone']}</p>
                  )}
                </div>

                <div>
                  <Label htmlFor="captain_password">Password *</Label>
                  <Input
                    id="captain_password"
                    type="password"
                    value={data.captain.password}
                    onChange={(e) => handleCaptainChange('password', e.target.value)}
                    placeholder="Choose a secure password"
                    className={errors['captain.password'] ? 'border-red-500' : ''}
                  />
                  {errors['captain.password'] && (
                    <p className="text-red-600 text-sm mt-1">{errors['captain.password']}</p>
                  )}
                </div>

                <div className="md:col-span-2">
                  <Label htmlFor="captain_password_confirmation">Confirm Password *</Label>
                  <Input
                    id="captain_password_confirmation"
                    type="password"
                    value={data.captain.password_confirmation}
                    onChange={(e) => handleCaptainChange('password_confirmation', e.target.value)}
                    placeholder="Confirm your password"
                    className={errors['captain.password_confirmation'] ? 'border-red-500' : ''}
                  />
                  {errors['captain.password_confirmation'] && (
                    <p className="text-red-600 text-sm mt-1">{errors['captain.password_confirmation']}</p>
                  )}
                </div>
              </div>
            </TabsContent>

            <TabsContent value="existing" className="space-y-4 mt-6">
              <div>
                <Label htmlFor="user_search">Search Existing Users</Label>
                <div className="relative">
                  <Search className="absolute left-3 top-1/2 transform -translate-y-1/2 text-gray-400 w-4 h-4" />
                  <Input
                    id="user_search"
                    placeholder="Search by name or email..."
                    value={searchTerm}
                    onChange={(e) => setSearchTerm(e.target.value)}
                    className="pl-10"
                  />
                </div>
              </div>

              <div className="max-h-64 overflow-y-auto space-y-2">
                {filteredUsers.length > 0 ? (
                  filteredUsers.map(user => (
                    <Card 
                      key={user.id} 
                      className="p-4 hover:bg-gray-50 cursor-pointer transition-colors"
                      onClick={() => selectExistingUser(user)}
                    >
                      <div className="flex items-center justify-between">
                        <div className="flex items-center gap-3">
                          <div className="w-10 h-10 bg-gray-100 rounded-full flex items-center justify-center">
                            <User className="w-5 h-5 text-gray-600" />
                          </div>
                          <div>
                            <h4 className="font-medium">{user.name}</h4>
                            <div className="flex items-center gap-4 text-sm text-gray-500">
                              <div className="flex items-center gap-1">
                                <Mail className="w-3 h-3" />
                                {user.email}
                              </div>
                              {user.phone && (
                                <div className="flex items-center gap-1">
                                  <Phone className="w-3 h-3" />
                                  {user.phone}
                                </div>
                              )}
                            </div>
                          </div>
                        </div>
                        <Button variant="outline" size="sm">
                          Select
                        </Button>
                      </div>
                    </Card>
                  ))
                ) : (
                  <Card className="p-6 text-center border-dashed">
                    <User className="w-12 h-12 text-gray-300 mx-auto mb-3" />
                    <p className="text-gray-500 mb-3">
                      {searchTerm ? `No users found matching "${searchTerm}"` : 'No users available'}
                    </p>
                    <Button 
                      variant="outline" 
                      onClick={() => setActiveTab('create')}
                    >
                      <UserPlus className="w-4 h-4 mr-2" />
                      Create New Captain Instead
                    </Button>
                  </Card>
                )}
              </div>
            </TabsContent>

            <TabsContent value="selected" className="space-y-4 mt-6">
              {data.captain.existing_user_id && (
                <Card className="p-4 bg-yellow-50 border-yellow-200">
                  <div className="flex items-center justify-between">
                    <div className="flex items-center gap-3">
                      <div className="w-10 h-10 bg-yellow-100 rounded-full flex items-center justify-center">
                        <Crown className="w-5 h-5 text-yellow-600" />
                      </div>
                      <div>
                        <h4 className="font-medium text-yellow-800">{data.captain.name}</h4>
                        <div className="flex items-center gap-4 text-sm text-yellow-600">
                          <div className="flex items-center gap-1">
                            <Mail className="w-3 h-3" />
                            {data.captain.email}
                          </div>
                          {data.captain.phone && (
                            <div className="flex items-center gap-1">
                              <Phone className="w-3 h-3" />
                              {data.captain.phone}
                            </div>
                          )}
                        </div>
                      </div>
                    </div>
                    <div className="flex items-center gap-2">
                      <Badge variant="secondary" className="bg-yellow-100 text-yellow-800">
                        Team Captain
                      </Badge>
                      <Button variant="outline" size="sm" onClick={clearSelection}>
                        Change
                      </Button>
                    </div>
                  </div>
                </Card>
              )}
            </TabsContent>
          </Tabs>
        </CardContent>
      </Card>

      {/* Captain Responsibilities */}
      <Card className="bg-blue-50 border-blue-200">
        <CardContent className="p-4">
          <h4 className="font-medium text-blue-800 mb-2 flex items-center gap-2">
            <Shield className="w-4 h-4" />
            Captain Responsibilities
          </h4>
          <ul className="text-sm text-blue-700 space-y-1">
            <li>• Lead team communications and coordination</li>
            <li>• Represent the team in official tournament matters</li>
            <li>• Manage team roster and player substitutions</li>
            <li>• Receive all official tournament communications</li>
            <li>• Make strategic decisions during matches</li>
          </ul>
        </CardContent>
      </Card>
    </div>
  );
};

export default CaptainStep;