import { useState } from "react";
import { Menu, X, Trophy, Users, Calendar, Award, Image, Video, Newspaper, LayoutDashboard, LogOut, UserPlus, Quote, AlertTriangle, Shield, UserX, Clock, Eye, UsersRound, HelpCircle, ChevronDown } from "lucide-react";
import { Button } from "@/components/ui/button";
import { DropdownMenu, DropdownMenuContent, DropdownMenuItem, DropdownMenuSeparator, DropdownMenuTrigger } from "@/components/ui/dropdown-menu";
import { cn } from "@/lib/utils";
import { Link, router, usePage } from "@inertiajs/react";

interface HeaderProps {
  transparent?: boolean;
}

const Header = ({ transparent = false }: HeaderProps) => {
  const { auth } = usePage().props as any;
  const user = auth.user;
  const isImpersonating = auth.impersonation?.is_impersonating || false;
  const [isMenuOpen, setIsMenuOpen] = useState(false);
  const [isExiting, setIsExiting] = useState(false);

  const handleExitImpersonation = () => {
    setIsExiting(true);
    router.post(route('admin.users.stopImpersonating'), {}, {
      onSuccess: () => {
      },
      onError: (errors) => {
        setIsExiting(false);
      }
    });
  };

  const navigation = [
    { 
      name: "Home",
      href: "/",
      icon: Trophy,
      isActive: window.location.pathname === "/"
    },
    { 
      name: "About", 
      href: "/about", 
      icon: Award,
      children: [
        {
          name: "About",
          href: "/about",
          icon: Award
        },
        {
          name: "Founder's Message",
          href: "/founders-message",
          icon: Quote
        },
        {
          name: "Gallery",
          href: "/gallery",
          icon: Image
        },
        {
          name: "Help",
          href: "/help",
          icon: HelpCircle
        }
      ]
    },
    { 
      name: "Teams", 
      href: "/teams", 
      icon: UsersRound,
      children: [
        {
          name: "View Teams",
          href: "/teams",
          icon: UsersRound
        },
        {
          name: "Register Team",
          href: "/register-team",
          icon: Users
        }
      ]
    },
    { 
      name: "Fixtures", 
      href: "/fixtures", 
      icon: Calendar 
    },
    { 
      name: "Vlog", 
      href: "/vlog", 
      icon: Video       
    },
    { 
      name: "News", 
      href: "/news", 
      icon: Newspaper
    },
    { 
      name: "Contact", 
      href: "/contact", 
      icon: Menu
    },
  ];

  const handleLogout = () => {
    router.post('/logout');
  };

  return (
    <>
      {/* Impersonation Banner */}
      {isImpersonating && auth.impersonation && (
        <div className="fixed top-0 left-0 right-0 z-[9999] bg-gradient-to-r from-red-600 to-red-700 shadow-lg border-b-2 border-red-800">
          <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
            <div className="flex items-center justify-between py-3">
              {/* Warning Icon and Message */}
              <div className="flex items-center space-x-3">
                <div className="flex items-center justify-center w-8 h-8 bg-white/20 rounded-full backdrop-blur-sm">
                  <AlertTriangle className="w-5 h-5 text-white animate-pulse" />
                </div>
                
                <div className="flex flex-col sm:flex-row sm:items-center sm:space-x-2">
                  <div className="flex items-center space-x-2">
                    <Shield className="w-4 h-4 text-yellow-300" />
                    <span className="text-white font-semibold text-sm">
                      IMPERSONATION MODE
                    </span>
                  </div>
                  
                  <div className="hidden sm:block text-white/80 text-sm">•</div>
                  
                  <div className="flex items-center space-x-2 text-white/90 text-sm">
                    <UserX className="w-4 h-4" />
                    <span>
                      Viewing as <strong className="text-white">{auth.impersonation.impersonated_user.name}</strong>
                    </span>
                    <span className="hidden md:inline text-white/70">
                      ({auth.impersonation.impersonated_user.email})
                    </span>
                  </div>
                </div>
              </div>

              {/* User Info and Exit Button */}
              <div className="flex items-center space-x-3">
                {/* Original User Info (Desktop) */}
                <div className="hidden lg:flex items-center space-x-2 text-white/80 text-sm">
                  <span>Admin:</span>
                  <span className="text-white font-medium">{auth.impersonation.original_user.name}</span>
                </div>

                {/* Exit Button */}
                <Button
                  onClick={handleExitImpersonation}
                  disabled={isExiting}
                  variant="secondary"
                  size="sm"
                  className="bg-white/20 hover:bg-white/30 text-white border-white/30 hover:border-white/50 backdrop-blur-sm transition-all duration-200 font-medium disabled:opacity-50 disabled:cursor-not-allowed"
                >
                  {isExiting ? (
                    <>
                      <div className="w-4 h-4 mr-2 animate-spin rounded-full border-2 border-white/30 border-t-white"></div>
                      <span className="hidden sm:inline">Exiting...</span>
                      <span className="sm:hidden">...</span>
                    </>
                  ) : (
                    <>
                      <UserX className="w-4 h-4 mr-2" />
                      <span className="hidden sm:inline">Exit Impersonation</span>
                      <span className="sm:hidden">Exit</span>
                    </>
                  )}
                </Button>
              </div>
            </div>
          </div>

          {/* Subtle animation line */}
          <div className="absolute bottom-0 left-0 right-0 h-0.5 bg-gradient-to-r from-transparent via-yellow-400 to-transparent opacity-60">
            <div className="h-full bg-gradient-to-r from-yellow-400 to-orange-400 animate-pulse"></div>
          </div>
        </div>
      )}

      {/* Main Header */}
      <header className={cn(
        "sticky z-40 transition-all duration-500 ease-in-out",
        isImpersonating ? "top-16" : "top-0",
        transparent 
          ? "bg-transparent backdrop-blur-none shadow-none border-b border-transparent" 
          : "bg-white/95 backdrop-blur-md shadow-xl border-b border-gray-100/50"
      )}>
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
        <div className="flex justify-between items-center h-20">
          {/* Logo */}
          <div className="flex items-center space-x-4">
            <div className="relative">
              <img src="/logo.png" alt="Tournament" className="w-10 h-10 hover:scale-110 transition-all duration-300" />
            </div>
            <div className="hidden sm:block">
              <h1 className={cn(
                "text-xl font-bold tracking-tight transition-colors duration-300",
                transparent ? "text-white" : "text-gray-900"
              )}>
              </h1>
            </div>
          </div>

          {/* Desktop Navigation */}
          <nav className="hidden md:flex space-x-2">
            {navigation.map((item) => (
              item.children ? (
                <DropdownMenu key={item.name}>
                  <DropdownMenuTrigger asChild>
                    <Button
                      variant="ghost"
                      className={cn(
                        "flex items-center space-x-2 px-4 py-2 rounded-lg transition-all duration-300 font-medium group relative overflow-hidden",
                        transparent 
                          ? "text-white/90 hover:text-white hover:bg-white/10 hover:shadow-lg" 
                          : "text-gray-700 hover:text-green-600 hover:bg-red-50 hover:shadow-md"
                      )}
                    >
                      <item.icon className="h-4 w-4 transition-transform duration-300 group-hover:scale-110" />
                      <span className="relative z-10">{item.name}</span>
                      <ChevronDown className="h-3 w-3 transition-transform duration-300 group-hover:rotate-180" />
                      {!transparent && (
                        <div className="absolute inset-0 bg-gradient-to-r from-green-500/10 to-red-500/10 opacity-0 group-hover:opacity-100 transition-opacity duration-300"></div>
                      )}
                    </Button>
                  </DropdownMenuTrigger>
                  <DropdownMenuContent align="start" className="w-56 p-2 bg-white/95 backdrop-blur-md border border-gray-200/50 shadow-xl">
                    {item.children.map((child) => (
                      <DropdownMenuItem key={child.name} asChild>
                        <Link 
                          href={child.href} 
                          className="flex items-center px-3 py-2 rounded-md hover:bg-red-50 hover:text-green-600 transition-all duration-200 group"
                        >
                          <child.icon className="h-4 w-4 mr-3 transition-transform duration-200 group-hover:scale-110" />
                          <span className="font-medium">{child.name}</span>
                        </Link>
                      </DropdownMenuItem>
                    ))}
                  </DropdownMenuContent>
                </DropdownMenu>
              ) : (
                <Link
                  key={item.name}
                  href={item.href}
                  className={cn(
                    "flex items-center space-x-2 px-4 py-2 rounded-lg transition-all duration-300 font-medium group relative overflow-hidden",
                    transparent 
                      ? "text-white/90 hover:text-white hover:bg-white/10 hover:shadow-lg" 
                      : "text-gray-700 hover:text-green-600 hover:bg-red-50 hover:shadow-md"
                  )}
                >
                  <item.icon className="h-4 w-4 transition-transform duration-300 group-hover:scale-110" />
                  <span className="relative z-10">{item.name}</span>
                  {!transparent && (
                    <div className="absolute inset-0 bg-gradient-to-r from-green-500/10 to-red-500/10 opacity-0 group-hover:opacity-100 transition-opacity duration-300"></div>
                  )}
                </Link>
              )
            ))}
          </nav>

          {/* Authentication Section */}
          <div className="hidden md:flex items-center space-x-3">
            {user ? (
              // Authenticated User Menu
              <DropdownMenu>
                <DropdownMenuTrigger asChild>
                  <Button 
                    variant="ghost"
                    size="sm"
                    className={cn(
                      "flex items-center space-x-2 px-4 py-2 rounded-lg transition-all duration-300 font-medium group relative overflow-hidden",
                      transparent 
                        ? "bg-white/10 border-white/20 text-white hover:bg-white/20 hover:shadow-lg" 
                        : "bg-gradient-to-r from-green-50 to-red-50 border border-green-200 text-green-700 hover:from-green-100 hover:to-red-100 hover:shadow-md"
                    )}
                  >
                    <UserPlus className="h-4 w-4 transition-transform duration-300 group-hover:scale-110" />
                    <span className="relative z-10">{user.name}</span>
                    <ChevronDown className="h-3 w-3 transition-transform duration-300 group-hover:rotate-180" />
                    {!transparent && (
                      <div className="absolute inset-0 bg-gradient-to-r from-green-500/10 to-red-500/10 opacity-0 group-hover:opacity-100 transition-opacity duration-300"></div>
                    )}
                  </Button>
                </DropdownMenuTrigger>
                <DropdownMenuContent align="end" className="w-56 p-2 bg-white/95 backdrop-blur-md border border-gray-200/50 shadow-xl">
                  <DropdownMenuItem asChild>
                    <Link 
                      href={route('dashboard')} 
                      className="flex items-center px-3 py-2 rounded-md hover:bg-red-50 hover:text-green-600 transition-all duration-200 group"
                    >
                      <LayoutDashboard className="h-4 w-4 mr-3 transition-transform duration-200 group-hover:scale-110" />
                      <span className="font-medium">Dashboard</span>
                    </Link>
                  </DropdownMenuItem>
                  <DropdownMenuSeparator className="my-2" />
                  <DropdownMenuItem 
                    onClick={handleLogout} 
                    className="flex items-center px-3 py-2 rounded-md hover:bg-red-50 hover:text-red-600 transition-all duration-200 group"
                  >
                    <LogOut className="h-4 w-4 mr-3 transition-transform duration-200 group-hover:scale-110" />
                    <span className="font-medium">Logout</span>
                  </DropdownMenuItem>
                </DropdownMenuContent>
              </DropdownMenu>
            ) : (
              // Guest User Buttons
              <>
                <Link href="/login">
                  <Button 
                    variant="ghost"
                    size="sm"
                    className={cn(
                      "px-4 py-2 rounded-lg transition-all duration-300 font-medium group relative overflow-hidden",
                      transparent 
                        ? "bg-white/10 border-white/20 text-white hover:bg-white/20 hover:shadow-lg" 
                        : "text-gray-700 hover:text-green-600 hover:bg-red-50 hover:shadow-md"
                    )}
                  >
                    <span className="relative z-10">Login</span>
                    {!transparent && (
                      <div className="absolute inset-0 bg-gradient-to-r from-green-500/10 to-red-500/10 opacity-0 group-hover:opacity-100 transition-opacity duration-300"></div>
                    )}
                  </Button>
                </Link>
                <Link href="/register-team">
                  <Button 
                    size="sm" 
                    className={cn(
                      "px-6 py-2 rounded-lg transition-all duration-300 font-medium group relative overflow-hidden shadow-md hover:shadow-lg",
                      transparent 
                        ? "bg-white text-green-600 hover:bg-white/90 hover:shadow-xl" 
                        : "bg-gradient-to-r from-green-600 to-red-600 text-white hover:from-green-700 hover:to-red-700 hover:shadow-xl"
                    )}
                  >
                    <span className="relative z-10">Register</span>
                    <div className="absolute inset-0 bg-gradient-to-r from-green-500/20 to-red-500/20 opacity-0 group-hover:opacity-100 transition-opacity duration-300"></div>
                  </Button>
                </Link>
              </>
            )}
          </div>

          {/* Mobile menu button */}
          <div className="md:hidden">
            <Button
              variant="ghost"
              size="sm"
              onClick={() => setIsMenuOpen(!isMenuOpen)}
              className={cn(
                "p-3 rounded-lg transition-all duration-300 group relative overflow-hidden",
                transparent 
                  ? "text-white hover:bg-white/10 hover:shadow-lg" 
                  : "text-gray-700 hover:text-green-600 hover:bg-red-50 hover:shadow-md"
              )}
            >
              <div className="relative z-10 transition-transform duration-300 group-hover:scale-110">
                {isMenuOpen ? <X className="h-6 w-6" /> : <Menu className="h-6 w-6" />}
              </div>
              {!transparent && (
                <div className="absolute inset-0 bg-gradient-to-r from-green-500/10 to-red-500/10 opacity-0 group-hover:opacity-100 transition-opacity duration-300"></div>
              )}
            </Button>
          </div>
        </div>
      </div>

      {/* Mobile Navigation */}
      <div className={cn(
        "md:hidden transition-all duration-500 ease-in-out",
        isMenuOpen ? "max-h-screen opacity-100" : "max-h-0 opacity-0 overflow-hidden"
      )}>
        <div className={cn(
          "px-6 pt-4 pb-8 space-y-2 border-t",
          transparent 
            ? "bg-black/30 backdrop-blur-lg border-white/20 shadow-2xl" 
            : "bg-white/95 backdrop-blur-md border-gray-200/50 shadow-xl"
        )}>
          {navigation.map((item) => (
            <div key={item.name}>
              {item.children ? (
                <div>
                  <div className={cn(
                    "flex items-center space-x-3 px-4 py-3 rounded-lg font-medium transition-all duration-300 group",
                    transparent
                      ? "text-white/90 hover:bg-white/10"
                      : "text-gray-700 hover:bg-red-50 hover:text-green-600"
                  )}>
                    <item.icon className="h-5 w-5 transition-transform duration-300 group-hover:scale-110" />
                    <span className="flex-1">{item.name}</span>
                    <ChevronDown className="h-4 w-4 transition-transform duration-300 group-hover:rotate-180" />
                  </div>
                  <div className="ml-6 space-y-1 mt-2">
                    {item.children.map((child) => (
                      <Link
                        key={child.name}
                        href={child.href}
                        className={cn(
                          "flex items-center space-x-3 px-4 py-2 rounded-lg transition-all duration-300 text-sm group",
                          transparent
                            ? "text-white/70 hover:text-white hover:bg-white/10"
                            : "text-gray-600 hover:text-green-600 hover:bg-red-50"
                        )}
                        onClick={() => setIsMenuOpen(false)}
                      >
                        <child.icon className="h-4 w-4 transition-transform duration-300 group-hover:scale-110" />
                        <span className="font-medium">{child.name}</span>
                      </Link>
                    ))}
                  </div>
                </div>
              ) : (
                <Link
                  href={item.href}
                  className={cn(
                    "flex items-center space-x-3 px-4 py-3 rounded-lg transition-all duration-300 group",
                    transparent
                      ? "text-white/90 hover:text-white hover:bg-white/10"
                      : "text-gray-700 hover:text-green-600 hover:bg-red-50"
                  )}
                  onClick={() => setIsMenuOpen(false)}
                >
                  <item.icon className="h-5 w-5 transition-transform duration-300 group-hover:scale-110" />
                  <span className="font-medium">{item.name}</span>
                </Link>
              )}
            </div>
          ))}
          <div className="pt-6 space-y-3 border-t border-gray-200/50">
            {user ? (
              // Authenticated Mobile Menu
              <>
                <Link href={route('dashboard')}>
                  <Button 
                    variant="ghost"
                    size="sm" 
                    className={cn(
                      "w-full px-4 py-3 rounded-lg transition-all duration-300 group relative overflow-hidden",
                      transparent 
                        ? "bg-white/10 border-white/20 text-white hover:bg-white/20 hover:shadow-lg" 
                        : "text-gray-700 hover:text-green-600 hover:bg-red-50 hover:shadow-md"
                    )}
                  >
                    <LayoutDashboard className="h-4 w-4 mr-3 transition-transform duration-300 group-hover:scale-110" />
                    <span className="font-medium">Dashboard</span>
                    {!transparent && (
                      <div className="absolute inset-0 bg-gradient-to-r from-green-500/10 to-red-500/10 opacity-0 group-hover:opacity-100 transition-opacity duration-300"></div>
                    )}
                  </Button>
                </Link>
                <Button 
                  onClick={handleLogout}
                  variant="ghost"
                  size="sm" 
                  className={cn(
                    "w-full px-4 py-3 rounded-lg transition-all duration-300 group relative overflow-hidden",
                    transparent 
                      ? "bg-white/10 border-white/20 text-white hover:bg-white/20 hover:shadow-lg" 
                      : "text-red-600 hover:bg-red-50 hover:shadow-md"
                  )}
                >
                  <LogOut className="h-4 w-4 mr-3 transition-transform duration-300 group-hover:scale-110" />
                  <span className="font-medium">Logout</span>
                  {!transparent && (
                    <div className="absolute inset-0 bg-gradient-to-r from-red-500/10 to-pink-500/10 opacity-0 group-hover:opacity-100 transition-opacity duration-300"></div>
                  )}
                </Button>
              </>
            ) : (
              // Guest Mobile Menu
              <>
                <Link href="/login">
                  <Button 
                    variant="ghost"
                    size="sm" 
                    className={cn(
                      "w-full px-4 py-3 rounded-lg transition-all duration-300 group relative overflow-hidden",
                      transparent 
                        ? "bg-white/10 border-white/20 text-white hover:bg-white/20 hover:shadow-lg" 
                        : "text-gray-700 hover:text-green-600 hover:bg-red-50 hover:shadow-md"
                    )}
                  >
                    <span className="font-medium">Login</span>
                    {!transparent && (
                      <div className="absolute inset-0 bg-gradient-to-r from-green-500/10 to-red-500/10 opacity-0 group-hover:opacity-100 transition-opacity duration-300"></div>
                    )}
                  </Button>
                </Link>
                <Link href="/register">
                  <Button 
                    size="sm" 
                    className={cn(
                      "w-full px-4 py-3 rounded-lg transition-all duration-300 group relative overflow-hidden shadow-md hover:shadow-lg",
                      transparent 
                        ? "bg-white text-green-600 hover:bg-white/90 hover:shadow-xl" 
                        : "bg-gradient-to-r from-green-600 to-red-600 text-white hover:from-green-700 hover:to-red-700 hover:shadow-xl"
                    )}
                  >
                    <span className="font-medium">Register</span>
                    <div className="absolute inset-0 bg-gradient-to-r from-green-500/20 to-red-500/20 opacity-0 group-hover:opacity-100 transition-opacity duration-300"></div>
                  </Button>
                </Link>
              </>
            )}
          </div>
        </div>
      </div>
    </header>
    </>
  );
};

export default Header;