import React from 'react';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { Badge } from '@/components/ui/badge';
import { AlertTriangle, Shield, Award, Target, User, Trophy, ArrowRight } from 'lucide-react';
import { usePermissions } from '@/hooks/use-permissions';
import { Link } from '@inertiajs/react';

interface RoleBasedErrorProps {
  error: {
    message: string;
    code?: string;
    details?: string;
  };
  onRetry?: () => void;
  onGoBack?: () => void;
}

export function RoleBasedError({ error, onRetry, onGoBack }: RoleBasedErrorProps) {
  const { isAdmin, isOrganizer, isReferee, isPlayer } = usePermissions();

  // Determine primary role
  const getPrimaryRole = () => {
    if (isAdmin()) return 'admin';
    if (isOrganizer()) return 'organizer';
    if (isReferee()) return 'referee';
    if (isPlayer()) return 'player';
    return 'user';
  };

  const primaryRole = getPrimaryRole();

  const roleConfig = {
    admin: {
      title: 'Administrative Error',
      description: 'An error occurred while performing administrative tasks',
      icon: Shield,
      color: 'text-green-600',
      bgColor: 'bg-green-50',
      suggestions: [
        'Check system logs for detailed error information',
        'Verify database connectivity and permissions',
        'Review user permissions and role assignments',
        'Contact system administrator if issue persists'
      ],
      actions: [
        { label: 'System Settings', href: '/admin/settings', icon: Shield },
        { label: 'User Management', href: '/admin/users', icon: User },
        { label: 'System Logs', href: '/admin/logs', icon: AlertTriangle }
      ]
    },
    organizer: {
      title: 'Tournament Management Error',
      description: 'An error occurred while managing tournament data',
      icon: Award,
      color: 'text-blue-600',
      bgColor: 'bg-blue-50',
      suggestions: [
        'Verify tournament data integrity',
        'Check player and team assignments',
        'Ensure proper tournament configuration',
        'Contact administrator for system issues'
      ],
      actions: [
        { label: 'Tournaments', href: '/admin/tournaments', icon: Trophy },
        { label: 'Players', href: '/admin/players', icon: User },
        { label: 'Teams', href: '/admin/teams', icon: User }
      ]
    },
    referee: {
      title: 'Match Management Error',
      description: 'An error occurred while managing match data',
      icon: Target,
      color: 'text-orange-600',
      bgColor: 'bg-orange-50',
      suggestions: [
        'Verify match schedule and assignments',
        'Check team and player data',
        'Ensure proper match configuration',
        'Contact organizer for tournament issues'
      ],
      actions: [
        { label: 'Matches', href: '/admin/matches', icon: Target },
        { label: 'Tournaments', href: '/admin/tournaments', icon: Trophy },
        { label: 'Schedule', href: '/admin/matches/schedule', icon: Target }
      ]
    },
    player: {
      title: 'Player Access Error',
      description: 'An error occurred while accessing player features',
      icon: User,
      color: 'text-purple-600',
      bgColor: 'bg-purple-50',
      suggestions: [
        'Verify your tournament participation',
        'Check your team assignments',
        'Ensure your profile is complete',
        'Contact tournament organizer for assistance'
      ],
      actions: [
        { label: 'My Tournaments', href: '/player/tournaments', icon: Trophy },
        { label: 'My Matches', href: '/player/matches', icon: Target },
        { label: 'My Profile', href: '/player/profile', icon: User }
      ]
    },
    user: {
      title: 'General Error',
      description: 'An error occurred while accessing the system',
      icon: AlertTriangle,
      color: 'text-gray-600',
      bgColor: 'bg-gray-50',
      suggestions: [
        'Try refreshing the page',
        'Check your internet connection',
        'Clear browser cache and cookies',
        'Contact support if issue persists'
      ],
      actions: [
        { label: 'Dashboard', href: '/admin/dashboard', icon: Trophy },
        { label: 'Home', href: '/', icon: ArrowRight }
      ]
    }
  };

  const config = roleConfig[primaryRole as keyof typeof roleConfig];

  return (
    <div className="flex min-h-[calc(100vh-64px)] items-center justify-center bg-background px-4 py-12 sm:px-6 lg:px-8">
      <Card className="w-full max-w-2xl">
        <CardHeader className="text-center">
          <div className={`inline-flex p-4 rounded-full ${config.bgColor} mb-4`}>
            <config.icon className={`w-12 h-12 ${config.color}`} />
          </div>
          <CardTitle className="text-2xl font-bold">{config.title}</CardTitle>
          <CardDescription className="text-lg">{config.description}</CardDescription>
        </CardHeader>
        <CardContent className="space-y-6">
          {/* Error Details */}
          <div className="p-4 bg-red-50 border border-red-200 rounded-lg">
            <div className="flex items-start gap-3">
              <AlertTriangle className="w-5 h-5 text-red-600 mt-0.5" />
              <div>
                <h3 className="font-semibold text-red-800">Error Details</h3>
                <p className="text-red-700 text-sm mt-1">{error.message}</p>
                {error.code && (
                  <p className="text-red-600 text-xs mt-1">Error Code: {error.code}</p>
                )}
                {error.details && (
                  <p className="text-red-600 text-xs mt-1">{error.details}</p>
                )}
              </div>
            </div>
          </div>

          {/* Role-specific Suggestions */}
          <div className="space-y-3">
            <h3 className="font-semibold text-foreground">Suggestions:</h3>
            <ul className="space-y-2">
              {config.suggestions.map((suggestion, index) => (
                <li key={index} className="flex items-start gap-2 text-sm">
                  <div className={`w-1.5 h-1.5 rounded-full mt-2 ${config.color.replace('text-', 'bg-')}`} />
                  <span className="text-muted-foreground">{suggestion}</span>
                </li>
              ))}
            </ul>
          </div>

          {/* Role-specific Actions */}
          <div className="space-y-3">
            <h3 className="font-semibold text-foreground">Quick Actions:</h3>
            <div className="grid gap-2">
              {config.actions.map((action, index) => (
                <Link key={index} href={action.href}>
                  <Button variant="outline" className="w-full justify-start">
                    <action.icon className="w-4 h-4 mr-2" />
                    {action.label}
                  </Button>
                </Link>
              ))}
            </div>
          </div>

          {/* Action Buttons */}
          <div className="flex gap-3 pt-4">
            {onRetry && (
              <Button onClick={onRetry} className="flex-1">
                Try Again
              </Button>
            )}
            {onGoBack && (
              <Button variant="outline" onClick={onGoBack} className="flex-1">
                Go Back
              </Button>
            )}
            <Link href="/admin/dashboard">
              <Button variant="outline" className="flex-1">
                Go to Dashboard
              </Button>
            </Link>
          </div>
        </CardContent>
      </Card>
    </div>
  );
} 