import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { Alert, AlertDescription } from '@/components/ui/alert';
import { usePermissions } from '@/hooks/use-permissions';
import { Link } from '@inertiajs/react';
import { 
    ShieldX, 
    Lock, 
    AlertTriangle, 
    Info, 
    ArrowLeft, 
    Mail, 
    RefreshCw,
    Home
} from 'lucide-react';
import { ReactNode } from 'react';

interface PermissionErrorProps {
    type?: 'page' | 'card' | 'inline' | 'modal';
    title?: string;
    message?: string;
    requiredPermission?: string | string[];
    requiredRole?: string | string[];
    showContactAdmin?: boolean;
    showRetry?: boolean;
    showBackButton?: boolean;
    onRetry?: () => void;
    children?: ReactNode;
}

export function PermissionError({
    type = 'card',
    title,
    message,
    requiredPermission,
    requiredRole,
    showContactAdmin = true,
    showRetry = false,
    showBackButton = true,
    onRetry,
    children
}: PermissionErrorProps) {
    const { user, roles } = usePermissions();

    const defaultTitle = title || 'Access Denied';
    const defaultMessage = message || 'You do not have permission to access this resource.';

    const getIcon = () => {
        switch (type) {
            case 'page':
                return <ShieldX className="w-8 h-8 text-red-600" />;
            case 'inline':
                return <Lock className="w-4 h-4 text-red-600" />;
            default:
                return <Lock className="w-6 h-6 text-red-600" />;
        }
    };

    const getRequiredText = () => {
        const parts = [];
        if (requiredPermission) {
            const perms = Array.isArray(requiredPermission) ? requiredPermission : [requiredPermission];
            parts.push(`Permission: ${perms.join(' or ')}`);
        }
        if (requiredRole) {
            const roles = Array.isArray(requiredRole) ? requiredRole : [requiredRole];
            parts.push(`Role: ${roles.join(' or ')}`);
        }
        return parts.join(' and ');
    };

    if (type === 'inline') {
        return (
            <Alert variant="destructive" className="border-red-200">
                <AlertTriangle className="h-4 w-4" />
                <AlertDescription className="flex items-center justify-between">
                    <span>{defaultMessage}</span>
                    {showRetry && onRetry && (
                        <Button variant="outline" size="sm" onClick={onRetry}>
                            <RefreshCw className="w-3 h-3 mr-1" />
                            Retry
                        </Button>
                    )}
                </AlertDescription>
            </Alert>
        );
    }

    if (type === 'page') {
        return (
            <div className="min-h-screen flex items-center justify-center bg-gray-50 px-4">
                <div className="max-w-lg w-full text-center space-y-6">
                    {/* Icon */}
                    <div className="mx-auto w-16 h-16 bg-red-100 rounded-full flex items-center justify-center">
                        {getIcon()}
                    </div>

                    {/* Title and Message */}
                    <div className="space-y-2">
                        <h1 className="text-3xl font-bold text-gray-900">{defaultTitle}</h1>
                        <p className="text-lg text-gray-600">{defaultMessage}</p>
                    </div>

                    {/* User Info */}
                    <Card className="text-left">
                        <CardHeader className="pb-3">
                            <CardTitle className="text-sm font-medium text-gray-700">Current Session</CardTitle>
                        </CardHeader>
                        <CardContent className="space-y-2 text-sm">
                            <div className="flex justify-between">
                                <span className="text-gray-500">User:</span>
                                <span className="font-medium">{user?.name || 'Unknown'}</span>
                            </div>
                            <div className="flex justify-between">
                                <span className="text-gray-500">Email:</span>
                                <span className="font-medium">{user?.email || 'Unknown'}</span>
                            </div>
                            <div className="flex justify-between">
                                <span className="text-gray-500">Roles:</span>
                                <span className="font-medium">{roles.length > 0 ? roles.join(', ') : 'None'}</span>
                            </div>
                            {getRequiredText() && (
                                <div className="flex justify-between">
                                    <span className="text-gray-500">Required:</span>
                                    <span className="font-medium text-red-600">{getRequiredText()}</span>
                                </div>
                            )}
                        </CardContent>
                    </Card>

                    {/* Actions */}
                    <div className="flex flex-col sm:flex-row gap-3 justify-center">
                        {showBackButton && (
                            <Button variant="outline" asChild>
                                <Link href="/admin/dashboard">
                                    <ArrowLeft className="w-4 h-4 mr-2" />
                                    Back to Dashboard
                                </Link>
                            </Button>
                        )}
                        
                        {showRetry && onRetry && (
                            <Button variant="outline" onClick={onRetry}>
                                <RefreshCw className="w-4 h-4 mr-2" />
                                Try Again
                            </Button>
                        )}

                        <Button asChild>
                            <Link href="/">
                                <Home className="w-4 h-4 mr-2" />
                                Go Home
                            </Link>
                        </Button>
                    </div>

                    {/* Contact Admin */}
                    {showContactAdmin && (
                        <Alert>
                            <Info className="h-4 w-4" />
                            <AlertDescription>
                                If you believe you should have access to this resource, please contact your administrator.
                                {showContactAdmin && (
                                    <Button variant="link" className="p-0 h-auto ml-1" asChild>
                                        <a href="mailto:admin@example.com">
                                            <Mail className="w-3 h-3 mr-1" />
                                            Contact Support
                                        </a>
                                    </Button>
                                )}
                            </AlertDescription>
                        </Alert>
                    )}

                    {children}
                </div>
            </div>
        );
    }

    // Default card type
    return (
        <Card className="border-red-200 bg-red-50">
            <CardHeader>
                <div className="flex items-center gap-3">
                    <div className="w-10 h-10 bg-red-100 rounded-full flex items-center justify-center">
                        {getIcon()}
                    </div>
                    <div>
                        <CardTitle className="text-red-800">{defaultTitle}</CardTitle>
                        <CardDescription className="text-red-600">
                            {defaultMessage}
                        </CardDescription>
                    </div>
                </div>
            </CardHeader>
            <CardContent className="space-y-4">
                {/* User Information */}
                <div className="bg-white rounded-lg p-3 space-y-2 text-sm">
                    <div className="flex justify-between">
                        <span className="text-gray-500">Current User:</span>
                        <span className="font-medium">{user?.name || 'Unknown'}</span>
                    </div>
                    <div className="flex justify-between">
                        <span className="text-gray-500">Your Roles:</span>
                        <span className="font-medium">{roles.length > 0 ? roles.join(', ') : 'None assigned'}</span>
                    </div>
                    {getRequiredText() && (
                        <div className="flex justify-between">
                            <span className="text-gray-500">Required:</span>
                            <span className="font-medium text-red-600">{getRequiredText()}</span>
                        </div>
                    )}
                </div>

                {/* Actions */}
                <div className="flex flex-wrap gap-2">
                    {showRetry && onRetry && (
                        <Button variant="outline" size="sm" onClick={onRetry}>
                            <RefreshCw className="w-3 h-3 mr-1" />
                            Retry
                        </Button>
                    )}
                    
                    {showContactAdmin && (
                        <Button variant="outline" size="sm" asChild>
                            <a href="mailto:admin@example.com">
                                <Mail className="w-3 h-3 mr-1" />
                                Contact Admin
                            </a>
                        </Button>
                    )}
                </div>

                {children}
            </CardContent>
        </Card>
    );
}

// Convenience components for common error scenarios
export function AccessDeniedPage({ requiredRole, requiredPermission }: { requiredRole?: string | string[]; requiredPermission?: string | string[] }) {
    return (
        <PermissionError
            type="page"
            title="Access Denied"
            message="You don't have the required permissions to access this page."
            requiredRole={requiredRole}
            requiredPermission={requiredPermission}
            showContactAdmin={true}
            showBackButton={true}
        />
    );
}

export function InsufficientPermissions({ action, resource }: { action: string; resource: string }) {
    return (
        <PermissionError
            type="card"
            title="Insufficient Permissions"
            message={`You don't have permission to ${action} ${resource}.`}
            showContactAdmin={true}
            showRetry={false}
        />
    );
}

export function RoleRequired({ requiredRole }: { requiredRole: string | string[] }) {
    const roleText = Array.isArray(requiredRole) ? requiredRole.join(' or ') : requiredRole;
    return (
        <PermissionError
            type="card"
            title="Role Required"
            message={`This action requires the ${roleText} role.`}
            requiredRole={requiredRole}
            showContactAdmin={true}
        />
    );
}