import { Badge } from '@/components/ui/badge';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card';
import { Button } from '@/components/ui/button';
import { Trophy, Users, User, Award, Calendar, TrendingUp, UsersRound, Gamepad2, Shield, Clock, Target, ChevronRight } from 'lucide-react';
import { Link, usePage } from '@inertiajs/react';
import TeamSelector from '@/components/TeamManager/TeamSelector';
import { TeamRegistrationAlert } from '@/components/TeamManager/TeamRegistrationAlert';
import { CategoryBadge } from '@/components/ui/category-badge';

interface Team {
  id: number;
  name: string;
  description?: string;
  logo?: string;
  website?: string;
  created_at: string;
  category?: 'men' | 'women' | 'youth_male' | 'youth_female' | 'children';
}

interface TeamManagerDashboardProps {
  stats: {
    total_tournaments: number;
    active_tournaments: number;
    total_players?: number;
    total_teams?: number;
    total_matches?: number;
    completed_matches?: number;
    total_users?: number;
    recent_matches: any[];
  };
  recentActivity: {
    recent_tournaments?: any[];
    recent_players?: any[];
    recent_teams?: any[];
  };
  teams?: Team[];
}

export function TeamManagerDashboard({ stats, recentActivity, teams = [] }: TeamManagerDashboardProps) {
  const { props: pageProps } = usePage();
  
  const teamManagerStats = [
    { title: 'My Team Tournaments', value: stats.active_tournaments, icon: Trophy, color: 'text-blue-600', href: '/team/tournaments' },
    { title: 'Team Players', value: stats.total_players || 0, icon: Users, color: 'text-green-600', href: '/team/players' },
    { title: 'Team Matches', value: stats.total_matches || 0, icon: Gamepad2, color: 'text-purple-600', href: '/team/matches' },
    { title: 'Team Performance', value: '85%', icon: TrendingUp, color: 'text-orange-600', href: '/team/analytics' }
  ];

  const teamManagerQuickActions = [
    { title: 'Manage Team', href: '/team/profile', icon: UsersRound, description: 'Update team information' },
    { title: 'Team Players', href: '/team/players', icon: Users, description: 'Manage team roster' },
    { title: 'Formations', href: '/team/formations', icon: Target, description: 'Create tactical formations' },
    { title: 'View Matches', href: '/team/matches', icon: Gamepad2, description: 'Check upcoming matches' },
    { title: 'Team Analytics', href: '/team/analytics', icon: TrendingUp, description: 'View team performance' }
  ];

  return (
    <div className="space-y-6 p-4">      
      {/* Team Manager Welcome Header */}
      <div className="flex items-center gap-4 p-6 bg-gradient-to-r from-indigo-50 to-purple-50 rounded-lg border">
        <div className="p-3 bg-indigo-100 rounded-full">
          <UsersRound className="w-8 h-8 text-indigo-600" />
        </div>
        <div>
          <h1 className="text-2xl font-bold text-gray-900">Team Manager Dashboard</h1>
          <p className="text-gray-600">Manage your team, players, and tournament participation</p>
          <div className="flex items-center gap-2 mt-2">
            <Badge variant="default" className="bg-indigo-100 text-indigo-800">Team Manager</Badge>
            <span className="text-sm text-gray-500">Team management access</span>
          </div>
        </div>
      </div>

      {/* Teams Overview */}
      {teams.length > 0 && (
        <div className="space-y-4">
          <div className="flex items-center justify-between">
            <div>
              <h2 className="text-xl font-semibold text-gray-900">Your Teams</h2>
              <p className="text-sm text-gray-600">
                {teams.length} team{teams.length !== 1 ? 's' : ''} across{' '}
                {new Set(teams.map(t => t.category)).size} categor{new Set(teams.map(t => t.category)).size !== 1 ? 'ies' : 'y'}
              </p>
            </div>
            <Link href="/team/profile" className="text-sm text-indigo-600 hover:text-indigo-800">
              Manage Teams
            </Link>
          </div>
          
          {/* Category Statistics */}
          {(() => {
            const teamsByCategory = teams.reduce((acc, team) => {
              const cat = team.category || 'uncategorized';
              if (!acc[cat]) acc[cat] = [];
              acc[cat].push(team);
              return acc;
            }, {} as Record<string, typeof teams>);
            
            return Object.keys(teamsByCategory).length > 1 ? (
              <div className="grid gap-2 md:grid-cols-5 mb-4">
                {Object.entries(teamsByCategory).map(([category, categoryTeams]) => (
                  <Card key={category} className="bg-gradient-to-br from-gray-50 to-gray-100">
                    <CardContent className="p-3">
                      <div className="flex items-center gap-2">
                        <CategoryBadge category={category as any} size="sm" />
                        <span className="text-lg font-bold">{categoryTeams.length}</span>
                      </div>
                    </CardContent>
                  </Card>
                ))}
              </div>
            ) : null;
          })()}
          
          <div className="grid gap-4 md:grid-cols-2 lg:grid-cols-3">
            {teams.map((team) => (
              <Card key={team.id} className="hover:shadow-md transition-shadow">
                <CardContent className="p-4">
                  <div className="flex items-start justify-between">
                    <div className="flex-1">
                      <div className="flex items-center gap-2 mb-1">
                        <h3 className="font-semibold text-gray-900">{team.name}</h3>
                        {team.category && (
                          <CategoryBadge category={team.category} size="sm" />
                        )}
                      </div>
                      {team.description && (
                        <p className="text-sm text-gray-600 mt-1 line-clamp-2">{team.description}</p>
                      )}
                      <div className="flex items-center gap-2 mt-2">
                        <Badge variant="outline" className="text-xs">
                          Created {new Date(team.created_at).toLocaleDateString()}
                        </Badge>
                      </div>
                    </div>
                    <Link 
                      href={`/team/profile?team_id=${team.id}`}
                      className="p-1 hover:bg-gray-100 rounded"
                    >
                      <ChevronRight className="w-4 h-4 text-gray-400" />
                    </Link>
                  </div>
                </CardContent>
              </Card>
            ))}
          </div>
        </div>
      )}

      {/* Team Manager Statistics */}
      <div className="grid gap-4 md:grid-cols-2 lg:grid-cols-4">
        {teamManagerStats.map((stat, index) => (
          <Card key={index} className="hover:shadow-md transition-shadow">
            <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
              <CardTitle className="text-sm font-medium">
                {stat.title}
              </CardTitle>
              <div className={stat.color}>
                <stat.icon className="h-4 w-4" />
              </div>
            </CardHeader>
            <CardContent>
              <div className="text-2xl font-bold">{stat.value}</div>
              {stat.href && (
                <Link href={stat.href} className="text-xs text-muted-foreground hover:text-primary">
                  View details →
                </Link>
              )}
            </CardContent>
          </Card>
        ))}
      </div>

      {/* Quick Actions */}
      <Card>
        <CardHeader>
          <CardTitle className="flex items-center gap-2">
            <Target className="h-5 w-5" />
            Quick Actions
          </CardTitle>
          <CardDescription>
            Common team management tasks
          </CardDescription>
        </CardHeader>
        <CardContent>
          <div className="grid gap-4 md:grid-cols-2 lg:grid-cols-4">
            {teamManagerQuickActions.map((action, index) => (
              <Link key={index} href={action.href}>
                <Card className="hover:shadow-md transition-shadow cursor-pointer">
                  <CardContent className="p-4">
                    <div className="flex items-center gap-3">
                      <div className="p-2 bg-gray-100 rounded-lg">
                        <action.icon className="h-5 w-5 text-gray-600" />
                      </div>
                      <div>
                        <h3 className="font-medium text-sm">{action.title}</h3>
                        <p className="text-xs text-muted-foreground">{action.description}</p>
                      </div>
                    </div>
                  </CardContent>
                </Card>
              </Link>
            ))}
          </div>
        </CardContent>
      </Card>

      {/* Recent Team Activity */}
      <div className="grid gap-6 md:grid-cols-2">
        {/* Recent Matches */}
        <Card>
          <CardHeader>
            <CardTitle className="flex items-center gap-2">
              <Gamepad2 className="h-5 w-5" />
              Recent Team Matches
            </CardTitle>
            <CardDescription>
              Latest match activity for your team
            </CardDescription>
          </CardHeader>
          <CardContent>
            <div className="space-y-3">
              {stats.recent_matches && stats.recent_matches.length > 0 ? (
                stats.recent_matches.map((match, index) => (
                  <div key={index} className="flex items-center justify-between p-3 bg-gray-50 rounded-lg">
                    <div>
                      <p className="text-sm font-medium">
                        {match.team1?.name || 'Team 1'} vs {match.team2?.name || 'Team 2'}
                      </p>
                      <p className="text-xs text-muted-foreground">
                        {match.tournament?.name || 'Unknown Tournament'}
                      </p>
                    </div>
                    <div className="flex items-center gap-2">
                      <Badge variant={match.status === 'completed' ? 'default' : 'secondary'}>
                        {match.status}
                      </Badge>
                      {match.status === 'completed' && (
                        <span className="text-sm font-medium">
                          {match.score_team1} - {match.score_team2}
                        </span>
                      )}
                    </div>
                  </div>
                ))
              ) : (
                <div className="text-center py-6">
                  <Gamepad2 className="h-12 w-12 text-gray-300 mx-auto mb-2" />
                  <p className="text-sm text-muted-foreground">No recent matches</p>
                </div>
              )}
            </div>
          </CardContent>
        </Card>

        {/* Team Performance Overview */}
        <Card>
          <CardHeader>
            <CardTitle className="flex items-center gap-2">
              <TrendingUp className="h-5 w-5" />
              Team Performance
            </CardTitle>
            <CardDescription>
              Your team's tournament performance
            </CardDescription>
          </CardHeader>
          <CardContent>
            <div className="space-y-4">
              <div className="flex items-center justify-between">
                <span className="text-sm font-medium">Win Rate</span>
                <span className="text-2xl font-bold text-green-600">85%</span>
              </div>
              <div className="flex items-center justify-between">
                <span className="text-sm font-medium">Total Matches</span>
                <span className="text-lg font-semibold">{stats.total_matches || 0}</span>
              </div>
              <div className="flex items-center justify-between">
                <span className="text-sm font-medium">Wins</span>
                <span className="text-lg font-semibold text-green-600">
                  {Math.round((stats.total_matches || 0) * 0.85)}
                </span>
              </div>
              <div className="flex items-center justify-between">
                <span className="text-sm font-medium">Losses</span>
                <span className="text-lg font-semibold text-red-600">
                  {Math.round((stats.total_matches || 0) * 0.15)}
                </span>
              </div>
            </div>
          </CardContent>
        </Card>
      </div>

      {/* Upcoming Team Events */}
      <Card>
        <CardHeader>
          <CardTitle className="flex items-center gap-2">
            <Calendar className="h-5 w-5" />
            Upcoming Team Events
          </CardTitle>
          <CardDescription>
            Important dates and events for your team
          </CardDescription>
        </CardHeader>
        <CardContent>
          <div className="space-y-3">
            <div className="flex items-center gap-3 p-3 bg-blue-50 rounded-lg">
              <div className="p-2 bg-blue-100 rounded-full">
                <Clock className="h-4 w-4 text-blue-600" />
              </div>
              <div>
                <p className="text-sm font-medium">Next Match</p>
                <p className="text-xs text-muted-foreground">Tomorrow at 2:00 PM</p>
              </div>
            </div>
            <div className="flex items-center gap-3 p-3 bg-green-50 rounded-lg">
              <div className="p-2 bg-green-100 rounded-full">
                <Trophy className="h-4 w-4 text-green-600" />
              </div>
              <div>
                <p className="text-sm font-medium">Tournament Finals</p>
                <p className="text-xs text-muted-foreground">Next Saturday at 3:00 PM</p>
              </div>
            </div>
            <div className="flex items-center gap-3 p-3 bg-purple-50 rounded-lg">
              <div className="p-2 bg-purple-100 rounded-full">
                <Users className="h-4 w-4 text-purple-600" />
              </div>
              <div>
                <p className="text-sm font-medium">Team Meeting</p>
                <p className="text-xs text-muted-foreground">This Friday at 6:00 PM</p>
              </div>
            </div>
          </div>
        </CardContent>
      </Card>
    </div>
  );
}
