import { Badge } from '@/components/ui/badge';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card';
import { Trophy, Gamepad, Clock, Award, Calendar, Target, Play, CheckCircle } from 'lucide-react';
import { Link } from '@inertiajs/react';

interface RefereeDashboardProps {
  stats: {
    total_tournaments: number;
    active_tournaments: number;
    total_matches?: number;
    completed_matches?: number;
  };
  recentActivity: {
    recent_tournaments?: any[];
    recent_players?: any[];
    recent_teams?: any[];
  };
}

export function RefereeDashboard({ stats, recentActivity }: RefereeDashboardProps) {
  const refereeStats = [
    { title: 'Active Tournaments', value: stats.active_tournaments, icon: Trophy, color: 'text-green-600', href: '/admin/tournaments' },
    { title: 'Pending Matches', value: stats.total_matches - stats.completed_matches, icon: Clock, color: 'text-orange-600', href: '/admin/matches' },
    { title: 'Completed Matches', value: stats.completed_matches, icon: Award, color: 'text-green-600', href: '/admin/matches' },
    { title: 'Total Matches', value: stats.total_matches, icon: Gamepad, color: 'text-purple-600', href: '/admin/matches' }
  ];

  const refereeQuickActions = [
    { title: 'Schedule Match', href: '/admin/matches/create', icon: Play, description: 'Create new match schedule' },
    { title: 'View Matches', href: '/admin/matches', icon: Gamepad, description: 'Manage all matches' },
    { title: 'Tournament Overview', href: '/admin/tournaments', icon: Trophy, description: 'View tournament details' },
    { title: 'Match Reports', href: '/admin/matches/reports', icon: CheckCircle, description: 'Submit match reports' }
  ];

  return (
    <div className="space-y-6 p-4">
      {/* Referee Welcome Header */}
      <div className="flex items-center gap-4 p-6 bg-gradient-to-r from-orange-50 to-red-50 rounded-lg border">
        <div className="p-3 bg-orange-100 rounded-full">
          <Target className="w-8 h-8 text-orange-600" />
        </div>
        <div>
          <h1 className="text-2xl font-bold text-gray-900">Match Management</h1>
          <p className="text-gray-600">Oversee matches, scores, and tournament progress</p>
          <div className="flex items-center gap-2 mt-2">
            <Badge variant="default" className="bg-orange-100 text-orange-800">Referee</Badge>
            <span className="text-sm text-gray-500">Match management access</span>
          </div>
        </div>
      </div>

      {/* Referee Statistics */}
      <div className="grid gap-4 md:grid-cols-2 lg:grid-cols-4">
        {refereeStats.map((stat, index) => (
          <Card key={index} className="hover:shadow-md transition-shadow">
            <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
              <CardTitle className="text-sm font-medium">
                {stat.title}
              </CardTitle>
              <div className={stat.color}>
                <stat.icon className="h-4 w-4" />
              </div>
            </CardHeader>
            <CardContent>
              <div className="text-2xl font-bold">{stat.value}</div>
              {stat.href && (
                <Link href={stat.href} className="text-xs text-blue-600 hover:underline">
                  View details →
                </Link>
              )}
            </CardContent>
          </Card>
        ))}
      </div>

      {/* Referee Quick Actions */}
      <Card>
        <CardHeader>
          <CardTitle className="flex items-center gap-2">
            <Calendar className="h-5 w-5" />
            Match Actions
          </CardTitle>
          <CardDescription>
            Quick access to match management functions
          </CardDescription>
        </CardHeader>
        <CardContent>
          <div className="grid gap-4 md:grid-cols-2 lg:grid-cols-4">
            {refereeQuickActions.map((action, index) => (
              <Link key={index} href={action.href}>
                <div className="flex items-center gap-3 rounded-lg border p-4 hover:bg-accent transition-colors">
                  <div className="p-2 bg-orange-100 rounded-lg">
                    <action.icon className="h-5 w-5 text-orange-600" />
                  </div>
                  <div>
                    <p className="font-medium">{action.title}</p>
                    <p className="text-xs text-muted-foreground">{action.description}</p>
                  </div>
                </div>
              </Link>
            ))}
          </div>
        </CardContent>
      </Card>

      {/* Match Overview */}
      <div className="grid gap-6 md:grid-cols-2">
        {/* Recent Matches */}
        <Card>
          <CardHeader>
            <CardTitle className="flex items-center gap-2">
              <Gamepad className="h-5 w-5" />
              Recent Matches
            </CardTitle>
            <CardDescription>
              Your latest match activity
            </CardDescription>
          </CardHeader>
          <CardContent>
            <div className="space-y-3">
              {/* Sample match data - in real app this would come from props */}
              <div className="flex items-center justify-between">
                <div>
                  <p className="text-sm font-medium">Team Alpha vs Team Beta</p>
                  <p className="text-xs text-muted-foreground">Tournament Championship</p>
                </div>
                <Badge variant="default">Completed</Badge>
              </div>
              <div className="flex items-center justify-between">
                <div>
                  <p className="text-sm font-medium">Team Gamma vs Team Delta</p>
                  <p className="text-xs text-muted-foreground">Quarter Finals</p>
                </div>
                <Badge variant="secondary">Scheduled</Badge>
              </div>
              <div className="flex items-center justify-between">
                <div>
                  <p className="text-sm font-medium">Team Echo vs Team Foxtrot</p>
                  <p className="text-xs text-muted-foreground">Semi Finals</p>
                </div>
                <Badge variant="outline">Pending</Badge>
              </div>
            </div>
          </CardContent>
        </Card>

        {/* Match Performance */}
        <Card>
          <CardHeader>
            <CardTitle className="flex items-center gap-2">
              <Award className="h-5 w-5" />
              Match Performance
            </CardTitle>
            <CardDescription>
              Your refereeing statistics
            </CardDescription>
          </CardHeader>
          <CardContent>
            <div className="space-y-4">
              <div className="flex items-center justify-between">
                <span className="text-sm">Matches Officiated</span>
                <Badge variant="default">{stats.completed_matches}</Badge>
              </div>
              <div className="flex items-center justify-between">
                <span className="text-sm">Pending Matches</span>
                <Badge variant="secondary">{stats.total_matches - stats.completed_matches}</Badge>
              </div>
              <div className="flex items-center justify-between">
                <span className="text-sm">Tournaments</span>
                <Badge variant="outline">{stats.active_tournaments}</Badge>
              </div>
              <div className="flex items-center justify-between">
                <span className="text-sm">Completion Rate</span>
                <Badge variant="default" className="bg-green-100 text-green-800">
                  {stats.total_matches > 0 ? Math.round((stats.completed_matches / stats.total_matches) * 100) : 0}%
                </Badge>
              </div>
            </div>
          </CardContent>
        </Card>
      </div>

      {/* Upcoming Matches */}
      <Card>
        <CardHeader>
          <CardTitle className="flex items-center gap-2">
            <Clock className="h-5 w-5" />
            Upcoming Matches
          </CardTitle>
          <CardDescription>
            Matches scheduled for the next 7 days
          </CardDescription>
        </CardHeader>
        <CardContent>
          <div className="space-y-3">
            <div className="flex items-center justify-between p-3 bg-blue-50 rounded-lg">
              <div>
                <p className="text-sm font-medium">Team Alpha vs Team Beta</p>
                <p className="text-xs text-muted-foreground">Tomorrow at 2:00 PM</p>
              </div>
              <div className="flex items-center gap-2">
                <Badge variant="outline">Tournament Championship</Badge>
                <Link href="/admin/matches/1" className="text-xs text-blue-600 hover:underline">
                  View Details →
                </Link>
              </div>
            </div>
            <div className="flex items-center justify-between p-3 bg-green-50 rounded-lg">
              <div>
                <p className="text-sm font-medium">Team Gamma vs Team Delta</p>
                <p className="text-xs text-muted-foreground">Friday at 3:30 PM</p>
              </div>
              <div className="flex items-center gap-2">
                <Badge variant="outline">Quarter Finals</Badge>
                <Link href="/admin/matches/2" className="text-xs text-blue-600 hover:underline">
                  View Details →
                </Link>
              </div>
            </div>
          </div>
        </CardContent>
      </Card>
    </div>
  );
} 