import { Badge } from '@/components/ui/badge';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card';
import { Trophy, Gamepad, User, Award, Calendar, TrendingUp, Star, UsersRound } from 'lucide-react';
import { Link } from '@inertiajs/react';

interface PlayerDashboardProps {
  stats: {
    total_tournaments: number;
    active_tournaments: number;
    total_matches?: number;
    completed_matches?: number;
    wins?: number;
    win_rate?: number;
  };
}

export function PlayerDashboard({ stats }: PlayerDashboardProps) {
  const playerStats = [
    { title: 'My Tournaments', value: stats?.total_tournaments || 0, icon: Trophy, color: 'text-blue-600' },
    { title: 'My Matches', value: stats?.total_matches || 0, icon: Gamepad, color: 'text-green-600' },
    { title: 'Wins', value: stats?.wins || 0, icon: Award, color: 'text-yellow-600' },
    { title: 'Win Rate', value: stats?.win_rate ? `${stats.win_rate}%` : '0%', icon: TrendingUp, color: 'text-purple-600' }
  ];

  return (
    <div className="space-y-6 p-4">
      {/* Player Welcome Header */}
      <div className="flex items-center gap-4 p-6 bg-gradient-to-r from-purple-50 to-pink-50 rounded-lg border">
        <div className="p-3 bg-purple-100 rounded-full">
          <User className="w-8 h-8 text-purple-600" />
        </div>
        <div>
          <h1 className="text-2xl font-bold text-gray-900">Player Dashboard</h1>
          <p className="text-gray-600">View your tournaments, matches, and performance</p>
          <div className="flex items-center gap-2 mt-2">
            <Badge variant="default" className="bg-purple-100 text-purple-800">Player</Badge>
            <span className="text-sm text-gray-500">Tournament participation access</span>
          </div>
        </div>
      </div>

      {/* Player Statistics */}
      <div className="grid gap-4 md:grid-cols-2 lg:grid-cols-4">
        {playerStats.map((stat, index) => (
          <Card key={index} className="hover:shadow-md transition-shadow">
            <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
              <CardTitle className="text-sm font-medium">
                {stat.title}
              </CardTitle>
              <div className={stat.color}>
                <stat.icon className="h-4 w-4" />
              </div>
            </CardHeader>
            <CardContent>
              <div className="text-2xl font-bold">{typeof stat.value === 'number' ? stat.value.toLocaleString() : stat.value}</div>
            </CardContent>
          </Card>
        ))}
      </div>

      {/* Player Quick Actions */}
      <Card>
        <CardHeader>
          <CardTitle className="flex items-center gap-2">
            <Calendar className="h-5 w-5" />
            Quick Actions
          </CardTitle>
          <CardDescription>
            Access your player features
          </CardDescription>
        </CardHeader>
        <CardContent>
          <div className="grid gap-4 md:grid-cols-2 lg:grid-cols-4">
            <Link href="/player/tournaments">
              <div className="flex items-center gap-3 rounded-lg border p-4 hover:bg-accent transition-colors">
                <div className="p-2 bg-purple-100 rounded-lg">
                  <Trophy className="h-5 w-5 text-purple-600" />
                </div>
                <div>
                  <p className="font-medium">My Tournaments</p>
                  <p className="text-xs text-muted-foreground">View your tournaments</p>
                </div>
              </div>
            </Link>
            <Link href="/player/team">
              <div className="flex items-center gap-3 rounded-lg border p-4 hover:bg-accent transition-colors">
                <div className="p-2 bg-purple-100 rounded-lg">
                  <UsersRound className="h-5 w-5 text-purple-600" />
                </div>
                <div>
                  <p className="font-medium">My Team</p>
                  <p className="text-xs text-muted-foreground">View team information</p>
                </div>
              </div>
            </Link>
            <Link href="/player/matches">
              <div className="flex items-center gap-3 rounded-lg border p-4 hover:bg-accent transition-colors">
                <div className="p-2 bg-purple-100 rounded-lg">
                  <Gamepad className="h-5 w-5 text-purple-600" />
                </div>
                <div>
                  <p className="font-medium">My Matches</p>
                  <p className="text-xs text-muted-foreground">Check your match schedule</p>
                </div>
              </div>
            </Link>
            <Link href="/player/profile">
              <div className="flex items-center gap-3 rounded-lg border p-4 hover:bg-accent transition-colors">
                <div className="p-2 bg-purple-100 rounded-lg">
                  <User className="h-5 w-5 text-purple-600" />
                </div>
                <div>
                  <p className="font-medium">My Profile</p>
                  <p className="text-xs text-muted-foreground">Update your profile</p>
                </div>
              </div>
            </Link>
            <Link href="/player/performance">
              <div className="flex items-center gap-3 rounded-lg border p-4 hover:bg-accent transition-colors">
                <div className="p-2 bg-purple-100 rounded-lg">
                  <Star className="h-5 w-5 text-purple-600" />
                </div>
                <div>
                  <p className="font-medium">Performance</p>
                  <p className="text-xs text-muted-foreground">View your statistics</p>
                </div>
              </div>
            </Link>
          </div>
        </CardContent>
      </Card>

      {/* Recent Matches */}
      <Card>
        <CardHeader>
          <CardTitle className="flex items-center gap-2">
            <Gamepad className="h-5 w-5" />
            Recent Matches
          </CardTitle>
          <CardDescription>
            Your latest match results
          </CardDescription>
        </CardHeader>
        <CardContent>
          <div className="space-y-3">
            <div className="flex items-center justify-between p-3 bg-green-50 rounded-lg">
              <div>
                <p className="text-sm font-medium">Team Alpha vs Team Beta</p>
                <p className="text-xs text-muted-foreground">Championship 2024 - Semi Finals</p>
              </div>
              <div className="flex items-center gap-2">
                <Badge variant="default" className="bg-green-100 text-green-800">Won</Badge>
                <span className="text-sm font-medium">3 - 1</span>
              </div>
            </div>
            <div className="flex items-center justify-between p-3 bg-red-50 rounded-lg">
              <div>
                <p className="text-sm font-medium">Team Gamma vs Team Delta</p>
                <p className="text-xs text-muted-foreground">Spring League - Quarter Finals</p>
              </div>
              <div className="flex items-center gap-2">
                <Badge variant="destructive">Lost</Badge>
                <span className="text-sm font-medium">1 - 2</span>
              </div>
            </div>
          </div>
        </CardContent>
      </Card>
    </div>
  );
} 