import { Badge } from '@/components/ui/badge';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card';
import { Trophy, Users, User, Award, Calendar, TrendingUp, Plus, Users2 } from 'lucide-react';
import { Link } from '@inertiajs/react';

interface OrganizerDashboardProps {
  stats: {
    total_tournaments: number;
    active_tournaments: number;
    total_players?: number;
    total_teams?: number;
    total_matches?: number;
    completed_matches?: number;
  };
  recentActivity: {
    recent_tournaments?: any[];
    recent_players?: any[];
    recent_teams?: any[];
  };
}

export function OrganizerDashboard({ stats, recentActivity }: OrganizerDashboardProps) {
  const organizerStats = [
    { title: 'Active Tournaments', value: stats.active_tournaments, icon: Trophy, color: 'text-green-600', href: '/admin/tournaments' },
    { title: 'Total Players', value: stats.total_players, icon: User, color: 'text-blue-600', href: '/admin/players' },
    { title: 'Total Teams', value: stats.total_teams, icon: Users2, color: 'text-purple-600', href: '/admin/teams' },
    { title: 'Total Matches', value: stats.total_matches, icon: Award, color: 'text-orange-600', href: '/admin/matches' }
  ];

  const organizerQuickActions = [
    { title: 'Create Tournament', href: '/admin/tournaments/create', icon: Plus, description: 'Start a new tournament' },
    { title: 'Add Players', href: '/admin/players/create', icon: User, description: 'Register new players' },
    { title: 'Create Teams', href: '/admin/teams/create', icon: Users2, description: 'Form new teams' },
    { title: 'View Analytics', href: '/admin/analytics', icon: TrendingUp, description: 'Tournament insights' }
  ];

  return (
    <div className="space-y-6 p-4">
      {/* Organizer Welcome Header */}
      <div className="flex items-center gap-4 p-6 bg-gradient-to-r from-blue-50 to-purple-50 rounded-lg border">
        <div className="p-3 bg-blue-100 rounded-full">
          <Award className="w-8 h-8 text-blue-600" />
        </div>
        <div>
          <h1 className="text-2xl font-bold text-gray-900">Tournament Management</h1>
          <p className="text-gray-600">Create and manage tournaments, players, and teams</p>
          <div className="flex items-center gap-2 mt-2">
            <Badge variant="default" className="bg-blue-100 text-blue-800">Organizer</Badge>
            <span className="text-sm text-gray-500">Tournament management access</span>
          </div>
        </div>
      </div>

      {/* Organizer Statistics */}
      <div className="grid gap-4 md:grid-cols-2 lg:grid-cols-4">
        {organizerStats.map((stat, index) => (
          <Card key={index} className="hover:shadow-md transition-shadow">
            <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
              <CardTitle className="text-sm font-medium">
                {stat.title}
              </CardTitle>
              <div className={stat.color}>
                <stat.icon className="h-4 w-4" />
              </div>
            </CardHeader>
            <CardContent>
              <div className="text-2xl font-bold">{stat.value}</div>
              {stat.href && (
                <Link href={stat.href} className="text-xs text-blue-600 hover:underline">
                  View details →
                </Link>
              )}
            </CardContent>
          </Card>
        ))}
      </div>

      {/* Organizer Quick Actions */}
      <Card>
        <CardHeader>
          <CardTitle className="flex items-center gap-2">
            <Calendar className="h-5 w-5" />
            Tournament Actions
          </CardTitle>
          <CardDescription>
            Quick access to tournament management functions
          </CardDescription>
        </CardHeader>
        <CardContent>
          <div className="grid gap-4 md:grid-cols-2 lg:grid-cols-4">
            {organizerQuickActions.map((action, index) => (
              <Link key={index} href={action.href}>
                <div className="flex items-center gap-3 rounded-lg border p-4 hover:bg-accent transition-colors">
                  <div className="p-2 bg-blue-100 rounded-lg">
                    <action.icon className="h-5 w-5 text-blue-600" />
                  </div>
                  <div>
                    <p className="font-medium">{action.title}</p>
                    <p className="text-xs text-muted-foreground">{action.description}</p>
                  </div>
                </div>
              </Link>
            ))}
          </div>
        </CardContent>
      </Card>

      {/* Tournament Overview */}
      <div className="grid gap-6 md:grid-cols-2">
        {/* Recent Tournaments */}
        <Card>
          <CardHeader>
            <CardTitle className="flex items-center gap-2">
              <Trophy className="h-5 w-5" />
              Recent Tournaments
            </CardTitle>
            <CardDescription>
              Your latest tournament activity
            </CardDescription>
          </CardHeader>
          <CardContent>
            <div className="space-y-3">
              {recentActivity.recent_tournaments && recentActivity.recent_tournaments.length > 0 ? (
                recentActivity.recent_tournaments.map((tournament, index) => (
                  <div key={index} className="flex items-center justify-between">
                    <div>
                      <p className="text-sm font-medium">{tournament.name}</p>
                      <p className="text-xs text-muted-foreground">
                        {tournament.start_date ? new Date(tournament.start_date).toLocaleDateString() : 'No date'}
                      </p>
                    </div>
                    <Badge variant={tournament.status === 'active' ? 'default' : 'secondary'}>
                      {tournament.status}
                    </Badge>
                  </div>
                ))
              ) : (
                <p className="text-sm text-muted-foreground">No recent tournaments</p>
              )}
            </div>
          </CardContent>
        </Card>

        {/* Recent Players */}
        <Card>
          <CardHeader>
            <CardTitle className="flex items-center gap-2">
              <Users className="h-5 w-5" />
              Recent Players
            </CardTitle>
            <CardDescription>
              Newly registered players in your tournaments
            </CardDescription>
          </CardHeader>
          <CardContent>
            <div className="space-y-3">
              {recentActivity.recent_players && recentActivity.recent_players.length > 0 ? (
                recentActivity.recent_players.map((player, index) => (
                  <div key={index} className="flex items-center justify-between">
                    <div>
                      <p className="text-sm font-medium">
                        {player.user?.name || player.display_name || 'Unknown Player'}
                      </p>
                      <p className="text-xs text-muted-foreground">
                        {player.created_at ? new Date(player.created_at).toLocaleDateString() : 'No date'}
                      </p>
                    </div>
                    <Badge variant="outline">
                      {player.status || 'active'}
                    </Badge>
                  </div>
                ))
              ) : (
                <p className="text-sm text-muted-foreground">No recent players</p>
              )}
            </div>
          </CardContent>
        </Card>
      </div>

      {/* Tournament Performance */}
      <Card>
        <CardHeader>
          <CardTitle className="flex items-center gap-2">
            <TrendingUp className="h-5 w-5" />
            Tournament Performance
          </CardTitle>
          <CardDescription>
            Overview of your tournament management
          </CardDescription>
        </CardHeader>
        <CardContent>
          <div className="grid gap-4 md:grid-cols-3">
            <div className="text-center p-4 bg-green-50 rounded-lg">
              <div className="text-2xl font-bold text-green-600">{stats.active_tournaments}</div>
              <div className="text-sm text-gray-600">Active Tournaments</div>
            </div>
            <div className="text-center p-4 bg-blue-50 rounded-lg">
              <div className="text-2xl font-bold text-blue-600">{stats.total_players}</div>
              <div className="text-sm text-gray-600">Registered Players</div>
            </div>
            <div className="text-center p-4 bg-purple-50 rounded-lg">
              <div className="text-2xl font-bold text-purple-600">{stats.completed_matches}</div>
              <div className="text-sm text-gray-600">Completed Matches</div>
            </div>
          </div>
        </CardContent>
      </Card>
    </div>
  );
} 