import { Badge } from '@/components/ui/badge';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card';
import { Trophy, Users, BarChart3, Settings, Shield, TrendingUp, Activity, Database, Calendar } from 'lucide-react';
import { Link } from '@inertiajs/react';

interface AdminDashboardProps {
  stats: {
    total_tournaments: number;
    active_tournaments: number;
    total_users?: number;
    total_players?: number;
    total_teams?: number;
    total_matches?: number;
    completed_matches?: number;
  };
  recentActivity: {
    recent_tournaments?: any[];
    recent_players?: any[];
    recent_teams?: any[];
  };
}

export function AdminDashboard({ stats, recentActivity }: AdminDashboardProps) {
  const adminStats = [
    { title: 'Total Users', value: stats.total_users, icon: Users, color: 'text-blue-600', href: '/admin/users' },
    { title: 'Active Tournaments', value: stats.active_tournaments, icon: Trophy, color: 'text-green-600', href: '/admin/tournaments' },
    { title: 'Total Players', value: stats.total_players, icon: Users, color: 'text-purple-600', href: '/admin/players' },
    { title: 'System Health', value: 'Good', icon: Shield, color: 'text-green-600', href: '/admin/settings' }
  ];

  const adminQuickActions = [
    { title: 'Manage Users', href: '/admin/users', icon: Users, description: 'View user accounts' },
    { title: 'Tournaments', href: '/admin/tournaments', icon: Trophy, description: 'View tournaments' },
    { title: 'Matches', href: '/admin/matches', icon: Calendar, description: 'View matches' },
    { title: 'Teams', href: '/admin/teams', icon: Users, description: 'View teams' }
  ];

  return (
    <div className="space-y-6 p-4">
      {/* Admin Welcome Header */}
      <div className="flex items-center gap-4 p-6 bg-gradient-to-r from-green-50 to-blue-50 rounded-lg border">
        <div className="p-3 bg-green-100 rounded-full">
          <Shield className="w-8 h-8 text-green-600" />
        </div>
        <div>
          <h1 className="text-2xl font-bold text-gray-900">System Administration</h1>
          <p className="text-gray-600">Manage tournaments, users, and system settings</p>
          <div className="flex items-center gap-2 mt-2">
            <Badge variant="default" className="bg-green-100 text-green-800">Administrator</Badge>
            <span className="text-sm text-gray-500">Full system access</span>
          </div>
        </div>
      </div>

      {/* Admin Statistics */}
      <div className="grid gap-4 md:grid-cols-2 lg:grid-cols-4">
        {adminStats.map((stat, index) => (
          <Card key={index} className="hover:shadow-md transition-shadow">
            <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
              <CardTitle className="text-sm font-medium">
                {stat.title}
              </CardTitle>
              <div className={stat.color}>
                <stat.icon className="h-4 w-4" />
              </div>
            </CardHeader>
            <CardContent>
              <div className="text-2xl font-bold">{stat.value}</div>
              {stat.href && (
                <Link href={stat.href} className="text-xs text-blue-600 hover:underline">
                  View details →
                </Link>
              )}
            </CardContent>
          </Card>
        ))}
      </div>

      {/* Admin Quick Actions */}
      <Card>
        <CardHeader>
          <CardTitle className="flex items-center gap-2">
            <Activity className="h-5 w-5" />
            Administrative Actions
          </CardTitle>
          <CardDescription>
            Quick access to system management functions
          </CardDescription>
        </CardHeader>
        <CardContent>
          <div className="grid gap-4 md:grid-cols-2 lg:grid-cols-4">
            {adminQuickActions.map((action, index) => (
              <Link key={index} href={action.href}>
                <div className="flex items-center gap-3 rounded-lg border p-4 hover:bg-accent transition-colors">
                  <div className="p-2 bg-blue-100 rounded-lg">
                    <action.icon className="h-5 w-5 text-blue-600" />
                  </div>
                  <div>
                    <p className="font-medium">{action.title}</p>
                    <p className="text-xs text-muted-foreground">{action.description}</p>
                  </div>
                </div>
              </Link>
            ))}
          </div>
        </CardContent>
      </Card>

      {/* System Overview */}
      <div className="grid gap-6 md:grid-cols-2">
        {/* Recent Users */}
        <Card>
          <CardHeader>
            <CardTitle className="flex items-center gap-2">
              <Users className="h-5 w-5" />
              Recent Users
            </CardTitle>
            <CardDescription>
              Newly registered users in the system
            </CardDescription>
          </CardHeader>
          <CardContent>
            <div className="space-y-3">
              {recentActivity.recent_players && recentActivity.recent_players.length > 0 ? (
                recentActivity.recent_players.map((player, index) => (
                  <div key={index} className="flex items-center justify-between">
                    <div>
                      <p className="text-sm font-medium">
                        {player.user?.name || player.display_name || 'Unknown User'}
                      </p>
                      <p className="text-xs text-muted-foreground">
                        {player.created_at ? new Date(player.created_at).toLocaleDateString() : 'No date'}
                      </p>
                    </div>
                    <Badge variant="outline">
                      {player.status || 'active'}
                    </Badge>
                  </div>
                ))
              ) : (
                <p className="text-sm text-muted-foreground">No recent users</p>
              )}
            </div>
          </CardContent>
        </Card>

        {/* System Status */}
        <Card>
          <CardHeader>
            <CardTitle className="flex items-center gap-2">
              <Database className="h-5 w-5" />
              System Status
            </CardTitle>
            <CardDescription>
              Current system health and performance
            </CardDescription>
          </CardHeader>
          <CardContent>
            <div className="space-y-4">
              <div className="flex items-center justify-between">
                <span className="text-sm">Database</span>
                <Badge variant="default" className="bg-green-100 text-green-800">Healthy</Badge>
              </div>
              <div className="flex items-center justify-between">
                <span className="text-sm">Cache</span>
                <Badge variant="default" className="bg-green-100 text-green-800">Active</Badge>
              </div>
              <div className="flex items-center justify-between">
                <span className="text-sm">Email Service</span>
                <Badge variant="default" className="bg-green-100 text-green-800">Online</Badge>
              </div>
              <div className="flex items-center justify-between">
                <span className="text-sm">Storage</span>
                <Badge variant="outline">75% Used</Badge>
              </div>
            </div>
          </CardContent>
        </Card>
      </div>
    </div>
  );
} 