import { usePermissions } from '@/hooks/use-permissions';
import { ReactNode } from 'react';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card';
import { AlertTriangle, Lock, ShieldX } from 'lucide-react';
import { Button } from '@/components/ui/button';
import { Link } from '@inertiajs/react';

interface ProtectedRouteProps {
    permission?: string | string[];
    role?: string | string[];
    requireAll?: boolean;
    children: ReactNode;
    fallbackType?: 'card' | 'page' | 'redirect';
    fallbackMessage?: string;
    redirectTo?: string;
}

/**
 * ProtectedRoute component - Protects entire pages/sections based on permissions
 */
export function ProtectedRoute({
    permission,
    role,
    requireAll = false,
    children,
    fallbackType = 'card',
    fallbackMessage,
    redirectTo = '/admin/dashboard'
}: ProtectedRouteProps) {
    const { hasPermission, hasRole, hasAllPermissions, hasAllRoles, user } = usePermissions();

    let hasAccess = false;

    // Check permissions
    if (permission) {
        const permissions = Array.isArray(permission) ? permission : [permission];
        hasAccess = requireAll 
            ? hasAllPermissions(permissions)
            : hasPermission(permission);
    }

    // Check roles
    if (role && (!permission || hasAccess)) {
        const roles = Array.isArray(role) ? role : [role];
        const roleAccess = requireAll 
            ? hasAllRoles(roles)
            : hasRole(role);
        
        hasAccess = permission ? (hasAccess && roleAccess) : roleAccess;
    }

    // If neither permission nor role specified, deny access
    if (!permission && !role) {
        hasAccess = false;
    }

    if (hasAccess) {
        return <>{children}</>;
    }

    // Render appropriate fallback
    const defaultMessage = fallbackMessage || 'You do not have permission to access this resource.';

    if (fallbackType === 'redirect') {
        window.location.href = redirectTo;
        return null;
    }

    if (fallbackType === 'page') {
        return (
            <div className="min-h-screen flex items-center justify-center bg-gray-50">
                <div className="max-w-md w-full">
                    <Card>
                        <CardHeader className="text-center">
                            <div className="mx-auto w-12 h-12 bg-red-100 rounded-full flex items-center justify-center mb-4">
                                <ShieldX className="w-6 h-6 text-red-600" />
                            </div>
                            <CardTitle className="text-xl">Access Denied</CardTitle>
                            <CardDescription>
                                {defaultMessage}
                            </CardDescription>
                        </CardHeader>
                        <CardContent className="text-center">
                            <div className="space-y-4">
                                <p className="text-sm text-gray-600">
                                    Current user: <strong>{user?.name || 'Unknown'}</strong>
                                </p>
                                <p className="text-sm text-gray-600">
                                    Required: {permission && `Permission: ${Array.isArray(permission) ? permission.join(', ') : permission}`}
                                    {permission && role && ' or '}
                                    {role && `Role: ${Array.isArray(role) ? role.join(', ') : role}`}
                                </p>
                                <Button asChild>
                                    <Link href={redirectTo}>
                                        Return to Dashboard
                                    </Link>
                                </Button>
                            </div>
                        </CardContent>
                    </Card>
                </div>
            </div>
        );
    }

    // Default card fallback
    return (
        <Card className="border-red-200">
            <CardHeader>
                <div className="flex items-center gap-2">
                    <Lock className="w-5 h-5 text-red-600" />
                    <CardTitle className="text-red-800">Access Restricted</CardTitle>
                </div>
                <CardDescription>
                    {defaultMessage}
                </CardDescription>
            </CardHeader>
            <CardContent>
                <div className="flex items-center gap-2 text-sm text-gray-600">
                    <AlertTriangle className="w-4 h-4" />
                    <span>
                        Contact your administrator if you believe you should have access.
                    </span>
                </div>
            </CardContent>
        </Card>
    );
}

// Convenience components for common route protection patterns
export function AdminRoute({ children, fallbackType = 'page' }: { children: ReactNode; fallbackType?: 'card' | 'page' | 'redirect' }) {
    return (
        <ProtectedRoute 
            role={['admin', 'super_admin']} 
            fallbackType={fallbackType}
            fallbackMessage="This section is restricted to administrators only."
        >
            {children}
        </ProtectedRoute>
    );
}

export function OrganizerRoute({ children, fallbackType = 'page' }: { children: ReactNode; fallbackType?: 'card' | 'page' | 'redirect' }) {
    return (
        <ProtectedRoute 
            role={['organizer', 'admin', 'super_admin']} 
            fallbackType={fallbackType}
            fallbackMessage="This section is restricted to tournament organizers and administrators."
        >
            {children}
        </ProtectedRoute>
    );
}

export function RefereeRoute({ children, fallbackType = 'page' }: { children: ReactNode; fallbackType?: 'card' | 'page' | 'redirect' }) {
    return (
        <ProtectedRoute 
            role={['referee', 'admin', 'super_admin']} 
            fallbackType={fallbackType}
            fallbackMessage="This section is restricted to referees and administrators."
        >
            {children}
        </ProtectedRoute>
    );
}