import { usePermissions } from '@/hooks/use-permissions';
import { Link, InertiaLinkProps } from '@inertiajs/react';
import { ReactNode } from 'react';
import { cn } from '@/lib/utils';

interface PermissionLinkProps extends InertiaLinkProps {
    permission?: string | string[];
    role?: string | string[];
    requireAll?: boolean;
    children: ReactNode;
    fallbackContent?: ReactNode;
    hideWhenNoAccess?: boolean;
    disabledClassName?: string;
}

/**
 * PermissionLink component - Link that shows/hides or disables based on permissions
 */
export function PermissionLink({
    permission,
    role,
    requireAll = false,
    children,
    fallbackContent = null,
    hideWhenNoAccess = false,
    disabledClassName = 'opacity-50 cursor-not-allowed pointer-events-none',
    className,
    ...linkProps
}: PermissionLinkProps) {
    const { hasPermission, hasRole, hasAllPermissions, hasAllRoles } = usePermissions();

    let hasAccess = false;

    // Check permissions
    if (permission) {
        const permissions = Array.isArray(permission) ? permission : [permission];
        hasAccess = requireAll 
            ? hasAllPermissions(permissions)
            : hasPermission(permission);
    }

    // Check roles
    if (role && (!permission || hasAccess)) {
        const roles = Array.isArray(role) ? role : [role];
        const roleAccess = requireAll 
            ? hasAllRoles(roles)
            : hasRole(role);
        
        hasAccess = permission ? (hasAccess && roleAccess) : roleAccess;
    }

    // If neither permission nor role specified, grant access
    if (!permission && !role) {
        hasAccess = true;
    }

    // Hide link if no access and hideWhenNoAccess is true
    if (!hasAccess && hideWhenNoAccess) {
        return fallbackContent ? <>{fallbackContent}</> : null;
    }

    // If no access, render as disabled span
    if (!hasAccess) {
        return (
            <span className={cn(className, disabledClassName)}>
                {children}
            </span>
        );
    }

    return (
        <Link {...linkProps} className={className}>
            {children}
        </Link>
    );
}

// Convenience components for common link patterns
interface ResourceLinkProps extends Omit<PermissionLinkProps, 'permission'> {
    resource: string;
}

export function ViewLink({ resource, children, ...props }: ResourceLinkProps) {
    return (
        <PermissionLink 
            permission={[`${resource}.view`, `${resource}.manage`]}
            {...props}
        >
            {children}
        </PermissionLink>
    );
}

export function CreateLink({ resource, children, ...props }: ResourceLinkProps) {
    return (
        <PermissionLink 
            permission={[`${resource}.create`, `${resource}.manage`]}
            {...props}
        >
            {children}
        </PermissionLink>
    );
}

export function EditLink({ resource, children, ...props }: ResourceLinkProps) {
    return (
        <PermissionLink 
            permission={[`${resource}.edit`, `${resource}.manage`]}
            {...props}
        >
            {children}
        </PermissionLink>
    );
}

export function AdminLink({ children, ...props }: Omit<PermissionLinkProps, 'role'>) {
    return (
        <PermissionLink 
            role={['admin', 'super_admin']}
            {...props}
        >
            {children}
        </PermissionLink>
    );
}

export function OrganizerLink({ children, ...props }: Omit<PermissionLinkProps, 'role'>) {
    return (
        <PermissionLink 
            role={['organizer', 'admin', 'super_admin']}
            {...props}
        >
            {children}
        </PermissionLink>
    );
}