import { usePermissions } from '@/hooks/use-permissions';
import { ReactNode } from 'react';

interface PermissionGateProps {
    permission?: string | string[];
    role?: string | string[];
    requireAll?: boolean;
    fallback?: ReactNode;
    children: ReactNode;
}

/**
 * PermissionGate component - Shows/hides content based on user permissions or roles
 * 
 * @param permission - Permission(s) required to show content
 * @param role - Role(s) required to show content  
 * @param requireAll - If true, user must have ALL permissions/roles (default: false - ANY)
 * @param fallback - Content to show when user lacks permissions
 * @param children - Content to show when user has permissions
 */
export function PermissionGate({ 
    permission, 
    role, 
    requireAll = false, 
    fallback = null, 
    children 
}: PermissionGateProps) {
    const { hasPermission, hasRole, hasAllPermissions, hasAllRoles } = usePermissions();

    let hasAccess = false;

    // Check permissions
    if (permission) {
        const permissions = Array.isArray(permission) ? permission : [permission];
        hasAccess = requireAll 
            ? hasAllPermissions(permissions)
            : hasPermission(permission);
    }

    // Check roles (if no permission specified or as additional check)
    if (role && (!permission || hasAccess)) {
        const roles = Array.isArray(role) ? role : [role];
        const roleAccess = requireAll 
            ? hasAllRoles(roles)
            : hasRole(role);
        
        hasAccess = permission ? (hasAccess && roleAccess) : roleAccess;
    }

    // If neither permission nor role specified, deny access
    if (!permission && !role) {
        hasAccess = false;
    }

    return hasAccess ? <>{children}</> : <>{fallback}</>;
}

// Convenience components for common permission patterns
export function AdminOnly({ children, fallback = null }: { children: ReactNode; fallback?: ReactNode }) {
    return (
        <PermissionGate role={['admin', 'super_admin']} fallback={fallback}>
            {children}
        </PermissionGate>
    );
}

export function OrganizerOnly({ children, fallback = null }: { children: ReactNode; fallback?: ReactNode }) {
    return (
        <PermissionGate role={['organizer', 'admin', 'super_admin']} fallback={fallback}>
            {children}
        </PermissionGate>
    );
}

export function RefereeOnly({ children, fallback = null }: { children: ReactNode; fallback?: ReactNode }) {
    return (
        <PermissionGate role={['referee', 'admin', 'super_admin']} fallback={fallback}>
            {children}
        </PermissionGate>
    );
}

export function PlayerOnly({ children, fallback = null }: { children: ReactNode; fallback?: ReactNode }) {
    return (
        <PermissionGate role={['player', 'admin', 'super_admin']} fallback={fallback}>
            {children}
        </PermissionGate>
    );
}

export function CanView({ resource, children, fallback = null }: { resource: string; children: ReactNode; fallback?: ReactNode }) {
    return (
        <PermissionGate permission={[`${resource}.view`, `${resource}.manage`]} fallback={fallback}>
            {children}
        </PermissionGate>
    );
}

export function CanCreate({ resource, children, fallback = null }: { resource: string; children: ReactNode; fallback?: ReactNode }) {
    return (
        <PermissionGate permission={[`${resource}.create`, `${resource}.manage`]} fallback={fallback}>
            {children}
        </PermissionGate>
    );
}

export function CanEdit({ resource, children, fallback = null }: { resource: string; children: ReactNode; fallback?: ReactNode }) {
    return (
        <PermissionGate permission={[`${resource}.edit`, `${resource}.manage`]} fallback={fallback}>
            {children}
        </PermissionGate>
    );
}

export function CanDelete({ resource, children, fallback = null }: { resource: string; children: ReactNode; fallback?: ReactNode }) {
    return (
        <PermissionGate permission={[`${resource}.delete`, `${resource}.manage`]} fallback={fallback}>
            {children}
        </PermissionGate>
    );
}