import { usePermissions } from '@/hooks/use-permissions';
import { Button, ButtonProps } from '@/components/ui/button';
import { ReactNode } from 'react';
import { Tooltip, TooltipContent, TooltipProvider, TooltipTrigger } from '@/components/ui/tooltip';

interface PermissionButtonProps extends ButtonProps {
    permission?: string | string[];
    role?: string | string[];
    requireAll?: boolean;
    children: ReactNode;
    fallbackContent?: ReactNode;
    showTooltip?: boolean;
    tooltipMessage?: string;
    hideWhenNoAccess?: boolean;
}

/**
 * PermissionButton component - Button that shows/hides or disables based on permissions
 */
export function PermissionButton({
    permission,
    role,
    requireAll = false,
    children,
    fallbackContent = null,
    showTooltip = true,
    tooltipMessage,
    hideWhenNoAccess = false,
    disabled,
    ...buttonProps
}: PermissionButtonProps) {
    const { hasPermission, hasRole, hasAllPermissions, hasAllRoles } = usePermissions();

    let hasAccess = false;

    // Check permissions
    if (permission) {
        const permissions = Array.isArray(permission) ? permission : [permission];
        hasAccess = requireAll 
            ? hasAllPermissions(permissions)
            : hasPermission(permission);
    }

    // Check roles
    if (role && (!permission || hasAccess)) {
        const roles = Array.isArray(role) ? role : [role];
        const roleAccess = requireAll 
            ? hasAllRoles(roles)
            : hasRole(role);
        
        hasAccess = permission ? (hasAccess && roleAccess) : roleAccess;
    }

    // If neither permission nor role specified, grant access
    if (!permission && !role) {
        hasAccess = true;
    }

    // Hide button if no access and hideWhenNoAccess is true
    if (!hasAccess && hideWhenNoAccess) {
        return fallbackContent ? <>{fallbackContent}</> : null;
    }

    // Determine if button should be disabled
    const isDisabled = disabled || !hasAccess;

    // Default tooltip message
    const defaultTooltipMessage = tooltipMessage || 
        'You do not have permission to perform this action';

    const button = (
        <Button
            {...buttonProps}
            disabled={isDisabled}
        >
            {children}
        </Button>
    );

    // Show tooltip if disabled due to permissions and showTooltip is true
    if (!hasAccess && showTooltip) {
        return (
            <TooltipProvider>
                <Tooltip>
                    <TooltipTrigger asChild>
                        {button}
                    </TooltipTrigger>
                    <TooltipContent>
                        <p>{defaultTooltipMessage}</p>
                    </TooltipContent>
                </Tooltip>
            </TooltipProvider>
        );
    }

    return button;
}

// Convenience components for common button patterns
interface ResourceButtonProps extends Omit<PermissionButtonProps, 'permission'> {
    resource: string;
}

export function CreateButton({ resource, children, ...props }: ResourceButtonProps) {
    return (
        <PermissionButton 
            permission={[`${resource}.create`, `${resource}.manage`]}
            tooltipMessage={`You need permission to create ${resource}`}
            {...props}
        >
            {children}
        </PermissionButton>
    );
}

export function EditButton({ resource, children, ...props }: ResourceButtonProps) {
    return (
        <PermissionButton 
            permission={[`${resource}.edit`, `${resource}.manage`]}
            tooltipMessage={`You need permission to edit ${resource}`}
            {...props}
        >
            {children}
        </PermissionButton>
    );
}

export function DeleteButton({ resource, children, ...props }: ResourceButtonProps) {
    return (
        <PermissionButton 
            permission={[`${resource}.delete`, `${resource}.manage`]}
            tooltipMessage={`You need permission to delete ${resource}`}
            {...props}
        >
            {children}
        </PermissionButton>
    );
}

export function AdminButton({ children, ...props }: Omit<PermissionButtonProps, 'role'>) {
    return (
        <PermissionButton 
            role={['admin', 'super_admin']}
            tooltipMessage="This action is restricted to administrators"
            {...props}
        >
            {children}
        </PermissionButton>
    );
}

export function OrganizerButton({ children, ...props }: Omit<PermissionButtonProps, 'role'>) {
    return (
        <PermissionButton 
            role={['organizer', 'admin', 'super_admin']}
            tooltipMessage="This action is restricted to organizers and administrators"
            {...props}
        >
            {children}
        </PermissionButton>
    );
}