import { NavFooter } from '@/components/nav-footer';
import { NavMain } from '@/components/nav-main';
import { NavUser } from '@/components/nav-user';
import { Sidebar, SidebarContent, SidebarFooter, SidebarHeader, SidebarMenu, SidebarMenuButton, SidebarMenuItem } from '@/components/ui/sidebar';
import { type NavItem } from '@/types';
import { usePermissions } from '@/hooks/use-permissions';
import { RoleBasedHelp } from '@/components/help';
import { Link } from '@inertiajs/react';
import { useState } from 'react';
import { 
    Gamepad, 
    LayoutGrid, 
    ListOrdered, 
    Settings, 
    Trophy, 
    Users, 
    UsersRound, 
    UserCog, 
    User,
    Image, 
    Globe, 
    Newspaper,
    Camera,
    MessageSquareQuote,
    Building2,
    Monitor,
    HardDrive,
    BarChart3,
    CreditCard,
    Key,
    HelpCircle,
    Shield,
    Video,
    Target,
    Award
} from 'lucide-react';
import AppLogo from './app-logo';

export function AppSidebar() {
    const { hasPermission, isAdmin, isOrganizer, isReferee, isPlayer, isTeamManager } = usePermissions();
    const [showHelp, setShowHelp] = useState(false);

    // Determine primary role for context-aware navigation
    const getPrimaryRole = () => {
        if (isAdmin()) return 'admin';
        if (isOrganizer()) return 'organizer';
        if (isReferee()) return 'referee';
        if (isTeamManager()) return 'team_manager';
        if (isPlayer()) return 'player';
        return 'user';
    };

    const primaryRole = getPrimaryRole();

    // Single array with all navigation items including permissions and roles
    const allNavItems: (NavItem & { permission?: string | string[], roles?: string[] })[] = [
        {
            title: 'Dashboard',
            href: '/admin/dashboard',
            icon: LayoutGrid,
            // Dashboard accessible to all authenticated users
        },
        {
            title: 'Tournaments',
            href: '/admin/tournaments',
            icon: Trophy,
            permission: ['tournaments.view', 'tournaments.manage'],
            roles: ['admin', 'organizer', 'referee'],
        },
        {
            title: 'Players',
            href: '/admin/players',
            icon: Users,
            permission: ['players.view', 'players.manage'],
            roles: ['admin', 'organizer'],
        },
        {
            title: 'Teams',
            href: '/admin/teams',
            icon: UsersRound,
            permission: ['teams.view', 'teams.manage'],
            roles: ['admin', 'organizer'],
        },
        {
            title: 'Matches',
            href: '/admin/matches',
            icon: Gamepad,
            permission: ['matches.view', 'matches.manage'],
            roles: ['admin', 'organizer', 'referee'],
        },
        {
            title: 'Brackets',
            href: '/admin/brackets',
            icon: ListOrdered,
            permission: ['brackets.view', 'brackets.manage'],
            roles: ['admin', 'organizer', 'referee'],
        },
        {
            title: 'Users',
            href: '/admin/users',
            icon: UserCog,
            permission: ['users.view', 'users.manage'],
            roles: ['admin'],
        },
        {
            title: 'Permissions',
            href: '/admin/permissions',
            icon: Shield,
            permission: ['permissions.view', 'permissions.manage'],
            roles: ['admin'],
            children: [
                {
                    title: 'Permissions',
                    href: '/admin/permissions',
                    icon: Shield,
                },
                {
                    title: 'Roles',
                    href: '/admin/roles',
                    icon: Shield,
                },
            ],
        },  
        {
            title: 'Payments',
            href: '/admin/payments',
            icon: CreditCard,
            permission: ['payments.view', 'payments.manage'],
            roles: ['admin'],
        },
        {
            title: 'Tournament Settings',
            href: '/admin/tournament-settings',
            icon: Settings,
            permission: ['settings.manage'],
            roles: ['admin'],
        },
        {
            title: 'Media',
            href: '/admin/media',
            icon: Image,
            permission: ['media.view', 'media.manage'],
            roles: ['admin', 'organizer', 'referee'],
        },
        {
            title: 'Frontend Content',
            href: '/admin/frontend',
            icon: Globe,
            permission: ['media.manage'],
            roles: ['admin', 'organizer'],
            children: [
                {
                    title: 'Hero Slides',
                    href: '/admin/frontend/hero-slides',
                    icon: Camera,
                },
                {
                    title: 'News Articles',
                    href: '/admin/frontend/news-articles',
                    icon: Newspaper,
                },
                {
                    title: 'Gallery',
                    href: '/admin/frontend/gallery',
                    icon: Image,
                },
                {
                    title: 'Vlog Videos',
                    href: '/admin/frontend/vlog',
                    icon: Video,
                    permission: ['vlog.view', 'vlog.create', 'vlog.edit', 'vlog.delete', 'vlog.manage']
                },
                {
                    title: 'Sponsors',
                    href: '/admin/frontend/sponsors',
                    icon: Building2,
                },
                {
                    title: 'Testimonials',
                    href: '/admin/frontend/testimonials',
                    icon: MessageSquareQuote,
                },
                {
                    title: 'Tournament Highlights',
                    href: '/admin/frontend/tournament-highlights',
                    icon: Trophy,
                },
                {
                    title: 'FAQs',
                    href: '/admin/faqs',
                    icon: HelpCircle,
                },
                {
                    title: 'Prize Awards',
                    href: '/admin/prize-awards',
                    icon: Award,
                },
            ],
        },
        // {
        //     title: 'Queue Monitor',
        //     href: '/admin/queue-monitor',
        //     icon: Monitor,
        //     permission: ['queue.monitor'],
        //     roles: ['admin'],
        // },
        // Team Manager-specific navigation
        {
            title: 'Team Profile',
            href: '/team/profile',
            icon: UsersRound,
            roles: ['team_manager'],
        },
        {
            title: 'Players',
            href: '/team/players',
            icon: Users,
            roles: ['team_manager'],
        },
        {
            title: 'Formations',
            href: '/team/formations',
            icon: Target,
            roles: ['team_manager'],
        },
        {
            title: 'Tournaments',
            href: '/team/tournaments',
            icon: Trophy,
            roles: ['team_manager'],
        },
        {
            title: 'Matches',
            href: '/team/matches',
            icon: Gamepad,
            roles: ['team_manager'],
        },
        {
            title: 'Payments',
            href: '/team/payments',
            icon: CreditCard,
            roles: ['team_manager'],
        },
        {
            title: 'Analytics',
            href: '/team/analytics',
            icon: BarChart3,
            roles: ['team_manager'],
        },
        // Player-specific navigation
        {
            title: 'My Tournaments',
            href: '/player/tournaments',
            icon: Trophy,
            roles: ['player'],
        },
        {
            title: 'My Team',
            href: '/player/team',
            icon: UsersRound,
            roles: ['player'],
        },
        {
            title: 'My Matches',
            href: '/player/matches',
            icon: Gamepad,
            roles: ['player'],
        },
        {
            title: 'My Profile',
            href: '/player/profile',
            icon: User,
            roles: ['player'],
        },
        {
            title: 'Performance Analytics',
            href: '/player/performance',
            icon: BarChart3,
            roles: ['player'],
        },

    ];

    // Filter navigation items based on permissions and roles
    const filterNavItems = (items: (NavItem & { permission?: string | string[], roles?: string[] })[]): NavItem[] => {
        return items.filter(item => {
            // Check role access first
            if (item.roles && item.roles.length > 0) {
                const hasRole = item.roles.includes(primaryRole);
                if (!hasRole) {
                    return false;
                }
            }

            // Check permission access
            if (item.permission) {
                const hasAccess = hasPermission(item.permission);
                if (!hasAccess) {
                    return false;
                }
            }

            // If no permission or role requirement, show to all authenticated users
            if (!item.permission && !item.roles) {
            }

            return true;
        }).map(item => {
            // Remove permission and roles properties and return clean NavItem
            const { permission, roles, ...navItem } = item;
            
            // Filter children if they exist
            if (navItem.children) {
                const filteredChildren = filterNavItems(navItem.children);
                // Only include the parent if it has visible children or if it's not just a container
                if (filteredChildren.length > 0) {
                    navItem.children = filteredChildren;
                } else {
                    // If no children are visible, still show parent if it has its own route
                    navItem.children = [];
                }
            }
            
            return navItem;
        });
    };

    const finalNavItems = filterNavItems(allNavItems);

    // Footer navigation items (analytics for admins only)
    const footerNavItems: NavItem[] = [];
    
    if (isAdmin() || isOrganizer()) {
        // footerNavItems.push({
        //     title: 'Analytics',
        //     href: '/admin/analytics',
        //     icon: BarChart3,
        // });
    }

    return (
        <>
            <Sidebar collapsible="icon" variant="inset">
                <SidebarHeader className="flex justify-center items-center">
                    <SidebarMenu className="flex justify-center items-center">
                        <SidebarMenuItem>
                            <SidebarMenuButton size="lg" asChild>
                                <Link href="/admin/dashboard" prefetch>
                                    <AppLogo  />
                                </Link>
                            </SidebarMenuButton>
                        </SidebarMenuItem>
                    </SidebarMenu>
                </SidebarHeader>

                <SidebarContent>
                    <NavMain items={finalNavItems} />
                </SidebarContent>

                <SidebarFooter>
                    <NavFooter
                        items={footerNavItems}
                        className="border-t border-sidebar-border/50"
                    />
                    <NavUser />
                    {/* <div className="p-4 border-t">
                        <button
                            onClick={() => setShowHelp(true)}
                            className="flex items-center gap-2 w-full px-3 py-2 text-sm text-muted-foreground hover:text-foreground hover:bg-accent rounded-md transition-colors"
                        >
                            <HelpCircle className="h-4 w-4" />
                            Help
                        </button>
                    </div> */}
                </SidebarFooter>
            </Sidebar>

            {showHelp && (
                <RoleBasedHelp onClose={() => setShowHelp(false)} />
            )}
        </>
    );
}
