import React, { useState, useEffect, useRef } from 'react';
import { router } from '@inertiajs/react';
import {
  Dialog,
  DialogContent,
  DialogDescription,
  DialogHeader,
  DialogTitle,
  DialogTrigger,
} from '@/components/ui/dialog';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { Checkbox } from '@/components/ui/checkbox';
import { Badge } from '@/components/ui/badge';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card';
import { Tabs, TabsContent, TabsList, TabsTrigger } from '@/components/ui/tabs';
import { 
  UserPlus, 
  UserMinus, 
  Search, 
  Users, 
  Mail, 
  Send, 
  Inbox,
  Calendar,
  Settings,
  X,
  CheckCircle,
  AlertCircle,
  ChevronDown,
  Loader2
} from 'lucide-react';

interface User {
  id: number;
  name: string;
  email: string;
  roles?: Array<{
    name: string;
    display_name: string;
  }>;
}

interface AssignedUser extends User {
  pivot: {
    can_send: boolean;
    can_receive: boolean;
    permissions: any;
    granted_at: string;
    expires_at?: string;
  };
}

interface Props {
  emailCredentialId: number;
  assignedUsers: AssignedUser[];
  onUserAssigned?: () => void;
  onUserRemoved?: () => void;
  onPermissionsUpdated?: () => void;
}

// Select2-like User Selection Component
const UserSelect: React.FC<{
  users: User[];
  selectedUsers: number[];
  onUserSelect: (userId: number, checked: boolean) => void;
  onSearch: (searchTerm: string) => void;
  isLoading: boolean;
  searchTerm: string;
  setSearchTerm: (term: string) => void;
}> = ({ users, selectedUsers, onUserSelect, onSearch, isLoading, searchTerm, setSearchTerm }) => {
  const [isOpen, setIsOpen] = useState(false);
  const [searchInput, setSearchInput] = useState('');
  const searchTimeoutRef = useRef<NodeJS.Timeout | null>(null);
  const dropdownRef = useRef<HTMLDivElement>(null);

  const handleSearchChange = (value: string) => {
    setSearchInput(value);
    
    // Debounce search
    if (searchTimeoutRef.current) {
      clearTimeout(searchTimeoutRef.current);
    }
    
    searchTimeoutRef.current = setTimeout(() => {
      onSearch(value);
    }, 300);
  };

  // Close dropdown when clicking outside
  useEffect(() => {
    const handleClickOutside = (event: MouseEvent) => {
      if (dropdownRef.current && !dropdownRef.current.contains(event.target as Node)) {
        setIsOpen(false);
      }
    };

    if (isOpen) {
      document.addEventListener('mousedown', handleClickOutside);
    }

    return () => {
      document.removeEventListener('mousedown', handleClickOutside);
    };
  }, [isOpen]);

  const selectedUsersData = users.filter(user => selectedUsers.includes(user.id));

  return (
    <div className="space-y-2">
      <Label>Select Users</Label>
      <div className="relative" ref={dropdownRef}>
        <div 
          className="min-h-[40px] border rounded-md p-2 cursor-pointer flex items-center justify-between hover:border-gray-400 transition-colors"
          onClick={() => setIsOpen(!isOpen)}
        >
          <div className="flex flex-wrap gap-1 flex-1">
            {selectedUsersData.length === 0 ? (
              <span className="text-muted-foreground">Search and select users...</span>
            ) : (
              selectedUsersData.map(user => (
                <Badge key={user.id} variant="secondary" className="flex items-center gap-1">
                  {user.name}
                  <X 
                    className="h-3 w-3 cursor-pointer hover:text-red-500" 
                    onClick={(e) => {
                      e.stopPropagation();
                      onUserSelect(user.id, false);
                    }}
                  />
                </Badge>
              ))
            )}
          </div>
          <ChevronDown className={`h-4 w-4 text-muted-foreground transition-transform ${isOpen ? 'rotate-180' : ''}`} />
        </div>

        {isOpen && (
          <div className="absolute z-50 w-full mt-1 bg-white border rounded-md shadow-lg max-h-60 overflow-hidden">
            <div className="p-2 border-b">
              <div className="relative">
                <Search className="absolute left-3 top-1/2 transform -translate-y-1/2 text-gray-400 h-4 w-4" />
                <Input
                  placeholder="Search users..."
                  value={searchInput}
                  onChange={(e) => handleSearchChange(e.target.value)}
                  className="pl-10"
                  autoFocus
                />
              </div>
            </div>
            
            <div className="max-h-48 overflow-y-auto">
              {isLoading ? (
                <div className="flex items-center justify-center py-4">
                  <Loader2 className="h-4 w-4 animate-spin mr-2" />
                  <span className="text-sm text-muted-foreground">Searching...</span>
                </div>
              ) : users.length === 0 ? (
                <div className="p-4 text-center text-muted-foreground">
                  <Users className="h-8 w-8 mx-auto mb-2 opacity-50" />
                  <p className="text-sm">No users found</p>
                </div>
              ) : (
                users.map(user => (
                  <div 
                    key={user.id} 
                    className="flex items-center p-2 hover:bg-gray-50 cursor-pointer"
                    onClick={() => onUserSelect(user.id, !selectedUsers.includes(user.id))}
                  >
                    <Checkbox
                      checked={selectedUsers.includes(user.id)}
                      onChange={() => {}} // Handled by parent div click
                      className="mr-3"
                    />
                    <div className="flex-1">
                      <div className="font-medium text-sm">{user.name}</div>
                      <div className="text-xs text-muted-foreground">{user.email}</div>
                      {user.roles && user.roles.length > 0 && (
                        <div className="flex gap-1 mt-1">
                          {user.roles.map(role => (
                            <Badge key={role.name} variant="outline" className="text-xs">
                              {role.display_name}
                            </Badge>
                          ))}
                        </div>
                      )}
                    </div>
                  </div>
                ))
              )}
            </div>
          </div>
        )}
      </div>
    </div>
  );
};

const UserAccessDialog: React.FC<Props> = ({
  emailCredentialId,
  assignedUsers,
  onUserAssigned,
  onUserRemoved,
  onPermissionsUpdated,
}) => {
  const [isOpen, setIsOpen] = useState(false);
  const [activeTab, setActiveTab] = useState('assign');
  const [searchTerm, setSearchTerm] = useState('');
  const [availableUsers, setAvailableUsers] = useState<User[]>([]);
  const [selectedUsers, setSelectedUsers] = useState<number[]>([]);
  const [permissions, setPermissions] = useState<Record<number, any>>({});
  const [isLoading, setIsLoading] = useState(false);
  const [isAssigning, setIsAssigning] = useState(false);

  // Load available users when dialog opens
  useEffect(() => {
    if (isOpen && activeTab === 'assign') {
      loadAvailableUsers('');
    }
  }, [isOpen, activeTab]);

  const loadAvailableUsers = async (search: string = '') => {
    setIsLoading(true);
    try {
      const url = route('admin.settings.emails.available-users', emailCredentialId) + 
        (search ? `?search=${encodeURIComponent(search)}` : '');
      const response = await fetch(url);
      const users = await response.json();
      setAvailableUsers(users);
    } catch (error) {
      console.error('Error loading available users:', error);
    } finally {
      setIsLoading(false);
    }
  };

  const handleSearch = (e: React.FormEvent) => {
    e.preventDefault();
    loadAvailableUsers(searchTerm);
  };

  const handleUserSelect = (userId: number, checked: boolean) => {
    if (checked) {
      setSelectedUsers([...selectedUsers, userId]);
      setPermissions(prev => ({
        ...prev,
        [userId]: {
          can_send: true,
          can_receive: true,
          permissions: null,
        }
      }));
    } else {
      setSelectedUsers(selectedUsers.filter(id => id !== userId));
      setPermissions(prev => {
        const newPermissions = { ...prev };
        delete newPermissions[userId];
        return newPermissions;
      });
    }
  };

  const handlePermissionChange = (userId: number, permission: string, value: boolean) => {
    setPermissions(prev => ({
      ...prev,
      [userId]: {
        ...prev[userId],
        [permission]: value,
      }
    }));
  };

  const handleAssignUsers = async () => {
    if (selectedUsers.length === 0) return;

    setIsAssigning(true);
    try {
      router.post(route('admin.settings.emails.assign-users', emailCredentialId), {
        user_ids: selectedUsers,
        permissions: permissions,
      }, {
        onSuccess: () => {
          setSelectedUsers([]);
          setPermissions({});
          onUserAssigned?.();
          setIsOpen(false);
        },
        onError: (errors) => {
          console.error('Error assigning users:', errors);
        },
        onFinish: () => {
          setIsAssigning(false);
        }
      });
    } catch (error) {
      console.error('Error assigning users:', error);
      setIsAssigning(false);
    }
  };

  const handleRemoveUser = async (userId: number) => {
    if (confirm('Are you sure you want to remove this user from the email account?')) {
      try {
        router.post(route('admin.settings.emails.remove-users', emailCredentialId), {
          user_ids: [userId],
        }, {
          onSuccess: () => {
            onUserRemoved?.();
          }
        });
      } catch (error) {
        console.error('Error removing user:', error);
      }
    }
  };

  const handleUpdatePermissions = async (userId: number, newPermissions: any) => {
    try {
      router.post(route('admin.settings.emails.update-user-permissions', emailCredentialId), {
        user_id: userId,
        permissions: newPermissions,
      }, {
        onSuccess: () => {
          onPermissionsUpdated?.();
        }
      });
    } catch (error) {
      console.error('Error updating permissions:', error);
    }
  };

  const formatDate = (dateString: string) => {
    return new Date(dateString).toLocaleDateString('en-US', {
      year: 'numeric',
      month: 'short',
      day: 'numeric',
      hour: '2-digit',
      minute: '2-digit',
    });
  };

  const isExpired = (expiresAt: string | null) => {
    if (!expiresAt) return false;
    return new Date(expiresAt) < new Date();
  };

  return (
    <Dialog open={isOpen} onOpenChange={setIsOpen}>
      <DialogTrigger asChild>
        <Button variant="outline" size="sm">
          <UserPlus className="h-4 w-4 mr-2" />
          Manage Users
        </Button>
      </DialogTrigger>
      <DialogContent className="max-w-4xl max-h-[80vh] overflow-hidden">
        <DialogHeader>
          <DialogTitle className="flex items-center gap-2">
            <Users className="h-5 w-5" />
            Manage User Access
          </DialogTitle>
          <DialogDescription>
            Grant or revoke access to this email account for specific users.
          </DialogDescription>
        </DialogHeader>

        <Tabs value={activeTab} onValueChange={setActiveTab} className="w-full">
          <TabsList className="grid w-full grid-cols-2">
            <TabsTrigger value="assign">Assign Users</TabsTrigger>
            <TabsTrigger value="manage">Manage Access</TabsTrigger>
          </TabsList>

          {/* Assign Users Tab */}
          <TabsContent value="assign" className="space-y-4">
            <Card>
              <CardHeader>
                <CardTitle className="text-lg">Assign New Users</CardTitle>
                <CardDescription>
                  Select users to grant access to this email account
                </CardDescription>
              </CardHeader>
              <CardContent className="space-y-4">
                <UserSelect
                  users={availableUsers}
                  selectedUsers={selectedUsers}
                  onUserSelect={handleUserSelect}
                  onSearch={loadAvailableUsers}
                  isLoading={isLoading}
                  searchTerm={searchTerm}
                  setSearchTerm={setSearchTerm}
                />

                {selectedUsers.length > 0 && (
                  <Card>
                    <CardHeader>
                      <CardTitle className="text-sm">Set Permissions</CardTitle>
                    </CardHeader>
                    <CardContent className="space-y-3">
                      {selectedUsers.map((userId) => {
                        const user = availableUsers.find(u => u.id === userId);
                        if (!user) return null;

                        return (
                          <div key={userId} className="space-y-2">
                            <div className="font-medium text-sm">{user.name}</div>
                            <div className="flex gap-4">
                              <div className="flex items-center space-x-2">
                                <Checkbox
                                  id={`send-${userId}`}
                                  checked={permissions[userId]?.can_send || false}
                                  onCheckedChange={(checked) => 
                                    handlePermissionChange(userId, 'can_send', !!checked)
                                  }
                                />
                                <Label htmlFor={`send-${userId}`} className="text-sm flex items-center gap-1">
                                  <Send className="h-3 w-3" />
                                  Send Emails
                                </Label>
                              </div>
                              <div className="flex items-center space-x-2">
                                <Checkbox
                                  id={`receive-${userId}`}
                                  checked={permissions[userId]?.can_receive || false}
                                  onCheckedChange={(checked) => 
                                    handlePermissionChange(userId, 'can_receive', !!checked)
                                  }
                                />
                                <Label htmlFor={`receive-${userId}`} className="text-sm flex items-center gap-1">
                                  <Inbox className="h-3 w-3" />
                                  Receive Emails
                                </Label>
                              </div>
                            </div>
                          </div>
                        );
                      })}
                    </CardContent>
                  </Card>
                )}

                <div className="flex justify-end gap-2">
                  <Button variant="outline" onClick={() => setIsOpen(false)}>
                    Cancel
                  </Button>
                  <Button 
                    onClick={handleAssignUsers}
                    disabled={selectedUsers.length === 0 || isAssigning}
                  >
                    {isAssigning ? (
                      <>
                        <div className="mr-2 h-4 w-4 animate-spin rounded-full border-2 border-white border-t-transparent"></div>
                        Assigning...
                      </>
                    ) : (
                      <>
                        <UserPlus className="h-4 w-4 mr-2" />
                        Assign {selectedUsers.length} User(s)
                      </>
                    )}
                  </Button>
                </div>
              </CardContent>
            </Card>
          </TabsContent>

          {/* Manage Access Tab */}
          <TabsContent value="manage" className="space-y-4">
            <Card>
              <CardHeader>
                <CardTitle className="text-lg">Current Users</CardTitle>
                <CardDescription>
                  Manage permissions for users with access to this email account
                </CardDescription>
              </CardHeader>
              <CardContent>
                {assignedUsers.length === 0 ? (
                  <div className="text-center py-8 text-muted-foreground">
                    <Users className="h-12 w-12 mx-auto mb-4 opacity-50" />
                    <p>No users assigned to this email account</p>
                  </div>
                ) : (
                  <div className="space-y-3">
                    {assignedUsers.map((user) => (
                      <div key={user.id} className="flex items-center justify-between p-4 border rounded-lg">
                        <div className="flex items-center gap-3">
                          <div>
                            <div className="font-medium">{user.name}</div>
                            <div className="text-sm text-muted-foreground">{user.email}</div>
                            <div className="flex gap-2 mt-1">
                              {user.pivot.can_send && (
                                <Badge variant="outline" className="text-xs">
                                  <Send className="h-3 w-3 mr-1" />
                                  Send
                                </Badge>
                              )}
                              {user.pivot.can_receive && (
                                <Badge variant="outline" className="text-xs">
                                  <Inbox className="h-3 w-3 mr-1" />
                                  Receive
                                </Badge>
                              )}
                            </div>
                            <div className="text-xs text-muted-foreground mt-1">
                              Granted: {formatDate(user.pivot.granted_at)}
                              {user.pivot.expires_at && (
                                <span className={`ml-2 ${isExpired(user.pivot.expires_at) ? 'text-red-600' : ''}`}>
                                  Expires: {formatDate(user.pivot.expires_at)}
                                  {isExpired(user.pivot.expires_at) && (
                                    <Badge variant="destructive" className="ml-2 text-xs">Expired</Badge>
                                  )}
                                </span>
                              )}
                            </div>
                          </div>
                        </div>
                        <div className="flex gap-2">
                          <Button
                            variant="outline"
                            size="sm"
                            onClick={() => handleRemoveUser(user.id)}
                            className="text-red-600 hover:text-red-700"
                          >
                            <UserMinus className="h-4 w-4" />
                          </Button>
                        </div>
                      </div>
                    ))}
                  </div>
                )}
              </CardContent>
            </Card>
          </TabsContent>
        </Tabs>
      </DialogContent>
    </Dialog>
  );
};

export default UserAccessDialog;
