import React from 'react';
import { Link, router } from '@inertiajs/react';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';
import { UsersRound } from 'lucide-react';

interface Team {
  id: number;
  name: string;
  description?: string;
  logo?: string;
}

interface TeamSelectorProps {
  teams: Team[];
  selectedTeamId: number;
  currentPath: string;
}

export default function TeamSelector({ teams, selectedTeamId, currentPath }: TeamSelectorProps) {
  const handleTeamChange = (teamId: string) => {
    const url = new URL(window.location.href);
    url.searchParams.set('team_id', teamId);
    router.visit(url.toString(), {
      preserveState: true,
      preserveScroll: true,
    });
  };

  if (teams.length <= 1) {
    return null; // Don't show selector if only one team
  }

  return (
    <div className="mb-6">
      <div className="flex items-center gap-3">
        <UsersRound className="w-5 h-5 text-gray-600" />
        <div className="flex-1">
          <label className="text-sm font-medium text-gray-700 mb-2 block">
            Select Team
          </label>
          <Select value={selectedTeamId.toString()} onValueChange={handleTeamChange}>
            <SelectTrigger className="w-full max-w-md">
              <SelectValue placeholder="Choose a team" />
            </SelectTrigger>
            <SelectContent>
              {teams.map((team) => (
                <SelectItem key={team.id} value={team.id.toString()}>
                  <div className="flex items-center gap-2">
                    {team.logo && (
                      <img
                        src={`/storage/${team.logo}`}
                        alt={team.name}
                        className="w-6 h-6 rounded-full object-cover"
                      />
                    )}
                    <div>
                      <div className="font-medium">{team.name}</div>
                      {team.description && (
                        <div className="text-xs text-gray-500 truncate max-w-48">
                          {team.description}
                        </div>
                      )}
                    </div>
                  </div>
                </SelectItem>
              ))}
            </SelectContent>
          </Select>
        </div>
      </div>
    </div>
  );
}
