import React, { useState, useEffect } from 'react';
import { Alert, AlertDescription, AlertTitle } from '@/components/ui/alert';
import { Badge } from '@/components/ui/badge';
import { Button } from '@/components/ui/button';
import { Card, CardContent } from '@/components/ui/card';
import { AlertTriangle, CreditCard, Users, XCircle, ChevronRight, CheckCircle, X, Minimize2, Maximize2, Bell } from 'lucide-react';
import { Link } from '@inertiajs/react';
import { CategoryBadge } from '@/components/ui/category-badge';

interface PendingPayment {
  payment_id: number;
  tournament_id: number;
  tournament_name: string;
  amount: number;
  currency: string;
  status: string;
  reference_number: string;
  created_at: string;
  payment_method: string;
}

interface TeamWithIssues {
  team_id: number;
  team_name: string;
  category?: 'men' | 'women' | 'youth_male' | 'youth_female' | 'children';
  has_pending_payment: boolean;
  has_unbilled_registrations: boolean;
  has_incomplete_players: boolean;
  pending_payments: PendingPayment[];
  unbilled_tournaments: Array<{
    id: number;
    name: string;
    amount: number;
    currency: string;
  }>;
  incomplete_players_count: number;
  total_players: number;
  completeness_percentage: number;
  minimum_players_met: boolean;
  minimum_players_required?: number;
}

interface TeamRegistrationStatus {
  has_warnings: boolean;
  teams_with_issues: TeamWithIssues[];
}

interface TeamRegistrationAlertProps {
  teamRegistrationStatus: TeamRegistrationStatus | null;
}

export function TeamRegistrationAlert({ teamRegistrationStatus }: TeamRegistrationAlertProps) {
  const [isMinimized, setIsMinimized] = useState(true); // Minimized by default
  const [isDismissed, setIsDismissed] = useState(false);

  // Load dismissed state from localStorage on mount
  useEffect(() => {
    const dismissed = localStorage.getItem('teamRegistrationAlertDismissed');
    if (dismissed === 'true') {
      setIsDismissed(true);
    }
  }, []);

  // Save dismissed state to localStorage
  const handleDismiss = () => {
    setIsDismissed(true);
    localStorage.setItem('teamRegistrationAlertDismissed', 'true');
  };

  // Clear dismissed state and show alert
  const handleShow = () => {
    setIsDismissed(false);
    setIsMinimized(false);
    localStorage.removeItem('teamRegistrationAlertDismissed');
  };

  if (!teamRegistrationStatus || !teamRegistrationStatus.has_warnings) {
    return null;
  }

  const { teams_with_issues } = teamRegistrationStatus;

  // Helper function to calculate totals by currency
  const calculatePaymentTotals = (payments: PendingPayment[]) => {
    const totals: Record<string, number> = {};
    payments.forEach(payment => {
      const currency = payment.currency || 'KES';
      totals[currency] = (totals[currency] || 0) + Number(payment.amount);
    });
    return totals;
  };

  // Helper function to format currency totals
  const formatCurrencyTotals = (payments: PendingPayment[]) => {
    const totals = calculatePaymentTotals(payments);
    return Object.entries(totals)
      .map(([currency, amount]) => `${currency} ${amount.toLocaleString()}`)
      .join(' + ');
  };

  // Calculate total issues across all teams
  const totalPendingPayments = teams_with_issues.reduce(
    (sum, team) => sum + team.pending_payments.length, 
    0
  );
  const totalIncompletePlayers = teams_with_issues.reduce(
    (sum, team) => sum + team.incomplete_players_count, 
    0
  );

  // If dismissed, show floating button
  if (isDismissed) {
    return (
      <div className="fixed bottom-6 right-6 z-50 animate-in slide-in-from-bottom-5">
        <Button
          onClick={handleShow}
          className="bg-red-600 hover:bg-red-700 text-white shadow-lg hover:shadow-xl transition-all duration-300 px-6 py-6 rounded-full group relative"
          size="lg"
        >
          <div className="flex items-center gap-3">
            <div className="relative">
              <Bell className="w-6 h-6 animate-pulse" />
              <span className="absolute -top-2 -right-2 bg-yellow-400 text-red-900 text-xs font-bold rounded-full w-5 h-5 flex items-center justify-center">
                {teams_with_issues.length}
              </span>
            </div>
            <div className="flex flex-col items-start">
              <span className="font-bold text-base">Action Required</span>
              <span className="text-xs opacity-90">
                {totalPendingPayments > 0 && `${totalPendingPayments} payment${totalPendingPayments !== 1 ? 's' : ''}`}
                {totalPendingPayments > 0 && totalIncompletePlayers > 0 && ', '}
                {totalIncompletePlayers > 0 && `${totalIncompletePlayers} player${totalIncompletePlayers !== 1 ? 's' : ''}`}
              </span>
            </div>
          </div>
        </Button>
      </div>
    );
  }

  return (
    <div className="space-y-6 mb-6">
      {/* Minimize/Dismiss Controls */}
      <div className="flex justify-end gap-2 mb-2">
        <Button
          variant="outline"
          size="sm"
          onClick={() => setIsMinimized(!isMinimized)}
          className="text-gray-600 hover:text-gray-900"
        >
          {isMinimized ? (
            <>
              <Maximize2 className="w-4 h-4 mr-2" />
              Expand Alerts
            </>
          ) : (
            <>
              <Minimize2 className="w-4 h-4 mr-2" />
              Minimize
            </>
          )}
        </Button>
        <Button
          variant="outline"
          size="sm"
          onClick={handleDismiss}
          className="text-gray-600 hover:text-red-600"
        >
          <X className="w-4 h-4 mr-2" />
          Dismiss
        </Button>
      </div>

      {/* Minimized View */}
      {isMinimized ? (
        <Card className="border-red-300 bg-red-100 cursor-pointer hover:bg-red-200 transition-colors" onClick={() => setIsMinimized(false)}>
          <CardContent className="p-4">
            <div className="flex items-center justify-between">
              <div className="flex items-center gap-3">
                <AlertTriangle className="w-6 h-6 text-red-600 animate-pulse" />
                <div>
                  <p className="font-bold text-red-900">
                    {teams_with_issues.length} Team{teams_with_issues.length !== 1 ? 's' : ''} Require{teams_with_issues.length === 1 ? 's' : ''} Action
                  </p>
                  <p className="text-sm text-red-800">
                    {totalPendingPayments > 0 && `${totalPendingPayments} pending payment${totalPendingPayments !== 1 ? 's' : ''}`}
                    {totalPendingPayments > 0 && totalIncompletePlayers > 0 && ' • '}
                    {totalIncompletePlayers > 0 && `${totalIncompletePlayers} incomplete player profile${totalIncompletePlayers !== 1 ? 's' : ''}`}
                  </p>
                </div>
              </div>
              <Button variant="ghost" size="sm" className="text-red-600 hover:text-red-700">
                <Maximize2 className="w-4 h-4 mr-2" />
                Click to expand
              </Button>
            </div>
          </CardContent>
        </Card>
      ) : (
        <>
          {/* Summary Header for Multiple Teams */}
          {teams_with_issues.length > 1 && (
            <Card className="border-amber-300 bg-amber-50">
              <CardContent className="p-4">
                <div className="flex items-center gap-3">
                  <AlertTriangle className="w-6 h-6 text-amber-600" />
                  <div>
                    <p className="font-bold text-amber-900">
                      You manage {teams_with_issues.length} teams with pending requirements
                    </p>
                    <p className="text-sm text-amber-800">
                      Each team must complete its requirements individually. Scroll down to see details for each team.
                    </p>
                  </div>
                </div>
              </CardContent>
            </Card>
          )}

          {teams_with_issues.map((teamIssue, index) => (
            <Card key={teamIssue.team_id} className="border-red-200 bg-red-50 shadow-lg">
              <CardContent className="p-6">
            {/* Header */}
            <div className="flex items-start gap-4 mb-4">
              <div className="p-3 bg-red-100 rounded-full flex-shrink-0">
                <AlertTriangle className="w-8 h-8 text-red-600" />
              </div>
              <div className="flex-1">
                <div className="flex items-center gap-2 mb-1">
                  <h3 className="text-xl font-bold text-red-900">
                    ⚠️ ACTION REQUIRED: {teamIssue.team_name}
                  </h3>
                  {teamIssue.category && (
                    <CategoryBadge category={teamIssue.category} size="sm" />
                  )}
                </div>
                <p className="text-red-800 font-medium">
                  This team cannot be reviewed for tournament participation until the following issues are resolved:
                </p>
                <p className="text-red-700 text-sm mt-1 italic">
                  Each team's registration is reviewed individually. Please complete all requirements for this team.
                </p>
              </div>
            </div>

            {/* Issues List */}
            <div className="space-y-4 ml-16">
              {/* Payment Issues */}
              {teamIssue.has_pending_payment && (
                <Alert className="border-red-300 bg-white">
                  <CreditCard className="h-5 w-5 text-red-600" />
                  <AlertTitle className="text-red-900 font-semibold mb-2">
                    💳 Pending Tournament Registration Fees
                  </AlertTitle>
                  <AlertDescription>
                    <div className="space-y-3">
                      <p className="text-red-800 font-medium">
                        You have {teamIssue.pending_payments.length} unpaid tournament registration {teamIssue.pending_payments.length === 1 ? 'fee' : 'fees'}. 
                        Payment must be completed before your team can be reviewed.
                      </p>
                      
                      <div className="space-y-3">
                        {teamIssue.pending_payments.map((payment) => (
                          <div key={payment.payment_id} className="border-2 border-red-200 rounded-lg overflow-hidden bg-gradient-to-r from-red-50 to-white">
                            <div className="p-4">
                              <div className="flex items-start justify-between mb-3">
                                <div className="flex-1">
                                  <h4 className="font-bold text-red-900 text-lg mb-1">{payment.tournament_name}</h4>
                                  <p className="text-sm text-red-700">Tournament Registration Fee</p>
                                </div>
                                <Badge 
                                  className={`text-sm font-bold px-3 py-1 ${
                                    payment.status === 'pending' ? 'bg-yellow-500 text-white hover:bg-yellow-600' :
                                    payment.status === 'failed' ? 'bg-red-600 text-white hover:bg-red-700' :
                                    'bg-blue-500 text-white hover:bg-blue-600'
                                  }`}
                                >
                                  {payment.status.toUpperCase()}
                                </Badge>
                              </div>
                              
                              <div className="grid grid-cols-2 md:grid-cols-4 gap-3 mb-3">
                                <div className="bg-white p-2 rounded border border-red-100">
                                  <p className="text-xs text-gray-600 mb-1">Amount Due</p>
                                  <p className="font-bold text-red-900 text-lg">
                                    {payment.currency || 'KES'} {Number(payment.amount).toLocaleString()}
                                  </p>
                                </div>
                                <div className="bg-white p-2 rounded border border-red-100">
                                  <p className="text-xs text-gray-600 mb-1">Payment Method</p>
                                  <p className="font-semibold text-gray-900 text-sm">{payment.payment_method}</p>
                                </div>
                                <div className="bg-white p-2 rounded border border-red-100">
                                  <p className="text-xs text-gray-600 mb-1">Reference</p>
                                  <p className="font-mono text-xs text-gray-900">{payment.reference_number}</p>
                                </div>
                                <div className="bg-white p-2 rounded border border-red-100">
                                  <p className="text-xs text-gray-600 mb-1">Billed On</p>
                                  <p className="text-xs text-gray-900">
                                    {new Date(payment.created_at).toLocaleDateString()}
                                  </p>
                                </div>
                              </div>

                              {payment.status === 'pending' && (
                                <div className="flex items-center gap-2 p-2 bg-yellow-50 rounded border border-yellow-200">
                                  <AlertTriangle className="w-4 h-4 text-yellow-600" />
                                  <p className="text-sm text-yellow-800 font-medium">
                                    Awaiting payment - Please complete this transaction
                                  </p>
                                </div>
                              )}
                              
                              {payment.status === 'failed' && (
                                <div className="flex items-center gap-2 p-2 bg-red-100 rounded border border-red-200">
                                  <XCircle className="w-4 h-4 text-red-600" />
                                  <p className="text-sm text-red-800 font-medium">
                                    Payment failed - Please retry the payment
                                  </p>
                                </div>
                              )}
                            </div>
                          </div>
                        ))}
                      </div>

                      <div className="bg-blue-50 border border-blue-200 rounded-lg p-4">
                        <p className="font-medium text-blue-900 mb-2">📌 Payment Instructions:</p>
                        <ul className="text-sm text-blue-800 space-y-1">
                          <li className="flex items-start gap-2">
                            <span className="text-blue-600 font-bold">1.</span>
                            <span>Click the "Complete Payment Now" button below</span>
                          </li>
                          <li className="flex items-start gap-2">
                            <span className="text-blue-600 font-bold">2.</span>
                            <span>Select the payment you want to complete</span>
                          </li>
                          <li className="flex items-start gap-2">
                            <span className="text-blue-600 font-bold">3.</span>
                            <span>Follow the M-Pesa prompts to complete payment</span>
                          </li>
                          <li className="flex items-start gap-2">
                            <span className="text-blue-600 font-bold">4.</span>
                            <span>Wait for payment confirmation (usually instant)</span>
                          </li>
                        </ul>
                      </div>

                      <Link href="/team/payments">
                        <Button className="w-full bg-red-600 hover:bg-red-700 text-white font-semibold text-base py-6">
                          <CreditCard className="w-5 h-5 mr-2" />
                          Complete Payment Now ({teamIssue.pending_payments.length} pending)
                          <ChevronRight className="w-5 h-5 ml-2" />
                        </Button>
                      </Link>
                    </div>
                  </AlertDescription>
                </Alert>
              )}

              {/* Player Profile Issues */}
              {teamIssue.has_incomplete_players && (
                <Alert className="border-red-300 bg-white">
                  <Users className="h-5 w-5 text-red-600" />
                  <AlertTitle className="text-red-900 font-semibold mb-2">
                    Incomplete Player Profiles
                  </AlertTitle>
                  <AlertDescription>
                    <div className="space-y-3">
                      <p className="text-red-800">
                        All players must have complete profiles before your team can be reviewed for tournament participation.
                      </p>
                      
                      <div className="grid grid-cols-1 md:grid-cols-3 gap-4 p-4 bg-red-50 rounded-lg border border-red-200">
                        <div className="text-center">
                          <p className="text-2xl font-bold text-red-900">{teamIssue.total_players}</p>
                          <p className="text-sm text-red-700">Total Players</p>
                        </div>
                        <div className="text-center">
                          <p className="text-2xl font-bold text-red-600">{teamIssue.incomplete_players_count}</p>
                          <p className="text-sm text-red-700">Incomplete Profiles</p>
                        </div>
                        <div className="text-center">
                          <p className="text-2xl font-bold text-orange-600">{teamIssue.completeness_percentage.toFixed(0)}%</p>
                          <p className="text-sm text-red-700">Profile Completion</p>
                        </div>
                      </div>

                      {!teamIssue.minimum_players_met && (
                        <div className="flex items-start gap-2 p-3 bg-red-100 rounded-lg border border-red-300">
                          <XCircle className="w-5 h-5 text-red-600 flex-shrink-0 mt-0.5" />
                          <div>
                            <p className="font-medium text-red-900">Minimum Players Requirement Not Met</p>
                            <p className="text-sm text-red-800">
                              Your team needs at least {teamIssue.minimum_players_required || 11} players with complete profiles.
                            </p>
                          </div>
                        </div>
                      )}

                      <div className="bg-blue-50 border border-blue-200 rounded-lg p-4">
                        <p className="font-medium text-blue-900 mb-2">Required Player Information:</p>
                        <ul className="text-sm text-blue-800 space-y-1 grid grid-cols-1 md:grid-cols-2 gap-2">
                          <li className="flex items-center gap-2">
                            <CheckCircle className="w-4 h-4 text-blue-600" />
                            Display Name
                          </li>
                          <li className="flex items-center gap-2">
                            <CheckCircle className="w-4 h-4 text-blue-600" />
                            Phone Number
                          </li>
                          <li className="flex items-center gap-2">
                            <CheckCircle className="w-4 h-4 text-blue-600" />
                            Date of Birth
                          </li>
                          <li className="flex items-center gap-2">
                            <CheckCircle className="w-4 h-4 text-blue-600" />
                            Gender
                          </li>
                          <li className="flex items-center gap-2">
                            <CheckCircle className="w-4 h-4 text-blue-600" />
                            Playing Position
                          </li>
                          <li className="flex items-center gap-2">
                            <CheckCircle className="w-4 h-4 text-blue-600" />
                            National ID Front
                          </li>
                          <li className="flex items-center gap-2">
                            <CheckCircle className="w-4 h-4 text-blue-600" />
                            National ID Back
                          </li>
                          <li className="flex items-center gap-2">
                            <CheckCircle className="w-4 h-4 text-blue-600" />
                            ID Number
                          </li>
                        </ul>
                      </div>

                      <Link href="/team/players">
                        <Button className="bg-red-600 hover:bg-red-700 text-white font-semibold">
                          <Users className="w-4 h-4 mr-2" />
                          Complete Player Profiles Now
                          <ChevronRight className="w-4 h-4 ml-1" />
                        </Button>
                      </Link>
                    </div>
                  </AlertDescription>
                </Alert>
              )}

              {/* Important Notice */}
              <div className="bg-amber-50 border-2 border-amber-400 rounded-lg p-4 shadow-lg">
                <div className="flex items-start gap-3">
                  <AlertTriangle className="w-6 h-6 text-amber-600 flex-shrink-0 mt-1" />
                  <div className="flex-1">
                    <p className="font-bold text-amber-900 text-xl mb-2">
                      🚫 TEAM REVIEW BLOCKED
                    </p>
                    <p className="text-amber-800 font-medium text-base mb-3">
                      Your team registration will NOT be reviewed by tournament organizers until:
                    </p>
                    <ul className="mt-2 space-y-2 text-amber-800">
                      {teamIssue.has_pending_payment && (
                        <li className="flex items-start gap-3 p-2 bg-amber-100 rounded">
                          <span className="w-5 h-5 bg-amber-600 rounded-full flex items-center justify-center text-white font-bold text-sm flex-shrink-0 mt-0.5">
                            {teamIssue.pending_payments.length}
                          </span>
                          <div>
                            <p className="font-semibold">All tournament registration fees are paid</p>
                            <p className="text-sm">
                              {teamIssue.pending_payments.length} payment{teamIssue.pending_payments.length !== 1 ? 's' : ''} totaling{' '}
                              {formatCurrencyTotals(teamIssue.pending_payments)}{' '}
                              pending
                            </p>
                          </div>
                        </li>
                      )}
                      {teamIssue.has_incomplete_players && (
                        <li className="flex items-start gap-3 p-2 bg-amber-100 rounded">
                          <span className="w-5 h-5 bg-amber-600 rounded-full flex items-center justify-center text-white font-bold text-sm flex-shrink-0 mt-0.5">
                            {teamIssue.incomplete_players_count}
                          </span>
                          <div>
                            <p className="font-semibold">All player profiles are 100% complete</p>
                            <p className="text-sm">
                              {teamIssue.incomplete_players_count} player{teamIssue.incomplete_players_count !== 1 ? 's' : ''} with incomplete profiles
                            </p>
                          </div>
                        </li>
                      )}
                    </ul>
                    <div className="mt-4 p-3 bg-red-100 border border-red-300 rounded">
                      <p className="text-red-900 font-bold text-base">
                        ⏱️ URGENT: Complete these requirements immediately!
                      </p>
                      <p className="text-red-800 text-sm mt-1">
                        Delays in completing payment and player profiles will prevent your team from participating in tournaments.
                        Tournament organizers cannot process your application until all requirements are met.
                      </p>
                    </div>
                  </div>
                </div>
              </div>
            </div>
              </CardContent>
            </Card>
          ))}
        </>
      )}
    </div>
  );
}

