import React from 'react';
import { Badge } from '@/components/ui/badge';

interface Player {
  id: number;
  user: {
    id: number;
    name: string;
    email: string;
  };
  display_name?: string;
  avatar?: string;
  football_position?: string;
  jersey_number?: number;
  status: string;
}

interface Position {
  x: number;
  y: number;
  label: string;
  playerId?: number;
}

interface FormationFieldProps {
  formation: {
    id: number;
    name: string;
    player_positions: Record<string, Position>;
    formation_data: any;
  };
  players: Player[];
  onPlayerClick?: (player: Player | null, position: string) => void;
  showPlayerNames?: boolean;
  className?: string;
}

export default function FormationField({ 
  formation, 
  players, 
  onPlayerClick,
  showPlayerNames = true,
  className = ""
}: FormationFieldProps) {
  
  const getPlayerForPosition = (positionKey: string) => {
    const position = formation.player_positions[positionKey];
    if (position?.playerId) {
      return players.find(p => p.id === position.playerId);
    }
    return null;
  };

  const getPositionColor = (key: string) => {
    if (key === 'goalkeeper') return 'bg-blue-500 text-white';
    if (key.startsWith('defender')) return 'bg-green-500 text-white';
    if (key.startsWith('midfielder')) return 'bg-yellow-500 text-white';
    if (key.startsWith('forward')) return 'bg-red-500 text-white';
    return 'bg-gray-500 text-white';
  };

  const getPositionType = (key: string) => {
    if (key === 'goalkeeper') return 'GK';
    if (key.startsWith('defender')) return 'DEF';
    if (key.startsWith('midfielder')) return 'MID';
    if (key.startsWith('forward')) return 'FWD';
    return 'POS';
  };

  return (
    <div className={`relative ${className}`}>
      {/* Football Field */}
      <div className="relative w-full h-[400px] bg-green-600 border-4 border-white rounded-lg overflow-hidden">
        {/* Field Lines */}
        <div className="absolute inset-0">
          {/* Center Line */}
          <div className="absolute left-1/2 top-0 bottom-0 w-0.5 bg-white transform -translate-x-0.5"></div>
          
          {/* Center Circle */}
          <div className="absolute top-1/2 left-1/2 w-32 h-32 border-2 border-white rounded-full transform -translate-x-1/2 -translate-y-1/2"></div>
          
          {/* Center Spot */}
          <div className="absolute top-1/2 left-1/2 w-2 h-2 bg-white rounded-full transform -translate-x-1/2 -translate-y-1/2"></div>
          
          {/* Goal Areas */}
          <div className="absolute left-0 top-1/2 w-16 h-32 border-2 border-white transform -translate-y-1/2">
            <div className="absolute left-0 top-1/2 w-8 h-8 border-2 border-white transform -translate-y-1/2"></div>
          </div>
          <div className="absolute right-0 top-1/2 w-16 h-32 border-2 border-white transform -translate-y-1/2">
            <div className="absolute right-0 top-1/2 w-8 h-8 border-2 border-white transform -translate-y-1/2"></div>
          </div>
          
          {/* Penalty Areas */}
          <div className="absolute left-0 top-1/2 w-24 h-48 border-2 border-white transform -translate-y-1/2">
            <div className="absolute left-0 top-1/2 w-12 h-12 border-2 border-white transform -translate-y-1/2"></div>
            <div className="absolute left-0 top-1/2 w-2 h-2 bg-white rounded-full transform -translate-y-1/2 translate-x-4"></div>
          </div>
          <div className="absolute right-0 top-1/2 w-24 h-48 border-2 border-white transform -translate-y-1/2">
            <div className="absolute right-0 top-1/2 w-12 h-12 border-2 border-white transform -translate-y-1/2"></div>
            <div className="absolute right-0 top-1/2 w-2 h-2 bg-white rounded-full transform -translate-y-1/2 -translate-x-4"></div>
          </div>
          
          {/* Corner Arcs */}
          <div className="absolute top-0 left-0 w-4 h-4 border-l-2 border-t-2 border-white rounded-tl-full"></div>
          <div className="absolute top-0 right-0 w-4 h-4 border-r-2 border-t-2 border-white rounded-tr-full"></div>
          <div className="absolute bottom-0 left-0 w-4 h-4 border-l-2 border-b-2 border-white rounded-bl-full"></div>
          <div className="absolute bottom-0 right-0 w-4 h-4 border-r-2 border-b-2 border-white rounded-br-full"></div>
        </div>

        {/* Players */}
        {Object.entries(formation.player_positions).map(([key, position]) => {
          const player = getPlayerForPosition(key);
          const x = (position.x / 100) * 100; // Convert to percentage
          const y = (position.y / 100) * 100; // Convert to percentage
          
          return (
            <div
              key={key}
              className="absolute transform -translate-x-1/2 -translate-y-1/2 cursor-pointer group"
              style={{
                left: `${x}%`,
                top: `${y}%`,
              }}
              onClick={() => onPlayerClick?.(player ?? null, key)}
            >
              {/* Player Circle */}
              <div className={`w-12 h-12 rounded-full border-2 border-white flex items-center justify-center text-xs font-bold shadow-lg hover:shadow-xl transition-all duration-200 group-hover:scale-110 ${getPositionColor(key)}`}>
                {player?.jersey_number || '?'}
              </div>
              
              {/* Player Name */}
              {showPlayerNames && (
                <div className="absolute top-14 left-1/2 transform -translate-x-1/2 text-center">
                  <div className="bg-black bg-opacity-75 text-white text-xs px-2 py-1 rounded whitespace-nowrap">
                    {player?.display_name || player?.user?.name || position.label}
                  </div>
                  <div className="text-xs text-white font-semibold mt-1">
                    {getPositionType(key)}
                  </div>
                </div>
              )}
              
              {/* Position Label (when no player assigned) */}
              {!player && (
                <div className="absolute top-14 left-1/2 transform -translate-x-1/2 text-center">
                  <div className="bg-gray-800 bg-opacity-75 text-white text-xs px-2 py-1 rounded whitespace-nowrap">
                    {position.label}
                  </div>
                  <div className="text-xs text-white font-semibold mt-1">
                    {getPositionType(key)}
                  </div>
                </div>
              )}
            </div>
          );
        })}
        
        {/* Formation Name */}
        <div className="absolute top-4 left-4 bg-black bg-opacity-75 text-white px-3 py-1 rounded text-sm font-semibold">
          {formation.name}
        </div>
        
        {/* Team Direction Indicator */}
        <div className="absolute top-4 right-4 bg-black bg-opacity-75 text-white px-3 py-1 rounded text-sm">
          ➡️ Attack
        </div>
      </div>
    </div>
  );
}
