import React, { useState } from 'react';
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Badge } from "@/components/ui/badge";
import { 
  Download, 
  Eye, 
  Settings, 
  Palette,
  User,
  Calendar,
  MapPin,
  Award,
  QrCode
} from 'lucide-react';

interface CertificateData {
  recipient: {
    name: string;
    displayName: string;
    team: string;
    position: string;
    jerseyNumber: string;
  };
  tournament: {
    name: string;
    startDate: string;
    endDate: string;
    location: string;
    edition: string;
  };
  certificateId: string;
  generatedAt: string;
  validUntil: string;
  styling: {
    header_color: string;
    accent_color: string;
    text_color: string;
    recipient_color: string;
    logo_text: string;
    logo_subtitle: string;
    logo_motto: string;
  };
}

const SvgCertificatePreview = () => {
  const [certificateData, setCertificateData] = useState<CertificateData>({
    recipient: {
      name: 'Victor Kipkoech',
      displayName: 'Victor Kipkoech',
      team: 'Lwanda K FC',
      position: 'Striker',
      jerseyNumber: '10',
    },
    tournament: {
      name: 'Milimus Tournament Championship 2024',
      startDate: 'August 3, 2025',
      endDate: 'August 31, 2025',
      location: 'Nairobi, Kenya',
      edition: '1st Edition',
    },
    certificateId: 'CERT-000075-770DE884',
    generatedAt: 'September 20, 2025',
    validUntil: 'September 20, 2026',
    styling: {
      header_color: '#c3292b',
      accent_color: '#206633',
      text_color: '#4f585a',
      recipient_color: '#2b672f',
      logo_text: 'THE MILIMUS',
      logo_subtitle: 'TOURNAMENT CUP',
      logo_motto: 'IN GOD WE TRUST',
    }
  });

  const [activeTab, setActiveTab] = useState<'preview' | 'edit'>('preview');

  const handleInputChange = (section: string, field: string, value: string) => {
    setCertificateData(prev => ({
      ...prev,
      [section]: {
        ...prev[section as keyof CertificateData],
        [field]: value
      }
    }));
  };

  const handleStylingChange = (field: string, value: string) => {
    setCertificateData(prev => ({
      ...prev,
      styling: {
        ...prev.styling,
        [field]: value
      }
    }));
  };

  const generateCertificate = () => {
    const params = new URLSearchParams();
    
    // Add recipient data
    Object.entries(certificateData.recipient).forEach(([key, value]) => {
      params.append(key, value);
    });
    
    // Add tournament data
    Object.entries(certificateData.tournament).forEach(([key, value]) => {
      params.append(`tournament_${key}`, value);
    });
    
    // Add certificate metadata
    params.append('certificate_id', certificateData.certificateId);
    params.append('generated_at', certificateData.generatedAt);
    params.append('valid_until', certificateData.validUntil);
    
    // Add styling
    Object.entries(certificateData.styling).forEach(([key, value]) => {
      params.append(key, value);
    });
    
    const downloadUrl = `/certificate/generate?${params.toString()}`;
    window.open(downloadUrl, '_blank');
  };

  const previewCertificate = () => {
    const params = new URLSearchParams();
    
    // Add recipient data
    Object.entries(certificateData.recipient).forEach(([key, value]) => {
      params.append(key, value);
    });
    
    // Add tournament data
    Object.entries(certificateData.tournament).forEach(([key, value]) => {
      params.append(`tournament_${key}`, value);
    });
    
    // Add certificate metadata
    params.append('certificate_id', certificateData.certificateId);
    params.append('generated_at', certificateData.generatedAt);
    params.append('valid_until', certificateData.validUntil);
    
    // Add styling
    Object.entries(certificateData.styling).forEach(([key, value]) => {
      params.append(key, value);
    });
    
    const previewUrl = `/certificate/preview?${params.toString()}`;
    window.open(previewUrl, '_blank');
  };

  return (
    <div className="max-w-6xl mx-auto p-6 space-y-6">
      <div className="text-center mb-8">
        <h1 className="text-3xl font-bold text-gray-900 mb-2">
          SVG Certificate Generator
        </h1>
        <p className="text-gray-600">
          Generate certificates using the exact SVG design with full customization options
        </p>
      </div>

      <div className="grid grid-cols-1 lg:grid-cols-3 gap-6">
        {/* Controls Panel */}
        <div className="lg:col-span-1 space-y-6">
          <Card>
            <CardHeader>
              <CardTitle className="flex items-center">
                <Settings className="h-5 w-5 mr-2" />
                Certificate Settings
              </CardTitle>
            </CardHeader>
            <CardContent className="space-y-4">
              <div className="flex space-x-2">
                <Button
                  variant={activeTab === 'preview' ? 'default' : 'outline'}
                  onClick={() => setActiveTab('preview')}
                  className="flex-1"
                >
                  <Eye className="h-4 w-4 mr-2" />
                  Preview
                </Button>
                <Button
                  variant={activeTab === 'edit' ? 'default' : 'outline'}
                  onClick={() => setActiveTab('edit')}
                  className="flex-1"
                >
                  <Settings className="h-4 w-4 mr-2" />
                  Edit
                </Button>
              </div>

              {activeTab === 'edit' && (
                <div className="space-y-4">
                  {/* Recipient Information */}
                  <div>
                    <h3 className="font-semibold text-sm text-gray-700 mb-3 flex items-center">
                      <User className="h-4 w-4 mr-2" />
                      Recipient Information
                    </h3>
                    <div className="space-y-3">
                      <div>
                        <Label htmlFor="name">Name</Label>
                        <Input
                          id="name"
                          value={certificateData.recipient.name}
                          onChange={(e) => handleInputChange('recipient', 'name', e.target.value)}
                          placeholder="Enter recipient name"
                        />
                      </div>
                      <div>
                        <Label htmlFor="team">Team</Label>
                        <Input
                          id="team"
                          value={certificateData.recipient.team}
                          onChange={(e) => handleInputChange('recipient', 'team', e.target.value)}
                          placeholder="Enter team name"
                        />
                      </div>
                      <div>
                        <Label htmlFor="position">Position</Label>
                        <Input
                          id="position"
                          value={certificateData.recipient.position}
                          onChange={(e) => handleInputChange('recipient', 'position', e.target.value)}
                          placeholder="Enter position"
                        />
                      </div>
                      <div>
                        <Label htmlFor="jerseyNumber">Jersey Number</Label>
                        <Input
                          id="jerseyNumber"
                          value={certificateData.recipient.jerseyNumber}
                          onChange={(e) => handleInputChange('recipient', 'jerseyNumber', e.target.value)}
                          placeholder="Enter jersey number"
                        />
                      </div>
                    </div>
                  </div>

                  {/* Tournament Information */}
                  <div>
                    <h3 className="font-semibold text-sm text-gray-700 mb-3 flex items-center">
                      <Award className="h-4 w-4 mr-2" />
                      Tournament Information
                    </h3>
                    <div className="space-y-3">
                      <div>
                        <Label htmlFor="tournament_name">Tournament Name</Label>
                        <Input
                          id="tournament_name"
                          value={certificateData.tournament.name}
                          onChange={(e) => handleInputChange('tournament', 'name', e.target.value)}
                          placeholder="Enter tournament name"
                        />
                      </div>
                      <div>
                        <Label htmlFor="start_date">Start Date</Label>
                        <Input
                          id="start_date"
                          value={certificateData.tournament.startDate}
                          onChange={(e) => handleInputChange('tournament', 'startDate', e.target.value)}
                          placeholder="Enter start date"
                        />
                      </div>
                      <div>
                        <Label htmlFor="end_date">End Date</Label>
                        <Input
                          id="end_date"
                          value={certificateData.tournament.endDate}
                          onChange={(e) => handleInputChange('tournament', 'endDate', e.target.value)}
                          placeholder="Enter end date"
                        />
                      </div>
                      <div>
                        <Label htmlFor="location">Location</Label>
                        <Input
                          id="location"
                          value={certificateData.tournament.location}
                          onChange={(e) => handleInputChange('tournament', 'location', e.target.value)}
                          placeholder="Enter location"
                        />
                      </div>
                    </div>
                  </div>

                  {/* Styling Options */}
                  <div>
                    <h3 className="font-semibold text-sm text-gray-700 mb-3 flex items-center">
                      <Palette className="h-4 w-4 mr-2" />
                      Styling Options
                    </h3>
                    <div className="space-y-3">
                      <div>
                        <Label htmlFor="header_color">Header Color</Label>
                        <div className="flex items-center space-x-2">
                          <Input
                            id="header_color"
                            type="color"
                            value={certificateData.styling.header_color}
                            onChange={(e) => handleStylingChange('header_color', e.target.value)}
                            className="w-12 h-8"
                          />
                          <Input
                            value={certificateData.styling.header_color}
                            onChange={(e) => handleStylingChange('header_color', e.target.value)}
                            className="flex-1"
                          />
                        </div>
                      </div>
                      <div>
                        <Label htmlFor="accent_color">Accent Color</Label>
                        <div className="flex items-center space-x-2">
                          <Input
                            id="accent_color"
                            type="color"
                            value={certificateData.styling.accent_color}
                            onChange={(e) => handleStylingChange('accent_color', e.target.value)}
                            className="w-12 h-8"
                          />
                          <Input
                            value={certificateData.styling.accent_color}
                            onChange={(e) => handleStylingChange('accent_color', e.target.value)}
                            className="flex-1"
                          />
                        </div>
                      </div>
                      <div>
                        <Label htmlFor="logo_text">Logo Text</Label>
                        <Input
                          id="logo_text"
                          value={certificateData.styling.logo_text}
                          onChange={(e) => handleStylingChange('logo_text', e.target.value)}
                          placeholder="Enter logo text"
                        />
                      </div>
                      <div>
                        <Label htmlFor="logo_subtitle">Logo Subtitle</Label>
                        <Input
                          id="logo_subtitle"
                          value={certificateData.styling.logo_subtitle}
                          onChange={(e) => handleStylingChange('logo_subtitle', e.target.value)}
                          placeholder="Enter logo subtitle"
                        />
                      </div>
                    </div>
                  </div>
                </div>
              )}

              {activeTab === 'preview' && (
                <div className="space-y-4">
                  <div className="space-y-3">
                    <div className="flex items-center justify-between">
                      <span className="text-sm font-medium">Recipient:</span>
                      <Badge variant="outline">{certificateData.recipient.name}</Badge>
                    </div>
                    <div className="flex items-center justify-between">
                      <span className="text-sm font-medium">Team:</span>
                      <Badge variant="outline">{certificateData.recipient.team}</Badge>
                    </div>
                    <div className="flex items-center justify-between">
                      <span className="text-sm font-medium">Tournament:</span>
                      <Badge variant="outline">{certificateData.tournament.name}</Badge>
                    </div>
                    <div className="flex items-center justify-between">
                      <span className="text-sm font-medium">Certificate ID:</span>
                      <Badge variant="secondary">{certificateData.certificateId}</Badge>
                    </div>
                  </div>

                  <div className="pt-4 border-t">
                    <h4 className="font-medium text-sm mb-3">Color Scheme</h4>
                    <div className="flex space-x-2">
                      <div className="w-8 h-8 rounded border" style={{ backgroundColor: certificateData.styling.header_color }}></div>
                      <div className="w-8 h-8 rounded border" style={{ backgroundColor: certificateData.styling.accent_color }}></div>
                      <div className="w-8 h-8 rounded border" style={{ backgroundColor: certificateData.styling.text_color }}></div>
                    </div>
                  </div>
                </div>
              )}

              <div className="pt-4 space-y-3">
                <Button onClick={previewCertificate} className="w-full" variant="outline">
                  <Eye className="h-4 w-4 mr-2" />
                  Preview Certificate
                </Button>
                <Button onClick={generateCertificate} className="w-full">
                  <Download className="h-4 w-4 mr-2" />
                  Download PDF
                </Button>
              </div>
            </CardContent>
          </Card>
        </div>

        {/* Certificate Preview */}
        <div className="lg:col-span-2">
          <Card>
            <CardHeader>
              <CardTitle className="flex items-center">
                <QrCode className="h-5 w-5 mr-2" />
                Certificate Preview
              </CardTitle>
            </CardHeader>
            <CardContent>
              <div className="bg-gray-100 p-8 rounded-lg">
                <div className="bg-white shadow-lg rounded-lg overflow-hidden" style={{ aspectRatio: '842/595' }}>
                  <iframe
                    src={`/certificate/preview?name=${encodeURIComponent(certificateData.recipient.name)}&team=${encodeURIComponent(certificateData.recipient.team)}&position=${encodeURIComponent(certificateData.recipient.position)}&tournament_name=${encodeURIComponent(certificateData.tournament.name)}&start_date=${encodeURIComponent(certificateData.tournament.startDate)}&end_date=${encodeURIComponent(certificateData.tournament.endDate)}&location=${encodeURIComponent(certificateData.tournament.location)}&certificate_id=${encodeURIComponent(certificateData.certificateId)}&generated_at=${encodeURIComponent(certificateData.generatedAt)}&valid_until=${encodeURIComponent(certificateData.validUntil)}&header_color=${encodeURIComponent(certificateData.styling.header_color)}&accent_color=${encodeURIComponent(certificateData.styling.accent_color)}&logo_text=${encodeURIComponent(certificateData.styling.logo_text)}&logo_subtitle=${encodeURIComponent(certificateData.styling.logo_subtitle)}&logo_motto=${encodeURIComponent(certificateData.styling.logo_motto)}`}
                    className="w-full h-full border-0"
                    title="Certificate Preview"
                  />
                </div>
              </div>
            </CardContent>
          </Card>
        </div>
      </div>
    </div>
  );
};

export default SvgCertificatePreview;
