import React from 'react';
import CertificatePreviewButton, { CertificateDownloadButton } from './CertificatePreviewButton';
import { Button } from '@/components/ui/button';
import { MoreHorizontal, Eye, Download, Printer } from 'lucide-react';
import {
    DropdownMenu,
    DropdownMenuContent,
    DropdownMenuItem,
    DropdownMenuTrigger,
} from '@/components/ui/dropdown-menu';

interface Player {
    id: number;
    display_name: string;
    team_name?: string;
    position?: string;
    jersey_number?: string;
    tournament_matches_played?: number;
    tournament_win_rate?: number;
    global_rating?: number;
    verification_status?: string;
    tournament?: {
        name: string;
    };
}

interface PlayerCertificateActionsProps {
    player: Player;
    showDropdown?: boolean;
    size?: 'default' | 'sm' | 'lg';
}

const PlayerCertificateActions: React.FC<PlayerCertificateActionsProps> = ({
    player,
    showDropdown = true,
    size = 'sm'
}) => {
    const handleDirectDownload = () => {
        const url = `/admin/players/${player.id}/certificate/print`;
        const link = document.createElement('a');
        link.href = url;
        link.download = `Certificate_${player.display_name.replace(/\s+/g, '_')}_${new Date().toISOString().split('T')[0]}.pdf`;
        link.target = '_blank';
        document.body.appendChild(link);
        link.click();
        document.body.removeChild(link);
    };

    const handleDirectPrint = () => {
        const url = `/admin/players/${player.id}/certificate`;
        const printWindow = window.open(url, '_blank', 'width=800,height=600');
        
        if (printWindow) {
            printWindow.onload = () => {
                printWindow.print();
                printWindow.onafterprint = () => {
                    printWindow.close();
                };
            };
        }
    };

    const handleDirectView = () => {
        const url = `/admin/players/${player.id}/certificate`;
        window.open(url, '_blank');
    };

    if (showDropdown) {
        return (
            <DropdownMenu>
                <DropdownMenuTrigger asChild>
                    <Button variant="ghost" size={size} className="h-8 w-8 p-0">
                        <span className="sr-only">Open certificate menu</span>
                        <MoreHorizontal className="h-4 w-4" />
                    </Button>
                </DropdownMenuTrigger>
                <DropdownMenuContent align="end">
                    <DropdownMenuItem onClick={handleDirectView}>
                        <Eye className="mr-2 h-4 w-4" />
                        View Certificate
                    </DropdownMenuItem>
                    <DropdownMenuItem onClick={handleDirectPrint}>
                        <Printer className="mr-2 h-4 w-4" />
                        Print Certificate
                    </DropdownMenuItem>
                    <DropdownMenuItem onClick={handleDirectDownload}>
                        <Download className="mr-2 h-4 w-4" />
                        Download PDF
                    </DropdownMenuItem>
                </DropdownMenuContent>
            </DropdownMenu>
        );
    }

    return (
        <div className="flex gap-2">
            <CertificatePreviewButton
                playerId={player.id}
                playerName={player.display_name}
                teamName={player.team_name}
                position={player.position}
                jerseyNumber={player.jersey_number}
                tournamentMatchesPlayed={player.tournament_matches_played}
                tournamentWinRate={player.tournament_win_rate}
                globalRating={player.global_rating}
                verificationStatus={player.verification_status}
                tournamentName={player.tournament?.name}
                variant="outline"
                size={size}
                showText={false}
            />
            <CertificateDownloadButton
                playerId={player.id}
                playerName={player.display_name}
                variant="default"
                size={size}
                showText={false}
            />
        </div>
    );
};

export default PlayerCertificateActions;
