import React, { useState, useEffect, useRef, useCallback } from 'react';
import { FilePond, registerPlugin } from 'react-filepond';
import { Dialog, DialogContent, DialogDescription, DialogHeader, DialogTitle, DialogTrigger } from '@/components/ui/dialog';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Badge } from '@/components/ui/badge';
import { Card, CardContent } from '@/components/ui/card';
import { Tabs, TabsContent, TabsList, TabsTrigger } from '@/components/ui/tabs';
import { Tooltip, TooltipContent, TooltipProvider, TooltipTrigger } from '@/components/ui/tooltip';
import { 
  Upload, 
  Image as ImageIcon, 
  Video, 
  FileText, 
  Music, 
  Search, 
  Grid3X3, 
  List, 
  Trash2, 
  ChevronLeft,
  ChevronRight,
  Copy,
  Eye,
  Filter,
  SortAsc,
  SortDesc,
  RefreshCw,
  X,
  Check,
  AlertCircle
} from 'lucide-react';
import { toast } from 'sonner';

// FilePond plugins
import FilePondPluginImagePreview from 'filepond-plugin-image-preview';
import FilePondPluginImageCrop from 'filepond-plugin-image-crop';
import FilePondPluginImageResize from 'filepond-plugin-image-resize';
import FilePondPluginImageTransform from 'filepond-plugin-image-transform';
import FilePondPluginFileValidateType from 'filepond-plugin-file-validate-type';
import FilePondPluginFileValidateSize from 'filepond-plugin-file-validate-size';

// FilePond styles
import 'filepond/dist/filepond.min.css';
import 'filepond-plugin-image-preview/dist/filepond-plugin-image-preview.css';

// Register plugins
registerPlugin(
  FilePondPluginImagePreview,
  FilePondPluginImageCrop,
  FilePondPluginImageResize,
  FilePondPluginImageTransform,
  FilePondPluginFileValidateType,
  FilePondPluginFileValidateSize
);

interface MediaFile {
  id: number;
  name: string;
  file_name: string;
  mime_type: string;
  size: number;
  url: string;
  human_readable_size: string;
  is_image: boolean;
  is_video: boolean;
  is_audio: boolean;
  is_document: boolean;
  alt_text?: string;
  description?: string;
  collection_name?: string;
  created_at: string;
  user?: {
    name: string;
  };
}

interface MediaManagerProps {
  trigger?: React.ReactNode;
  onSelect?: (file: MediaFile | MediaFile[]) => void;
  multiple?: boolean;
  acceptedTypes?: string[];
  collection?: string;
  maxFiles?: number;
  allowCrop?: boolean;
  cropAspectRatio?: number;
  className?: string;
  showPreview?: boolean;
  allowDelete?: boolean;
}

type SortField = 'name' | 'size' | 'created_at' | 'type';
type SortOrder = 'asc' | 'desc';
type ViewMode = 'grid' | 'list';

const MediaManager: React.FC<MediaManagerProps> = ({
  trigger,
  onSelect,
  multiple = false,
  acceptedTypes = ['image/*', 'video/*', 'audio/*', 'application/pdf', 'text/*'],
  collection = 'default',
  maxFiles = 10,
  allowCrop = true,
  cropAspectRatio,
  className = '',
  showPreview = true,
  allowDelete = true
}) => {
  const [isOpen, setIsOpen] = useState(false);
  const [activeTab, setActiveTab] = useState('upload');
  const [libraryTab, setLibraryTab] = useState('images');
  const [files, setFiles] = useState<any[]>([]);
  const [mediaFiles, setMediaFiles] = useState<MediaFile[]>([]);
  const [loading, setLoading] = useState(false);
  const [selectedFiles, setSelectedFiles] = useState<MediaFile[]>([]);
  const [viewMode, setViewMode] = useState<ViewMode>('grid');
  const [searchQuery, setSearchQuery] = useState('');
  const [imageSearchQuery, setImageSearchQuery] = useState('');
  const [documentSearchQuery, setDocumentSearchQuery] = useState('');
  const [currentPage, setCurrentPage] = useState(1);
  const [itemsPerPage] = useState(24);
  const [sortField, setSortField] = useState<SortField>('created_at');
  const [sortOrder, setSortOrder] = useState<SortOrder>('desc');
  const [filterType, setFilterType] = useState<string>('all');
  const [isUploading, setIsUploading] = useState(false);
  const [uploadProgress, setUploadProgress] = useState(0);

  const filePondRef = useRef<FilePond>(null);
  const searchTimeoutRef = useRef<NodeJS.Timeout | null>(null);

  // Debounced search for better performance
  const debouncedSearch = useCallback((query: string, type: 'images' | 'documents') => {
    if (searchTimeoutRef.current) {
      clearTimeout(searchTimeoutRef.current);
    }
    
    searchTimeoutRef.current = setTimeout(() => {
      if (type === 'images') {
        setImageSearchQuery(query);
      } else {
        setDocumentSearchQuery(query);
      }
      setCurrentPage(1);
    }, 300);
  }, []);

  // Load media files when dialog opens or searches change
  useEffect(() => {
    if (isOpen) {
      setMediaFiles([]);
      loadMediaFiles();
    }
  }, [isOpen, imageSearchQuery, documentSearchQuery, sortField, sortOrder, filterType]);

  // Reset current page when switching tabs
  useEffect(() => {
    setCurrentPage(1);
  }, [libraryTab]);

  const loadMediaFiles = async () => {
    setLoading(true);
    try {
      const params = new URLSearchParams({
        collection: collection,
        per_page: '100',
        sort: sortField,
        order: sortOrder,
        filter: filterType
      });
      
      const response = await fetch(`/media?${params}`, {
        headers: {
          'Cache-Control': 'no-cache',
          'X-Requested-With': 'XMLHttpRequest'
        }
      });
      const data = await response.json();
      
      if (data.success && Array.isArray(data.data)) {
        setMediaFiles(data.data);
      } else {
        setMediaFiles([]);
        toast.error('Failed to load media files');
      }
    } catch (error) {
      setMediaFiles([]);
      toast.error('Failed to load media files');
    } finally {
      setLoading(false);
    }
  };

  const handleFileUpload = async (files: any[]) => {
    setIsUploading(true);
    setUploadProgress(0);
    
    try {
      const formData = new FormData();
      files.forEach((file, index) => {
        formData.append(`files[${index}]`, file.file);
      });
      formData.append('collection', collection);

      const xhr = new XMLHttpRequest();
      
      xhr.upload.addEventListener('progress', (e) => {
        if (e.lengthComputable) {
          const progress = (e.loaded / e.total) * 100;
          setUploadProgress(progress);
        }
      });

      xhr.addEventListener('load', async () => {
        if (xhr.status === 200) {
          const data = JSON.parse(xhr.responseText);
          if (data.success) {
            toast.success('Files uploaded successfully');
            setFiles([]);
            setMediaFiles([]);
            await loadMediaFiles();
          } else {
            toast.error(data.message || 'Upload failed');
          }
        } else {
          toast.error('Upload failed');
        }
        setIsUploading(false);
        setUploadProgress(0);
      });

      xhr.addEventListener('error', () => {
        toast.error('Upload failed');
        setIsUploading(false);
        setUploadProgress(0);
      });

      xhr.open('POST', '/media/upload');
      xhr.setRequestHeader('X-CSRF-TOKEN', document.querySelector('meta[name="csrf-token"]')?.getAttribute('content') || '');
      xhr.setRequestHeader('X-Requested-With', 'XMLHttpRequest');
      xhr.send(formData);
    } catch (error) {
      toast.error('Upload failed');
      setIsUploading(false);
      setUploadProgress(0);
    }
  };

  const handleFileSelect = (file: MediaFile) => {
    if (multiple) {
      setSelectedFiles(prev => {
        const exists = prev.find(f => f.id === file.id);
        if (exists) {
          return prev.filter(f => f.id !== file.id);
        } else {
          return [...prev, file];
        }
      });
    } else {
      setSelectedFiles([file]);
    }
  };

  const handleConfirmSelection = () => {
    if (onSelect) {
      onSelect(multiple ? selectedFiles : selectedFiles[0]);
    }
    setIsOpen(false);
    setSelectedFiles([]);
  };

  const handleFileDelete = async (fileId: number) => {
    try {
      const response = await fetch(`/media/${fileId}`, {
        method: 'DELETE',
        headers: {
          'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]')?.getAttribute('content') || '',
          'X-Requested-With': 'XMLHttpRequest'
        }
      });

      const data = await response.json();
      
      if (data.success) {
        toast.success('File deleted successfully');
        setSelectedFiles(prev => prev.filter(f => f.id !== fileId));
        setMediaFiles([]);
        await loadMediaFiles();
      } else {
        toast.error(data.message || 'Delete failed');
      }
    } catch (error) {
      toast.error('Delete failed');
    }
  };

  const handleFileCopy = async (file: MediaFile) => {
    try {
      await navigator.clipboard.writeText(file.url);
      toast.success('File URL copied to clipboard');
    } catch (error) {
      toast.error('Failed to copy URL');
    }
  };

  const getFileIcon = (file: MediaFile) => {
    if (file.is_image) return <ImageIcon className="h-12 w-12 text-blue-500" />;
    if (file.is_video) return <Video className="h-12 w-12 text-red-500" />;
    if (file.is_audio) return <Music className="h-12 w-12 text-green-500" />;
    return <FileText className="h-12 w-12 text-gray-500" />;
  };

  const formatFileSize = (bytes: number) => {
    if (bytes === 0) return '0 Bytes';
    const k = 1024;
    const sizes = ['Bytes', 'KB', 'MB', 'GB'];
    const i = Math.floor(Math.log(bytes) / Math.log(k));
    return parseFloat((bytes / Math.pow(k, i)).toFixed(2)) + ' ' + sizes[i];
  };

  const getFilteredFiles = (type: 'images' | 'documents') => {
    if (!Array.isArray(mediaFiles)) return [];
    
    const searchQuery = type === 'images' ? imageSearchQuery : documentSearchQuery;
    
    return mediaFiles.filter(file => {
      const matchesSearch = file.name.toLowerCase().includes(searchQuery.toLowerCase());
      const matchesType = type === 'images' 
        ? (file.is_image || file.is_video) 
        : (file.is_document || file.is_audio || (!file.is_image && !file.is_video && !file.is_document && !file.is_audio));
      return matchesSearch && matchesType;
    });
  };

  const getPaginatedFiles = (type: 'images' | 'documents') => {
    const filteredFiles = getFilteredFiles(type);
    const startIndex = (currentPage - 1) * itemsPerPage;
    const endIndex = startIndex + itemsPerPage;
    return {
      files: filteredFiles.slice(startIndex, endIndex),
      total: filteredFiles.length,
      totalPages: Math.ceil(filteredFiles.length / itemsPerPage)
    };
  };

  const renderFileCard = (file: MediaFile, type: 'images' | 'documents') => {
    const isSelected = selectedFiles.some(f => f.id === file.id);
    
    return (
      <Card
        key={file.id}
        className={`cursor-pointer transition-all hover:shadow-lg group ${
          isSelected ? 'ring-2 ring-primary bg-primary/5' : 'hover:bg-gray-50'
        }`}
        onClick={() => handleFileSelect(file)}
      >
        <CardContent className="p-4">
          <div className="relative">
            <div className="aspect-square bg-gray-100  flex items-center justify-center overflow-hidden">
              {file.is_image ? (
                <img
                  src={file.url}
                  alt={file.alt_text || file.name}
                  className="w-full h-full object-cover rounded-lg group-hover:scale-105 transition-transform duration-200"
                  loading="lazy"
                />
              ) : (
                getFileIcon(file)
              )}
              
              {/* Overlay with actions */}
              <div className="absolute inset-0 bg-black/0 group-hover:bg-black/20 transition-colors duration-200 rounded-lg flex items-center justify-center opacity-0 group-hover:opacity-100">
                <div className="flex space-x-2">
                  <TooltipProvider>
                    <Tooltip>
                      <TooltipTrigger asChild>
                        <Button
                          size="sm"
                          variant="secondary"
                          onClick={(e) => {
                            e.stopPropagation();
                            handleFileCopy(file);
                          }}
                        >
                          <Copy className="h-4 w-4" />
                        </Button>
                      </TooltipTrigger>
                      <TooltipContent>Copy URL</TooltipContent>
                    </Tooltip>
                  </TooltipProvider>
                  
                  {allowDelete && (
                    <TooltipProvider>
                      <Tooltip>
                        <TooltipTrigger asChild>
                          <Button
                            size="sm"
                            variant="destructive"
                            onClick={(e) => {
                              e.stopPropagation();
                              handleFileDelete(file.id);
                            }}
                          >
                            <Trash2 className="h-4 w-4" />
                          </Button>
                        </TooltipTrigger>
                        <TooltipContent>Delete</TooltipContent>
                      </Tooltip>
                    </TooltipProvider>
                  )}
                </div>
              </div>
              
              {/* Selection indicator */}
              {isSelected && (
                <div className="absolute top-3 right-3 bg-primary text-white rounded-full p-1">
                  <Check className="h-4 w-4" />
                </div>
              )}
            </div>
            
            <div className="text-sm text-gray-600 truncate font-medium" title={file.name}>
              {file.name}
            </div>
            <div className="text-xs text-gray-400 mt-1">
              {formatFileSize(file.size)}
            </div>
          </div>
        </CardContent>
      </Card>
    );
  };

  return (
    <Dialog open={isOpen} onOpenChange={setIsOpen}>
      <DialogTrigger asChild>
        {trigger || (
          <Button variant="outline" className={className}>
            <Upload className="h-4 w-4 mr-2" />
            Media Manager
          </Button>
        )}
      </DialogTrigger>
      <DialogContent className="max-w-[95vw] w-full h-[90vh] flex flex-col">
        <DialogHeader className="flex-shrink-0">
          <DialogTitle className="flex items-center gap-2">
            <Upload className="h-5 w-5" />
            Media Manager
          </DialogTitle>
          <DialogDescription>
            Upload, manage, and select media files for your content.
          </DialogDescription>
        </DialogHeader>
          
        <div className="flex flex-col flex-1 min-h-0">
          <Tabs value={activeTab} onValueChange={setActiveTab} className="flex flex-col flex-1 min-h-0">
            <TabsList className="grid w-full grid-cols-2 flex-shrink-0">
              <TabsTrigger value="upload" className="flex items-center gap-2">
                <Upload className="h-4 w-4" />
                Upload Files
              </TabsTrigger>
              <TabsTrigger value="library" className="flex items-center gap-2">
                <ImageIcon className="h-4 w-4" />
                Media Library
              </TabsTrigger>
            </TabsList>

            <TabsContent value="upload" className="flex-1 flex flex-col min-h-0 mt-4">
              <div className="flex-1 overflow-y-auto">
                <div className="space-y-4 p-1">
                  {isUploading && (
                    <div className="bg-blue-50 border border-blue-200 rounded-lg p-4">
                      <div className="flex items-center gap-2 mb-2">
                        <RefreshCw className="h-4 w-4 animate-spin text-blue-600" />
                        <span className="text-sm font-medium text-blue-800">Uploading files...</span>
                      </div>
                      <div className="w-full bg-blue-200 rounded-full h-2">
                        <div 
                          className="bg-blue-600 h-2 rounded-full transition-all duration-300"
                          style={{ width: `${uploadProgress}%` }}
                        />
                      </div>
                      <div className="text-xs text-blue-600 mt-1">{Math.round(uploadProgress)}% complete</div>
                    </div>
                  )}
                  
                  <div className="border-2 border-dashed border-gray-300 rounded-lg p-6 hover:border-primary transition-colors">
                    <FilePond
                      ref={filePondRef}
                      files={files}
                      onupdatefiles={setFiles}
                      allowMultiple={true}
                      maxFiles={maxFiles}
                      acceptedFileTypes={acceptedTypes}
                      server={{
                        process: {
                          url: '/media/upload',
                          method: 'POST',
                          headers: {
                            'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]')?.getAttribute('content') || '',
                          },
                          withCredentials: false,
                          ondata: (formData) => {
                            formData.append('collection', collection);
                            return formData;
                          },
                          onload: (response) => {
                            const data = JSON.parse(response);
                            if (data.success && data.data) {
                              setTimeout(() => {
                                loadMediaFiles();
                              }, 500);
                              
                              if (Array.isArray(data.data)) {
                                return data.data[0]?.id || data.serverId || data.id;
                              } else {
                                return data.data.id || data.serverId || data.id;
                              }
                            }
                            return data.serverId || data.id;
                          },
                          onerror: (response) => {
                            const data = JSON.parse(response);
                            toast.error(data.message || 'Upload failed');
                          }
                        }
                      }}
                      labelIdle='<div class="text-center"><Upload className="h-8 w-8 mx-auto mb-2 text-gray-400" /><div>Drag & Drop your files or <span class="text-primary">Browse</span></div></div>'
                      labelFileProcessing='Uploading'
                      labelFileProcessingComplete='Upload complete'
                      labelFileProcessingAborted='Upload cancelled'
                      labelFileProcessingError='Error during upload'
                      stylePanelLayout="compact"
                      styleLoadIndicatorPosition="center bottom"
                      styleProgressIndicatorPosition="right bottom"
                      styleButtonRemoveItemPosition="left bottom"
                      styleButtonProcessItemPosition="right bottom"
                      imageCropAspectRatio={cropAspectRatio?.toString() || '16:9'}
                      imageResizeTargetWidth={1920}
                      imageResizeTargetHeight={1080}
                      imageResizeMode="contain"
                      imageTransformOutputQuality={85}
                      imageTransformOutputMimeType="image/jpeg"
                      allowImageCrop={allowCrop}
                      allowImageResize={true}
                      allowImageTransform={true}
                      allowFileTypeValidation={true}
                      allowFileSizeValidation={true}
                      maxFileSize="10MB"
                      className="w-full"
                    />
                  </div>
                </div>
              </div>
            </TabsContent>

            <TabsContent value="library" className="flex-1 flex flex-col min-h-0 mt-4">
              <Tabs value={libraryTab} onValueChange={setLibraryTab} className="flex-1 flex flex-col min-h-0">
                <div className="flex items-center justify-between mb-4 flex-shrink-0">
                  <TabsList className="grid w-full grid-cols-2 max-w-md">
                    <TabsTrigger value="images" className="flex items-center gap-2">
                      <ImageIcon className="h-4 w-4" />
                      Images & Videos
                    </TabsTrigger>
                    <TabsTrigger value="documents" className="flex items-center gap-2">
                      <FileText className="h-4 w-4" />
                      Documents & Audio
                    </TabsTrigger>
                  </TabsList>
                  
                  <div className="flex items-center space-x-2">
                    <Button
                      variant={viewMode === 'grid' ? 'default' : 'outline'}
                      size="sm"
                      onClick={() => setViewMode('grid')}
                    >
                      <Grid3X3 className="h-4 w-4" />
                    </Button>
                    <Button
                      variant={viewMode === 'list' ? 'default' : 'outline'}
                      size="sm"
                      onClick={() => setViewMode('list')}
                    >
                      <List className="h-4 w-4" />
                    </Button>
                    
                    <Button
                      variant="outline"
                      size="sm"
                      onClick={loadMediaFiles}
                      disabled={loading}
                    >
                      <RefreshCw className={`h-4 w-4 ${loading ? 'animate-spin' : ''}`} />
                    </Button>
                  </div>
                </div>

                <div className="mb-4 flex items-center gap-4 flex-shrink-0">
                  <div className="relative flex-1 max-w-md">
                    <Search className="absolute left-3 top-1/2 transform -translate-y-1/2 text-gray-400 h-4 w-4" />
                    <Input
                      placeholder={`Search ${libraryTab === 'images' ? 'images and videos' : 'documents and audio'}...`}
                      value={libraryTab === 'images' ? imageSearchQuery : documentSearchQuery}
                      onChange={(e) => debouncedSearch(e.target.value, libraryTab as 'images' | 'documents')}
                      className="pl-10"
                    />
                    {(libraryTab === 'images' ? imageSearchQuery : documentSearchQuery) && (
                      <Button
                        variant="ghost"
                        size="sm"
                        className="absolute right-2 top-1/2 transform -translate-y-1/2 h-6 w-6 p-0"
                        onClick={() => {
                          if (libraryTab === 'images') {
                            setImageSearchQuery('');
                          } else {
                            setDocumentSearchQuery('');
                          }
                        }}
                      >
                        <X className="h-3 w-3" />
                      </Button>
                    )}
                  </div>
                </div>

                <TabsContent value="images" className="flex-1 flex flex-col min-h-0">
                   {loading ? (
                     <div className="flex items-center justify-center h-64">
                       <div className="flex flex-col items-center gap-4">
                         <RefreshCw className="h-8 w-8 animate-spin text-primary" />
                         <div className="text-gray-500">Loading media files...</div>
                       </div>
                     </div>
                   ) : (
                      <>
                        <div className="flex-1 overflow-y-auto">
                          {viewMode === 'grid' ? (
                            <div className="grid grid-cols-2 gap-2 p-0">
                              {getPaginatedFiles('images').files.map((file) => renderFileCard(file, 'images'))}
                            </div>
                          ) : (
                           <div className="p-0">
                             {getPaginatedFiles('images').files.map((file) => {
                              const isSelected = selectedFiles.some(f => f.id === file.id);
                              return (
                                <Card
                                  key={file.id}
                                  className={`cursor-pointer transition-all hover:shadow-md group ${
                                    isSelected ? 'ring-2 ring-primary bg-primary/5' : ''
                                  }`}
                                  onClick={() => handleFileSelect(file)}
                                >
                                  <CardContent className="p-0">
                                    <div className="flex items-center">
                                      <div className="flex-shrink-0">
                                        {file.is_image ? (
                                          <img
                                            src={file.url}
                                            alt={file.alt_text || file.name}
                                            className="w-16 object-cover rounded"
                                            loading="lazy"
                                          />
                                        ) : (
                                          getFileIcon(file)
                                        )}
                                      </div>
                                      <div className="flex-1 min-w-0">
                                        <div className="text-sm font-medium text-gray-900 truncate">
                                          {file.name}
                                        </div>
                                        <div className="text-xs text-gray-500">
                                          {formatFileSize(file.size)} • {file.mime_type}
                                        </div>
                                        {file.collection_name && (
                                          <Badge variant="secondary" className="mt-1">
                                            {file.collection_name}
                                          </Badge>
                                        )}
                                      </div>
                                      <div className="flex items-center opacity-0 group-hover:opacity-100 transition-opacity">
                                        <Button
                                          variant="ghost"
                                          size="sm"
                                          onClick={(e) => {
                                            e.stopPropagation();
                                            handleFileCopy(file);
                                          }}
                                        >
                                          <Copy className="h-4 w-4" />
                                        </Button>
                                        {allowDelete && (
                                          <Button
                                            variant="ghost"
                                            size="sm"
                                            onClick={(e) => {
                                              e.stopPropagation();
                                              handleFileDelete(file.id);
                                            }}
                                          >
                                            <Trash2 className="h-4 w-4 text-red-500" />
                                          </Button>
                                        )}
                                      </div>
                                    </div>
                                  </CardContent>
                                </Card>
                              );
                            })}
                          </div>
                        )}
                        
                        {getPaginatedFiles('images').files.length === 0 && !loading && (
                          <div className="text-center py-12">
                            <ImageIcon className="h-12 w-12 text-gray-300 mx-auto mb-4" />
                            <div className="text-gray-400 mb-2">No images or videos found</div>
                            <div className="text-sm text-gray-500">
                              {imageSearchQuery 
                                ? 'Try adjusting your search'
                                : 'Upload some images or videos to get started'
                              }
                            </div>
                          </div>
                        )}
                      </div>
                      
                      {getPaginatedFiles('images').totalPages > 1 && (
                        <div className="flex items-center justify-between px-4 py-3 border-t bg-gray-50 flex-shrink-0">
                          <div className="text-sm text-gray-700">
                            Showing {((currentPage - 1) * itemsPerPage) + 1} to {Math.min(currentPage * itemsPerPage, getPaginatedFiles('images').total)} of {getPaginatedFiles('images').total} results
                          </div>
                          <div className="flex items-center space-x-2">
                            <Button
                              variant="outline"
                              size="sm"
                              onClick={() => setCurrentPage(prev => Math.max(1, prev - 1))}
                              disabled={currentPage === 1}
                            >
                              <ChevronLeft className="h-4 w-4" />
                              Previous
                            </Button>
                            <span className="text-sm">
                              Page {currentPage} of {getPaginatedFiles('images').totalPages}
                            </span>
                            <Button
                              variant="outline"
                              size="sm"
                              onClick={() => setCurrentPage(prev => Math.min(getPaginatedFiles('images').totalPages, prev + 1))}
                              disabled={currentPage === getPaginatedFiles('images').totalPages}
                            >
                              Next
                              <ChevronRight className="h-4 w-4" />
                            </Button>
                          </div>
                        </div>
                      )}
                    </>
                  )}
                </TabsContent>

                <TabsContent value="documents" className="flex-1 flex flex-col min-h-0">
                   {loading ? (
                     <div className="flex items-center justify-center h-64">
                       <div className="flex flex-col items-center gap-4">
                         <RefreshCw className="h-8 w-8 animate-spin text-primary" />
                         <div className="text-gray-500">Loading media files...</div>
                       </div>
                     </div>
                   ) : (
                      <>
                        <div className="flex-1 overflow-y-auto">
                         {viewMode === 'grid' ? (
                           <div className="grid grid-cols-2 md:grid-cols-3 lg:grid-cols-4 gap-4 p-4">
                             {getPaginatedFiles('documents').files.map((file) => renderFileCard(file, 'documents'))}
                           </div>
                           ) : (
                           <div className="space-y-2 p-4">
                             {getPaginatedFiles('documents').files.map((file) => { 
                              const isSelected = selectedFiles.some(f => f.id === file.id);
                              return (
                                <Card
                                  key={file.id}
                                  className={`cursor-pointer transition-all hover:shadow-md group ${
                                    isSelected ? 'ring-2 ring-primary bg-primary/5' : ''
                                  }`}
                                  onClick={() => handleFileSelect(file)}
                                >
                                  <CardContent className="p-4">
                                    <div className="flex items-center space-x-4">
                                      <div className="flex-shrink-0">
                                        {getFileIcon(file)}
                                      </div>
                                      <div className="flex-1 min-w-0">
                                        <div className="text-sm font-medium text-gray-900 truncate">
                                          {file.name}
                                        </div>
                                        <div className="text-xs text-gray-500">
                                          {formatFileSize(file.size)} • {file.mime_type}
                                        </div>
                                        {file.collection_name && (
                                          <Badge variant="secondary" className="mt-1">
                                            {file.collection_name}
                                          </Badge>
                                        )}
                                      </div>
                                      <div className="flex items-center space-x-2 opacity-0 group-hover:opacity-100 transition-opacity">
                                        <Button
                                          variant="ghost"
                                          size="sm"
                                          onClick={(e) => {
                                            e.stopPropagation();
                                            handleFileCopy(file);
                                          }}
                                        >
                                          <Copy className="h-4 w-4" />
                                        </Button>
                                        {allowDelete && (
                                          <Button
                                            variant="ghost"
                                            size="sm"
                                            onClick={(e) => {
                                              e.stopPropagation();
                                              handleFileDelete(file.id);
                                            }}
                                          >
                                            <Trash2 className="h-4 w-4 text-red-500" />
                                          </Button>
                                        )}
                                      </div>
                                    </div>
                                  </CardContent>
                                </Card>
                              );
                            })}
                          </div>
                        )}
                        
                        {getPaginatedFiles('documents').files.length === 0 && !loading && (
                          <div className="text-center py-12">
                            <FileText className="h-12 w-12 text-gray-300 mx-auto mb-4" />
                            <div className="text-gray-400 mb-2">No documents or audio files found</div>
                            <div className="text-sm text-gray-500">
                              {documentSearchQuery 
                                ? 'Try adjusting your search'
                                : 'Upload some documents or audio files to get started'
                              }
                            </div>
                          </div>
                        )}
                      </div>
                      
                      {getPaginatedFiles('documents').totalPages > 1 && (
                        <div className="flex items-center justify-between px-4 py-3 border-t bg-gray-50 flex-shrink-0">
                          <div className="text-sm text-gray-700">
                            Showing {((currentPage - 1) * itemsPerPage) + 1} to {Math.min(currentPage * itemsPerPage, getPaginatedFiles('documents').total)} of {getPaginatedFiles('documents').total} results
                          </div>
                          <div className="flex items-center space-x-2">
                            <Button
                              variant="outline"
                              size="sm"
                              onClick={() => setCurrentPage(prev => Math.max(1, prev - 1))}
                              disabled={currentPage === 1}
                            >
                              <ChevronLeft className="h-4 w-4" />
                              Previous
                            </Button>
                            <span className="text-sm">
                              Page {currentPage} of {getPaginatedFiles('documents').totalPages}
                            </span>
                            <Button
                              variant="outline"
                              size="sm"
                              onClick={() => setCurrentPage(prev => Math.min(getPaginatedFiles('documents').totalPages, prev + 1))}
                              disabled={currentPage === getPaginatedFiles('documents').totalPages}
                            >
                              Next
                              <ChevronRight className="h-4 w-4" />
                            </Button>
                          </div>
                        </div>
                      )}
                    </>
                  )}
                </TabsContent>
              </Tabs>
            </TabsContent>
          </Tabs>

          {selectedFiles.length > 0 && (
            <div className="border-t pt-4 mt-4 flex-shrink-0">
              <div className="flex items-center justify-between">
                <div className="text-sm text-gray-600">
                  {multiple 
                    ? `${selectedFiles.length} file(s) selected`
                    : '1 file selected'
                  }
                </div>
                <div className="flex items-center space-x-2">
                  <Button
                    variant="outline"
                    onClick={() => setSelectedFiles([])}
                  >
                    Clear Selection
                  </Button>
                  <Button onClick={handleConfirmSelection}>
                    {multiple ? 'Select Files' : 'Select File'}
                  </Button>
                </div>
              </div>
            </div>
          )}
        </div>
      </DialogContent>
    </Dialog>
  );
};

export default MediaManager; 