import React, { useState } from 'react';
import { Button } from '@/components/ui/button';
import { AlertTriangle, LogOut, User, Shield, X, Clock, Eye, UserX } from 'lucide-react';
import { router, usePage } from '@inertiajs/react';

interface ImpersonationBannerProps {
  impersonatedUser: {
    id: number;
    name: string;
    email: string;
  };
  originalUser: {
    id: number;
    name: string;
    email: string;
  };
}

const ImpersonationBanner: React.FC<ImpersonationBannerProps> = ({ 
  impersonatedUser, 
  originalUser 
}) => {
  const [isExiting, setIsExiting] = useState(false);
  const [startTime] = useState(new Date());

  const handleExitImpersonation = () => {
    setIsExiting(true);
    router.post(route('admin.users.stopImpersonating'), {}, {
      onSuccess: () => {
      },
      onError: (errors) => {
        setIsExiting(false);
      }
    });
  };

  const formatTime = (date: Date) => {
    return date.toLocaleTimeString('en-US', { 
      hour12: false, 
      hour: '2-digit', 
      minute: '2-digit' 
    });
  };

  return (
    <div className="fixed top-0 left-0 right-0 z-[9999] bg-gradient-to-r from-red-600 to-red-700 shadow-lg border-b-2 border-red-800">
      <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
        <div className="flex items-center justify-between py-3">
          {/* Warning Icon and Message */}
          <div className="flex items-center space-x-3">
            <div className="flex items-center justify-center w-8 h-8 bg-white/20 rounded-full backdrop-blur-sm">
              <AlertTriangle className="w-5 h-5 text-white animate-pulse" />
            </div>
            
            <div className="flex flex-col sm:flex-row sm:items-center sm:space-x-2">
              <div className="flex items-center space-x-2">
                <Shield className="w-4 h-4 text-yellow-300" />
                <span className="text-white font-semibold text-sm">
                  IMPERSONATION MODE
                </span>
              </div>
              
              <div className="hidden sm:block text-white/80 text-sm">•</div>
              
              <div className="flex items-center space-x-2 text-white/90 text-sm">
                <User className="w-4 h-4" />
                <span>
                  Viewing as <strong className="text-white">{impersonatedUser.name}</strong>
                </span>
                <span className="hidden md:inline text-white/70">
                  ({impersonatedUser.email})
                </span>
              </div>
            </div>
          </div>

          {/* User Info and Exit Button */}
          <div className="flex items-center space-x-3">
            {/* Session Info (Desktop) */}
            <div className="hidden xl:flex items-center space-x-4 text-white/70 text-xs">
              <div className="flex items-center space-x-1">
                <Clock className="w-3 h-3" />
                <span>Started: {formatTime(startTime)}</span>
              </div>
              <div className="flex items-center space-x-1">
                <Eye className="w-3 h-3" />
                <span>Admin: {originalUser.name}</span>
              </div>
            </div>

            {/* Original User Info (Tablet) */}
            <div className="hidden lg:flex xl:hidden items-center space-x-2 text-white/80 text-sm">
              <span>Admin:</span>
              <span className="text-white font-medium">{originalUser.name}</span>
            </div>

            {/* Exit Button */}
            <Button
              onClick={handleExitImpersonation}
              disabled={isExiting}
              variant="secondary"
              size="sm"
              className="bg-white/20 hover:bg-white/30 text-white border-white/30 hover:border-white/50 backdrop-blur-sm transition-all duration-200 font-medium disabled:opacity-50 disabled:cursor-not-allowed"
            >
              {isExiting ? (
                <>
                  <div className="w-4 h-4 mr-2 animate-spin rounded-full border-2 border-white/30 border-t-white"></div>
                  <span className="hidden sm:inline">Exiting...</span>
                  <span className="sm:hidden">...</span>
                </>
              ) : (
                <>
                  <UserX className="w-4 h-4 mr-2" />
                  <span className="hidden sm:inline">Exit Impersonation</span>
                  <span className="sm:hidden">Exit</span>
                </>
              )}
            </Button>
          </div>
        </div>
      </div>

      {/* Subtle animation line */}
      <div className="absolute bottom-0 left-0 right-0 h-0.5 bg-gradient-to-r from-transparent via-yellow-400 to-transparent opacity-60">
        <div className="h-full bg-gradient-to-r from-yellow-400 to-orange-400 animate-pulse"></div>
      </div>
    </div>
  );
};

export default ImpersonationBanner;