import React from 'react';
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from '@/components/ui/card';
import { Alert, AlertDescription } from '@/components/ui/alert';
import { Badge } from '@/components/ui/badge';
import { Info, AlertTriangle, CheckCircle } from 'lucide-react';

interface ConditionalSettingsProps {
  condition: boolean;
  title: string;
  description: string;
  children: React.ReactNode;
  type?: 'info' | 'warning' | 'success';
  actionRequired?: boolean;
  actionText?: string;
}

const ConditionalSettings: React.FC<ConditionalSettingsProps> = ({
  condition,
  title,
  description,
  children,
  type = 'info',
  actionRequired = false,
  actionText
}) => {
  if (!condition) {
    return null;
  }

  const getIcon = () => {
    switch (type) {
      case 'warning':
        return <AlertTriangle className="h-4 w-4" />;
      case 'success':
        return <CheckCircle className="h-4 w-4" />;
      default:
        return <Info className="h-4 w-4" />;
    }
  };

  const getAlertVariant = () => {
    switch (type) {
      case 'warning':
        return 'destructive';
      case 'success':
        return 'default';
      default:
        return 'secondary';
    }
  };

  return (
    <Card>
      <CardHeader>
        <CardTitle className="flex items-center gap-2">
          {getIcon()}
          {title}
          {actionRequired && (
            <Badge variant="outline" className="text-xs">
              Action Required
            </Badge>
          )}
        </CardTitle>
        <CardDescription>
          {description}
        </CardDescription>
      </CardHeader>
      <CardContent className="space-y-4">
        {actionRequired && actionText && (
          <Alert variant={getAlertVariant()}>
            <AlertTriangle className="h-4 w-4" />
            <AlertDescription>
              {actionText}
            </AlertDescription>
          </Alert>
        )}
        {children}
      </CardContent>
    </Card>
  );
};

export default ConditionalSettings;