import React, { useState } from 'react';
import { Dialog, DialogContent, DialogHeader, DialogTitle, DialogFooter } from '@/components/ui/dialog';
import { Button } from '@/components/ui/button';
import { Badge } from '@/components/ui/badge';
import { Download, Printer, Eye, X, ExternalLink } from 'lucide-react';
import { toast } from 'sonner';

interface CertificateData {
    player_name: string;
    team_name?: string;
    position?: string;
    jersey_number?: string;
    tournament_matches_played?: number;
    tournament_win_rate?: number;
    global_rating?: number;
    digital_id: string;
    issued_date: string;
    tournament_name: string;
    description?: string;
    verification_status?: string;
}

interface CertificatePreviewDialogProps {
    isOpen: boolean;
    onClose: () => void;
    certificateData: CertificateData;
    playerId: number;
}

const CertificatePreviewDialog: React.FC<CertificatePreviewDialogProps> = ({
    isOpen,
    onClose,
    certificateData,
    playerId
}) => {
    const [isLoading, setIsLoading] = useState(false);

    const handleDownload = () => {
        // Use direct link approach for PDF download
        const url = `/admin/players/${playerId}/certificate/download`;
        const link = document.createElement('a');
        link.href = url;
        link.download = `Certificate_${certificateData.player_name.replace(/[^a-zA-Z0-9]/g, '_')}_${certificateData.issued_date.replace(/[^a-zA-Z0-9]/g, '_')}.pdf`;
        link.target = '_blank';
        document.body.appendChild(link);
        link.click();
        document.body.removeChild(link);
        toast.success('Certificate download started!');
    };

    const handlePrint = async () => {
        setIsLoading(true);
        try {
            // Open certificate in new window for printing
            const url = `/admin/players/${playerId}/certificate/print`;
            const printWindow = window.open(url, '_blank', 'width=800,height=600');
            
            if (printWindow) {
                printWindow.onload = () => {
                    printWindow.print();
                    printWindow.onafterprint = () => {
                        printWindow.close();
                    };
                };
            }
            
            toast.success('Print dialog opened!');
        } catch (error) {
            console.error('Print error:', error);
            toast.error('Failed to open print dialog');
        } finally {
            setIsLoading(false);
        }
    };

    const handleViewFull = () => {
        const url = `/admin/players/${playerId}/certificate`;
        window.open(url, '_blank');
    };

    const getStatusColor = (status?: string) => {
        switch (status?.toLowerCase()) {
            case 'verified':
                return 'bg-green-100 text-green-800';
            case 'pending':
                return 'bg-yellow-100 text-yellow-800';
            case 'rejected':
                return 'bg-red-100 text-red-800';
            default:
                return 'bg-gray-100 text-gray-800';
        }
    };

    return (
        <Dialog open={isOpen} onOpenChange={onClose}>
            <DialogContent className="min-w-150 max-h-[90vh] overflow-y-auto">
                <DialogHeader>
                    <DialogTitle className="flex items-center gap-2">
                        <Eye className="h-5 w-5" />
                        Certificate Preview
                    </DialogTitle>
                </DialogHeader>

                <div className="space-y-6">
                    {/* Certificate Preview */}
                    <div className="border rounded-lg overflow-hidden bg-white shadow-sm">
                        <div className="p-4 bg-gray-50 border-b">
                            <div className="flex items-center justify-between">
                                <h3 className="font-semibold text-gray-900">Certificate Preview</h3>
                                <Badge className={getStatusColor(certificateData.verification_status)}>
                                    {certificateData.verification_status || 'Verified'}
                                </Badge>
                            </div>
                        </div>
                        
                        {/* Certificate iframe */}
                        <div className="relative">
                            <iframe
                                src={`/admin/players/${playerId}/certificate`}
                                className="w-full h-96 border-0"
                                title="Certificate Preview"
                                sandbox="allow-same-origin allow-scripts"
                            />
                            
                            {/* Overlay for better UX */}
                            <div className="absolute inset-0 pointer-events-none">
                                <div className="absolute top-4 right-4">
                                    <Button
                                        size="sm"
                                        variant="outline"
                                        onClick={handleViewFull}
                                        className="pointer-events-auto bg-white/90 hover:bg-white"
                                    >
                                        <ExternalLink className="h-4 w-4 mr-1" />
                                        View Full
                                    </Button>
                                </div>
                            </div>
                        </div>
                    </div>

                    {/* Certificate Details */}
                    {/* <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
                        <div className="space-y-4">
                            <h4 className="font-semibold text-gray-900">Player Information</h4>
                            <div className="space-y-2">
                                <div className="flex justify-between">
                                    <span className="text-gray-600">Name:</span>
                                    <span className="font-medium">{certificateData.player_name}</span>
                                </div>
                                {certificateData.team_name && (
                                    <div className="flex justify-between">
                                        <span className="text-gray-600">Team:</span>
                                        <span className="font-medium">{certificateData.team_name}</span>
                                    </div>
                                )}
                                {certificateData.position && (
                                    <div className="flex justify-between">
                                        <span className="text-gray-600">Position:</span>
                                        <span className="font-medium">{certificateData.position}</span>
                                    </div>
                                )}
                                {certificateData.jersey_number && (
                                    <div className="flex justify-between">
                                        <span className="text-gray-600">Jersey #:</span>
                                        <span className="font-medium">#{certificateData.jersey_number}</span>
                                    </div>
                                )}
                            </div>
                        </div>

                        <div className="space-y-4">
                            <h4 className="font-semibold text-gray-900">Tournament Stats</h4>
                            <div className="space-y-2">
                                <div className="flex justify-between">
                                    <span className="text-gray-600">Tournament:</span>
                                    <span className="font-medium">{certificateData.tournament_name}</span>
                                </div>
                                {certificateData.tournament_matches_played !== undefined && (
                                    <div className="flex justify-between">
                                        <span className="text-gray-600">Matches Played:</span>
                                        <span className="font-medium">{certificateData.tournament_matches_played}</span>
                                    </div>
                                )}
                                {certificateData.tournament_win_rate !== undefined && (
                                    <div className="flex justify-between">
                                        <span className="text-gray-600">Win Rate:</span>
                                        <span className="font-medium">{certificateData.tournament_win_rate}%</span>
                                    </div>
                                )}
                                {certificateData.global_rating !== undefined && (
                                    <div className="flex justify-between">
                                        <span className="text-gray-600">Global Rating:</span>
                                        <span className="font-medium">{certificateData.global_rating}</span>
                                    </div>
                                )}
                            </div>
                        </div>
                    </div> */}

                    {/* Certificate Details */}
                    {/* <div className="space-y-4">
                        <h4 className="font-semibold text-gray-900">Certificate Details</h4>
                        <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                            <div className="flex justify-between">
                                <span className="text-gray-600">Certificate ID:</span>
                                <span className="font-mono text-sm">{certificateData.digital_id}</span>
                            </div>
                            <div className="flex justify-between">
                                <span className="text-gray-600">Issued Date:</span>
                                <span className="font-medium">{certificateData.issued_date}</span>
                            </div>
                        </div>
                        
                        {certificateData.description && (
                            <div className="mt-4 p-3 bg-gray-50 rounded-lg">
                                <p className="text-sm text-gray-700 italic">
                                    {certificateData.description}
                                </p>
                            </div>
                        )}
                    </div> */}
                </div>

                <DialogFooter className="flex gap-2">
                    <Button
                        variant="outline"
                        onClick={onClose}
                        disabled={isLoading}
                    >
                        <X className="h-4 w-4 mr-2" />
                        Close
                    </Button>
                    
                    <Button
                        variant="outline"
                        onClick={handlePrint}
                        disabled={isLoading}
                    >
                        <Printer className="h-4 w-4 mr-2" />
                        {isLoading ? 'Loading...' : 'Print'}
                    </Button>
                    
                    <Button
                        onClick={handleDownload}
                        disabled={isLoading}
                        className="bg-blue-600 hover:bg-blue-700"
                    >
                        <Download className="h-4 w-4 mr-2" />
                        {isLoading ? 'Downloading...' : 'Download PDF'}
                    </Button>
                </DialogFooter>
            </DialogContent>
        </Dialog>
    );
};

export default CertificatePreviewDialog;
