import React from 'react';
import { Button } from '@/components/ui/button';
import { Eye, Award } from 'lucide-react';
import { toast } from 'sonner';

interface CertificatePreviewButtonProps {
    playerId: number;
    playerName: string;
    teamName?: string;
    position?: string;
    jerseyNumber?: string;
    tournamentMatchesPlayed?: number;
    tournamentWinRate?: number;
    globalRating?: number;
    verificationStatus?: string;
    tournamentName?: string;
    variant?: 'default' | 'outline' | 'ghost' | 'link';
    size?: 'default' | 'sm' | 'lg';
    showText?: boolean;
}

const CertificatePreviewButton: React.FC<CertificatePreviewButtonProps> = ({
    playerId,
    playerName,
    teamName,
    position,
    jerseyNumber,
    tournamentMatchesPlayed,
    tournamentWinRate,
    globalRating,
    verificationStatus,
    tournamentName,
    variant = 'outline',
    size = 'sm',
    showText = true,
}) => {
    const handlePreview = () => {
        // Generate certificate data
        const certificateId = `CERT-${String(playerId).padStart(6, '0')}-${Math.random().toString(36).substr(2, 8).toUpperCase()}`;
        const issuedDate = new Date().toLocaleDateString('en-US', { 
            year: 'numeric', 
            month: 'long', 
            day: 'numeric' 
        });

        const certificateData = {
            player_name: playerName,
            team_name: teamName || '',
            position: position || 'Participant',
            jersey_number: jerseyNumber || undefined,
            tournament_matches_played: tournamentMatchesPlayed || 0,
            tournament_win_rate: tournamentWinRate || 0,
            global_rating: globalRating || 0,
            digital_id: certificateId,
            issued_date: issuedDate,
            tournament_name: tournamentName || 'The Milimus Tournament Cup',
            verification_status: verificationStatus || 'verified',
        };

        // Store data in sessionStorage for the dialog to pick up
        sessionStorage.setItem('certificatePreviewData', JSON.stringify({
            playerId,
            certificateData
        }));

        // Trigger custom event to open dialog
        window.dispatchEvent(new CustomEvent('openCertificatePreview', {
            detail: { playerId, certificateData }
        }));

        toast.success('Opening certificate preview...');
    };

    return (
        <Button
            variant={variant}
            size={size}
            onClick={handlePreview}
            className="flex items-center gap-1"
        >
            <Eye className="h-4 w-4" />
            {showText && 'Preview Certificate'}
        </Button>
    );
};

// Alternative button for direct certificate access
export const CertificateDownloadButton: React.FC<CertificatePreviewButtonProps> = ({
    playerId,
    playerName,
    variant = 'default',
    size = 'sm',
    showText = true,
}) => {
    const handleDownload = async () => {
        try {
            // Fetch the PDF file
            const url = `/admin/players/${playerId}/certificate/print`;
            const response = await fetch(url, {
                method: 'GET',
                headers: {
                    'Accept': 'application/pdf',
                },
            });

            if (!response.ok) {
                throw new Error(`HTTP error! status: ${response.status}`);
            }

            // Get the PDF blob
            const blob = await response.blob();
            
            // Create download link
            const downloadUrl = window.URL.createObjectURL(blob);
            const link = document.createElement('a');
            link.href = downloadUrl;
            
            // Generate filename
            const cleanPlayerName = playerName.replace(/[^a-zA-Z0-9]/g, '_');
            const dateStr = new Date().toISOString().split('T')[0];
            link.download = `Certificate_${cleanPlayerName}_${dateStr}.pdf`;
            
            // Trigger download
            document.body.appendChild(link);
            link.click();
            document.body.removeChild(link);
            
            // Clean up the blob URL
            window.URL.revokeObjectURL(downloadUrl);
            
            toast.success('Certificate downloaded successfully!');
        } catch (error) {
            console.error('Download error:', error);
            toast.error('Failed to download certificate. Please try again.');
        }
    };

    return (
        <Button
            variant={variant}
            size={size}
            onClick={handleDownload}
            className="flex items-center gap-1 bg-green-600 hover:bg-green-700"
        >
            <Award className="h-4 w-4" />
            {showText && 'Download PDF'}
        </Button>
    );
};

export default CertificatePreviewButton;
