import React from 'react';
import { Card, CardContent } from '@/components/ui/card';
import { RadioGroup, RadioGroupItem } from '@/components/ui/radio-group';
import { Label } from '@/components/ui/label';
import { Alert, AlertDescription } from '@/components/ui/alert';
import { Info, CheckCircle2 } from 'lucide-react';
import { cn } from '@/lib/utils';

interface Category {
    key: string;
    name: string;
    age_min: number | null;
    age_max: number | null;
    gender: string | null;
    gender_required: boolean;
    fee: number;
    icon: string;
    color: string;
    description: string;
}

interface CategorySelectorProps {
    categories: Record<string, Category>;
    selectedCategory: string;
    onCategoryChange: (category: string) => void;
    currency?: string;
    disabled?: boolean;
}

export default function CategorySelector({
    categories,
    selectedCategory,
    onCategoryChange,
    currency = 'KES',
    disabled = false
}: CategorySelectorProps) {
    const colorClasses: Record<string, string> = {
        blue: 'border-blue-500 bg-blue-50',
        pink: 'border-pink-500 bg-pink-50',
        purple: 'border-purple-500 bg-purple-50',
        green: 'border-green-500 bg-green-50',
    };

    return (
        <div className="space-y-4">
            <Alert>
                <Info className="h-4 w-4" />
                <AlertDescription>
                    Select the category that matches your team. All players must meet the age and gender requirements for the selected category.
                </AlertDescription>
            </Alert>

            <RadioGroup value={selectedCategory} onValueChange={onCategoryChange} disabled={disabled}>
                <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                    {Object.entries(categories).map(([key, category]) => {
                        const isSelected = selectedCategory === key;
                        const borderColor = colorClasses[category.color] || 'border-gray-300 bg-gray-50';

                        return (
                            <label htmlFor={`category-${key}`} key={key}>
                                <Card
                                    className={cn(
                                        'cursor-pointer transition-all',
                                        isSelected ? `border-2 ${borderColor} ring-2 ring-offset-2 ring-${category.color}-500` : 'border hover:border-gray-400',
                                        disabled && 'opacity-50 cursor-not-allowed'
                                    )}
                                >
                                    <CardContent className="p-4">
                                        <div className="flex items-start gap-3">
                                            <RadioGroupItem
                                                value={key}
                                                id={`category-${key}`}
                                                className="mt-1"
                                                disabled={disabled}
                                            />
                                            <div className="flex-1">
                                                <div className="flex items-center gap-2 cursor-pointer font-semibold text-base">
                                                    <span className="text-2xl">{category.icon}</span>
                                                    <span>{category.name}</span>
                                                    {isSelected && <CheckCircle2 className="w-4 h-4 text-green-600 ml-auto" />}
                                                </div>

                                            <p className="text-sm text-gray-600 mt-1 mb-2">
                                                {category.description}
                                            </p>

                                            <div className="space-y-1 text-xs text-gray-600">
                                                {category.age_min !== null && category.age_max !== null && (
                                                    <div className="flex items-center gap-1">
                                                        <span className="font-semibold">Age:</span>
                                                        <span>{category.age_min} - {category.age_max} years</span>
                                                    </div>
                                                )}
                                                {category.age_min !== null && category.age_max === null && (
                                                    <div className="flex items-center gap-1">
                                                        <span className="font-semibold">Age:</span>
                                                        <span>{category.age_min}+ years</span>
                                                    </div>
                                                )}
                                                {category.age_min === null && category.age_max !== null && (
                                                    <div className="flex items-center gap-1">
                                                        <span className="font-semibold">Age:</span>
                                                        <span>Under {category.age_max + 1} years</span>
                                                    </div>
                                                )}
                                                {category.gender_required && category.gender && (
                                                    <div className="flex items-center gap-1">
                                                        <span className="font-semibold">Gender:</span>
                                                        <span className="capitalize">{category.gender}</span>
                                                    </div>
                                                )}
                                                {!category.gender_required && (
                                                    <div className="flex items-center gap-1">
                                                        <span className="font-semibold">Gender:</span>
                                                        <span>Any</span>
                                                    </div>
                                                )}
                                            </div>

                                            <div className="mt-3 pt-3 border-t border-gray-200">
                                                <div className="flex items-center justify-between">
                                                    <span className="text-sm font-semibold text-gray-700">Entry Fee:</span>
                                                    <span className="text-lg font-bold text-gray-900">
                                                        {currency} {category.fee.toLocaleString()}
                                                    </span>
                                                </div>
                                            </div>
                                        </div>
                                    </div>
                                </CardContent>
                            </Card>
                            </label>
                        );
                    })}
                </div>
            </RadioGroup>

            {selectedCategory && (
                <Alert className="bg-green-50 border-green-200">
                    <CheckCircle2 className="h-4 w-4 text-green-600" />
                    <AlertDescription className="text-green-800">
                        <strong>Selected:</strong> {categories[selectedCategory]?.name} - Entry fee will be {currency} {categories[selectedCategory]?.fee.toLocaleString()}
                    </AlertDescription>
                </Alert>
            )}
        </div>
    );
}

