# User Interface

## 7.1 Design System

### Color Palette
The system uses a modern, professional color scheme optimized for accessibility and user experience.

```css
/* Primary Colors */
--primary-50: #eff6ff;
--primary-100: #dbeafe;
--primary-500: #3b82f6;
--primary-600: #2563eb;
--primary-700: #1d4ed8;

/* Secondary Colors */
--secondary-50: #f8fafc;
--secondary-100: #f1f5f9;
--secondary-500: #64748b;
--secondary-600: #475569;

/* Success Colors */
--success-50: #f0fdf4;
--success-500: #22c55e;
--success-600: #16a34a;

/* Warning Colors */
--warning-50: #fffbeb;
--warning-500: #f59e0b;
--warning-600: #d97706;

/* Error Colors */
--error-50: #fef2f2;
--error-500: #ef4444;
--error-600: #dc2626;
```

### Typography
```css
/* Font Families */
--font-sans: 'Inter', system-ui, -apple-system, sans-serif;
--font-mono: 'JetBrains Mono', 'Fira Code', monospace;

/* Font Sizes */
--text-xs: 0.75rem;    /* 12px */
--text-sm: 0.875rem;   /* 14px */
--text-base: 1rem;     /* 16px */
--text-lg: 1.125rem;    /* 18px */
--text-xl: 1.25rem;     /* 20px */
--text-2xl: 1.5rem;     /* 24px */
--text-3xl: 1.875rem;   /* 30px */
--text-4xl: 2.25rem;    /* 36px */

/* Font Weights */
--font-normal: 400;
--font-medium: 500;
--font-semibold: 600;
--font-bold: 700;
```

### Spacing System
```css
/* Spacing Scale */
--space-1: 0.25rem;   /* 4px */
--space-2: 0.5rem;    /* 8px */
--space-3: 0.75rem;   /* 12px */
--space-4: 1rem;      /* 16px */
--space-5: 1.25rem;   /* 20px */
--space-6: 1.5rem;    /* 24px */
--space-8: 2rem;      /* 32px */
--space-10: 2.5rem;    /* 40px */
--space-12: 3rem;      /* 48px */
--space-16: 4rem;      /* 64px */
--space-20: 5rem;      /* 80px */
```

### Component Library
The system uses a comprehensive component library built with Tailwind CSS and Radix UI.

```typescript
// Button Component Variants
interface ButtonProps {
  variant: 'primary' | 'secondary' | 'outline' | 'ghost' | 'danger';
  size: 'sm' | 'md' | 'lg';
  loading?: boolean;
  disabled?: boolean;
  children: React.ReactNode;
}

// Input Component Variants
interface InputProps {
  variant: 'default' | 'filled' | 'outline';
  size: 'sm' | 'md' | 'lg';
  state: 'default' | 'error' | 'success' | 'warning';
}

// Card Component Variants
interface CardProps {
  variant: 'default' | 'elevated' | 'outlined';
  padding: 'none' | 'sm' | 'md' | 'lg';
}
```

## 7.2 User Roles & Dashboards

### Super Admin Dashboard
```typescript
const SuperAdminDashboard: React.FC = () => {
  return (
    <AdminLayout title="Super Admin Dashboard">
      <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6 mb-8">
        <StatCard
          title="Total Users"
          value={totalUsers}
          change={{ value: 12, type: 'increase' }}
          icon={<UsersIcon />}
        />
        <StatCard
          title="Active Tournaments"
          value={activeTournaments}
          change={{ value: 5, type: 'increase' }}
          icon={<TrophyIcon />}
        />
        <StatCard
          title="Total Teams"
          value={totalTeams}
          change={{ value: 8, type: 'increase' }}
          icon={<TeamIcon />}
        />
        <StatCard
          title="System Health"
          value="99.9%"
          change={{ value: 0.1, type: 'increase' }}
          icon={<HeartIcon />}
        />
      </div>

      <div className="grid grid-cols-1 lg:grid-cols-2 gap-6">
        <Card title="Recent Activity">
          <ActivityFeed activities={recentActivities} />
        </Card>
        
        <Card title="System Alerts">
          <AlertList alerts={systemAlerts} />
        </Card>
      </div>
    </AdminLayout>
  );
};
```

### Tournament Organizer Dashboard
```typescript
const OrganizerDashboard: React.FC = () => {
  return (
    <AdminLayout title="Tournament Organizer Dashboard">
      <div className="mb-6">
        <div className="flex justify-between items-center">
          <h1 className="text-2xl font-bold text-gray-900">My Tournaments</h1>
          <Button variant="primary" onClick={() => router.push('/admin/tournaments/create')}>
            Create Tournament
          </Button>
        </div>
      </div>

      <div className="grid grid-cols-1 lg:grid-cols-3 gap-6 mb-8">
        <StatCard
          title="My Tournaments"
          value={myTournaments.length}
          icon={<TrophyIcon />}
        />
        <StatCard
          title="Total Participants"
          value={totalParticipants}
          icon={<UsersIcon />}
        />
        <StatCard
          title="Active Matches"
          value={activeMatches}
          icon={<PlayIcon />}
        />
      </div>

      <Card title="Tournament Management">
        <DataTable
          data={myTournaments}
          columns={tournamentColumns}
          onRowClick={(tournament) => router.push(`/admin/tournaments/${tournament.id}`)}
        />
      </Card>
    </AdminLayout>
  );
};
```

### Team Manager Dashboard
```typescript
const TeamManagerDashboard: React.FC = () => {
  return (
    <TeamManagerLayout team={team}>
      <div className="mb-6">
        <h1 className="text-2xl font-bold text-gray-900">Team Dashboard</h1>
        <p className="text-gray-600">Manage your team and tournament participation</p>
      </div>

      <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6 mb-8">
        <StatCard
          title="Team Players"
          value={team.players.length}
          icon={<UsersIcon />}
        />
        <StatCard
          title="Active Tournaments"
          value={activeTournaments}
          icon={<TrophyIcon />}
        />
        <StatCard
          title="Matches Played"
          value={matchesPlayed}
          icon={<PlayIcon />}
        />
        <StatCard
          title="Team Rating"
          value={teamRating}
          icon={<StarIcon />}
        />
      </div>

      <div className="grid grid-cols-1 lg:grid-cols-2 gap-6">
        <Card title="Recent Matches">
          <MatchList matches={recentMatches} />
        </Card>
        
        <Card title="Team Statistics">
          <TeamStats stats={teamStats} />
        </Card>
      </div>
    </TeamManagerLayout>
  );
};
```

### Player Dashboard
```typescript
const PlayerDashboard: React.FC = () => {
  return (
    <MainLayout title="Player Dashboard">
      <div className="mb-6">
        <h1 className="text-2xl font-bold text-gray-900">Welcome, {player.name}</h1>
        <p className="text-gray-600">Track your performance and tournament participation</p>
      </div>

      <div className="grid grid-cols-1 md:grid-cols-3 gap-6 mb-8">
        <StatCard
          title="Tournaments Played"
          value={tournamentsPlayed}
          icon={<TrophyIcon />}
        />
        <StatCard
          title="Goals Scored"
          value={goalsScored}
          icon={<TargetIcon />}
        />
        <StatCard
          title="Team Position"
          value={teamPosition}
          icon={<RankingIcon />}
        />
      </div>

      <div className="grid grid-cols-1 lg:grid-cols-2 gap-6">
        <Card title="Performance Statistics">
          <PlayerStats stats={playerStats} />
        </Card>
        
        <Card title="Upcoming Matches">
          <MatchList matches={upcomingMatches} />
        </Card>
      </div>
    </MainLayout>
  );
};
```

## 7.3 Tournament Management UI

### Tournament Creation Form
```typescript
const TournamentCreationForm: React.FC = () => {
  const [step, setStep] = useState(1);
  const [formData, setFormData] = useState<TournamentFormData>({
    name: '',
    description: '',
    type: 'knockout',
    start_date: '',
    end_date: '',
    max_participants: 16,
    entry_fee: 0
  });

  const steps = [
    { id: 1, name: 'Basic Information', description: 'Tournament details' },
    { id: 2, name: 'Settings', description: 'Rules and configuration' },
    { id: 3, name: 'Review', description: 'Review and publish' }
  ];

  return (
    <AdminLayout title="Create Tournament">
      <div className="max-w-4xl mx-auto">
        <div className="mb-8">
          <h1 className="text-3xl font-bold text-gray-900">Create Tournament</h1>
          <p className="text-gray-600">Set up a new tournament with all the necessary details</p>
        </div>

        <div className="mb-8">
          <nav aria-label="Progress">
            <ol className="flex items-center">
              {steps.map((stepItem, stepIdx) => (
                <li key={stepItem.id} className={`${stepIdx !== steps.length - 1 ? 'pr-8 sm:pr-20' : ''} relative`}>
                  <div className="flex items-center">
                    <div className={`flex h-10 w-10 items-center justify-center rounded-full ${
                      step >= stepItem.id 
                        ? 'bg-indigo-600 text-white' 
                        : 'bg-gray-200 text-gray-500'
                    }`}>
                      {step > stepItem.id ? (
                        <CheckIcon className="h-6 w-6" />
                      ) : (
                        <span className="text-sm font-medium">{stepItem.id}</span>
                      )}
                    </div>
                    <div className="ml-4 min-w-0">
                      <p className={`text-sm font-medium ${
                        step >= stepItem.id ? 'text-indigo-600' : 'text-gray-500'
                      }`}>
                        {stepItem.name}
                      </p>
                      <p className="text-sm text-gray-500">{stepItem.description}</p>
                    </div>
                  </div>
                </li>
              ))}
            </ol>
          </nav>
        </div>

        <Card>
          {step === 1 && <BasicInformationStep data={formData} onChange={setFormData} />}
          {step === 2 && <SettingsStep data={formData} onChange={setFormData} />}
          {step === 3 && <ReviewStep data={formData} onSubmit={handleSubmit} />}
        </Card>
      </div>
    </AdminLayout>
  );
};
```

### Tournament List View
```typescript
const TournamentListView: React.FC = () => {
  const [tournaments, setTournaments] = useState<Tournament[]>([]);
  const [filters, setFilters] = useState<TournamentFilters>({
    status: 'all',
    type: 'all',
    search: ''
  });

  const columns: Column<Tournament>[] = [
    {
      key: 'name',
      label: 'Tournament',
      sortable: true,
      render: (value, tournament) => (
        <div className="flex items-center">
          <div className="flex-shrink-0 h-10 w-10">
            <img
              className="h-10 w-10 rounded-full"
              src={tournament.logo || '/default-tournament.png'}
              alt={tournament.name}
            />
          </div>
          <div className="ml-4">
            <div className="text-sm font-medium text-gray-900">{tournament.name}</div>
            <div className="text-sm text-gray-500">{tournament.description}</div>
          </div>
        </div>
      )
    },
    {
      key: 'status',
      label: 'Status',
      sortable: true,
      render: (value) => (
        <StatusBadge status={value} />
      )
    },
    {
      key: 'start_date',
      label: 'Start Date',
      sortable: true,
      render: (value) => formatDate(value)
    },
    {
      key: 'participants',
      label: 'Participants',
      render: (value, tournament) => (
        <span>{tournament.current_participants} / {tournament.max_participants}</span>
      )
    },
    {
      key: 'actions',
      label: 'Actions',
      render: (value, tournament) => (
        <div className="flex space-x-2">
          <Button
            variant="outline"
            size="sm"
            onClick={() => router.push(`/admin/tournaments/${tournament.id}`)}
          >
            View
          </Button>
          <Button
            variant="outline"
            size="sm"
            onClick={() => router.push(`/admin/tournaments/${tournament.id}/edit`)}
          >
            Edit
          </Button>
        </div>
      )
    }
  ];

  return (
    <AdminLayout title="Tournaments">
      <div className="mb-6">
        <div className="flex justify-between items-center">
          <h1 className="text-2xl font-bold text-gray-900">Tournaments</h1>
          <Button
            variant="primary"
            onClick={() => router.push('/admin/tournaments/create')}
          >
            Create Tournament
          </Button>
        </div>
      </div>

      <Card>
        <div className="p-6 border-b border-gray-200">
          <TournamentFilters
            filters={filters}
            onChange={setFilters}
          />
        </div>
        
        <DataTable
          data={tournaments}
          columns={columns}
          loading={loading}
          onRowClick={(tournament) => router.push(`/admin/tournaments/${tournament.id}`)}
        />
      </Card>
    </AdminLayout>
  );
};
```

## 7.4 Team Management UI

### Team Registration Form
```typescript
const TeamRegistrationForm: React.FC = () => {
  const [currentStep, setCurrentStep] = useState(1);
  const [formData, setFormData] = useState<TeamRegistrationData>({
    team: {
      name: '',
      description: '',
      founded_year: '',
      location: ''
    },
    captain: {
      name: '',
      email: '',
      phone: '',
      date_of_birth: ''
    },
    players: [],
    officials: []
  });

  const steps = [
    { id: 1, name: 'Team Information', description: 'Basic team details' },
    { id: 2, name: 'Captain Details', description: 'Team captain information' },
    { id: 3, name: 'Add Players', description: 'Register team players' },
    { id: 4, name: 'Add Officials', description: 'Register team officials' },
    { id: 5, name: 'Review & Submit', description: 'Review and submit registration' }
  ];

  return (
    <MainLayout title="Team Registration">
      <div className="max-w-4xl mx-auto">
        <div className="mb-8">
          <h1 className="text-3xl font-bold text-gray-900">Register Your Team</h1>
          <p className="text-gray-600">Complete the registration process to participate in tournaments</p>
        </div>

        <div className="mb-8">
          <ProgressSteps steps={steps} currentStep={currentStep} />
        </div>

        <Card>
          {currentStep === 1 && (
            <TeamInformationStep
              data={formData.team}
              onChange={(team) => setFormData(prev => ({ ...prev, team }))}
              onNext={() => setCurrentStep(2)}
            />
          )}
          {currentStep === 2 && (
            <CaptainDetailsStep
              data={formData.captain}
              onChange={(captain) => setFormData(prev => ({ ...prev, captain }))}
              onNext={() => setCurrentStep(3)}
              onBack={() => setCurrentStep(1)}
            />
          )}
          {currentStep === 3 && (
            <PlayersStep
              data={formData.players}
              onChange={(players) => setFormData(prev => ({ ...prev, players }))}
              onNext={() => setCurrentStep(4)}
              onBack={() => setCurrentStep(2)}
            />
          )}
          {currentStep === 4 && (
            <OfficialsStep
              data={formData.officials}
              onChange={(officials) => setFormData(prev => ({ ...prev, officials }))}
              onNext={() => setCurrentStep(5)}
              onBack={() => setCurrentStep(3)}
            />
          )}
          {currentStep === 5 && (
            <ReviewStep
              data={formData}
              onSubmit={handleSubmit}
              onBack={() => setCurrentStep(4)}
            />
          )}
        </Card>
      </div>
    </MainLayout>
  );
};
```

### Player Management Interface
```typescript
const PlayerManagementInterface: React.FC = () => {
  const [players, setPlayers] = useState<Player[]>([]);
  const [selectedPlayer, setSelectedPlayer] = useState<Player | null>(null);
  const [showAddModal, setShowAddModal] = useState(false);

  return (
    <TeamManagerLayout team={team}>
      <div className="mb-6">
        <div className="flex justify-between items-center">
          <h1 className="text-2xl font-bold text-gray-900">Team Players</h1>
          <Button
            variant="primary"
            onClick={() => setShowAddModal(true)}
          >
            Add Player
          </Button>
        </div>
      </div>

      <div className="grid grid-cols-1 lg:grid-cols-3 gap-6">
        <div className="lg:col-span-2">
          <Card title="Players">
            <div className="space-y-4">
              {players.map((player) => (
                <PlayerCard
                  key={player.id}
                  player={player}
                  onClick={() => setSelectedPlayer(player)}
                  selected={selectedPlayer?.id === player.id}
                />
              ))}
            </div>
          </Card>
        </div>

        <div className="lg:col-span-1">
          {selectedPlayer ? (
            <PlayerDetailsCard
              player={selectedPlayer}
              onEdit={() => setShowAddModal(true)}
              onRemove={() => handleRemovePlayer(selectedPlayer.id)}
            />
          ) : (
            <Card title="Select a Player">
              <p className="text-gray-500">Click on a player to view details</p>
            </Card>
          )}
        </div>
      </div>

      <Modal
        isOpen={showAddModal}
        onClose={() => setShowAddModal(false)}
        title="Add Player"
        size="lg"
      >
        <PlayerForm
          player={selectedPlayer}
          onSubmit={handlePlayerSubmit}
          onCancel={() => setShowAddModal(false)}
        />
      </Modal>
    </TeamManagerLayout>
  );
};
```

## 7.5 Player Management UI

### Player Profile Interface
```typescript
const PlayerProfileInterface: React.FC = () => {
  const [player, setPlayer] = useState<Player | null>(null);
  const [activeTab, setActiveTab] = useState<'profile' | 'statistics' | 'matches'>('profile');

  const tabs = [
    { id: 'profile', name: 'Profile', icon: <UserIcon /> },
    { id: 'statistics', name: 'Statistics', icon: <ChartIcon /> },
    { id: 'matches', name: 'Matches', icon: <PlayIcon /> }
  ];

  return (
    <MainLayout title="Player Profile">
      <div className="max-w-4xl mx-auto">
        <div className="mb-8">
          <div className="flex items-center space-x-6">
            <div className="flex-shrink-0">
              <img
                className="h-20 w-20 rounded-full"
                src={player?.profile_image || '/default-player.png'}
                alt={player?.name}
              />
            </div>
            <div>
              <h1 className="text-3xl font-bold text-gray-900">{player?.name}</h1>
              <p className="text-gray-600">{player?.position} • {player?.team?.name}</p>
              <p className="text-sm text-gray-500">Jersey #{player?.jersey_number}</p>
            </div>
          </div>
        </div>

        <div className="mb-6">
          <nav className="flex space-x-8">
            {tabs.map((tab) => (
              <button
                key={tab.id}
                onClick={() => setActiveTab(tab.id as any)}
                className={`flex items-center space-x-2 py-2 px-1 border-b-2 font-medium text-sm ${
                  activeTab === tab.id
                    ? 'border-indigo-500 text-indigo-600'
                    : 'border-transparent text-gray-500 hover:text-gray-700 hover:border-gray-300'
                }`}
              >
                {tab.icon}
                <span>{tab.name}</span>
              </button>
            ))}
          </nav>
        </div>

        <div className="mt-6">
          {activeTab === 'profile' && <PlayerProfileTab player={player} />}
          {activeTab === 'statistics' && <PlayerStatisticsTab player={player} />}
          {activeTab === 'matches' && <PlayerMatchesTab player={player} />}
        </div>
      </div>
    </MainLayout>
  );
};
```

## 7.6 Mobile Responsiveness

### Responsive Design Principles
```css
/* Mobile First Approach */
.container {
  @apply w-full px-4 mx-auto;
}

/* Breakpoints */
@media (min-width: 640px) {
  .container { @apply max-w-sm; }
}

@media (min-width: 768px) {
  .container { @apply max-w-md; }
}

@media (min-width: 1024px) {
  .container { @apply max-w-lg; }
}

@media (min-width: 1280px) {
  .container { @apply max-w-xl; }
}
```

### Mobile Navigation
```typescript
const MobileNavigation: React.FC = () => {
  const [isOpen, setIsOpen] = useState(false);

  return (
    <div className="lg:hidden">
      <button
        onClick={() => setIsOpen(!isOpen)}
        className="inline-flex items-center justify-center p-2 rounded-md text-gray-400 hover:text-gray-500 hover:bg-gray-100 focus:outline-none focus:ring-2 focus:ring-inset focus:ring-indigo-500"
      >
        <span className="sr-only">Open main menu</span>
        {isOpen ? (
          <XMarkIcon className="block h-6 w-6" />
        ) : (
          <Bars3Icon className="block h-6 w-6" />
        )}
      </button>

      {isOpen && (
        <div className="fixed inset-0 z-50 lg:hidden">
          <div className="fixed inset-0 bg-gray-600 bg-opacity-75" onClick={() => setIsOpen(false)} />
          <div className="relative flex-1 flex flex-col max-w-xs w-full bg-white">
            <div className="absolute top-0 right-0 -mr-12 pt-2">
              <button
                onClick={() => setIsOpen(false)}
                className="ml-1 flex items-center justify-center h-10 w-10 rounded-full focus:outline-none focus:ring-2 focus:ring-inset focus:ring-white"
              >
                <XMarkIcon className="h-6 w-6 text-white" />
              </button>
            </div>
            <NavigationMenu items={navigationItems} />
          </div>
        </div>
      )}
    </div>
  );
};
```

### Responsive Tables
```typescript
const ResponsiveTable: React.FC<{ data: any[]; columns: Column[] }> = ({ data, columns }) => {
  return (
    <div className="overflow-x-auto">
      <table className="min-w-full divide-y divide-gray-200">
        <thead className="bg-gray-50">
          <tr>
            {columns.map((column) => (
              <th
                key={column.key}
                className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider"
              >
                {column.label}
              </th>
            ))}
          </tr>
        </thead>
        <tbody className="bg-white divide-y divide-gray-200">
          {data.map((item, index) => (
            <tr key={index} className="hover:bg-gray-50">
              {columns.map((column) => (
                <td key={column.key} className="px-6 py-4 whitespace-nowrap text-sm text-gray-900">
                  {column.render ? column.render(item[column.key], item) : item[column.key]}
                </td>
              ))}
            </tr>
          ))}
        </tbody>
      </table>
    </div>
  );
};
```

### Mobile-Optimized Forms
```typescript
const MobileForm: React.FC = () => {
  return (
    <form className="space-y-6">
      <div className="grid grid-cols-1 gap-6 sm:grid-cols-2">
        <Input
          label="First Name"
          name="first_name"
          required
          className="sm:col-span-1"
        />
        <Input
          label="Last Name"
          name="last_name"
          required
          className="sm:col-span-1"
        />
      </div>
      
      <Input
        label="Email"
        name="email"
        type="email"
        required
        className="w-full"
      />
      
      <Input
        label="Phone"
        name="phone"
        type="tel"
        required
        className="w-full"
      />
      
      <div className="flex flex-col space-y-4 sm:flex-row sm:space-y-0 sm:space-x-4">
        <Button
          type="submit"
          variant="primary"
          className="w-full sm:w-auto"
        >
          Submit
        </Button>
        <Button
          type="button"
          variant="outline"
          className="w-full sm:w-auto"
        >
          Cancel
        </Button>
      </div>
    </form>
  );
};
```

---

## UI/UX Best Practices

### Accessibility
- **Keyboard Navigation**: All interactive elements are keyboard accessible
- **Screen Reader Support**: Proper ARIA labels and semantic HTML
- **Color Contrast**: WCAG AA compliant color combinations
- **Focus Management**: Clear focus indicators and logical tab order

### Performance
- **Lazy Loading**: Components and images load on demand
- **Code Splitting**: Route-based code splitting
- **Optimized Images**: WebP format with fallbacks
- **Caching**: Strategic caching of static assets

### User Experience
- **Loading States**: Clear loading indicators
- **Error Handling**: User-friendly error messages
- **Success Feedback**: Confirmation of successful actions
- **Progressive Enhancement**: Works without JavaScript

---

## Next Steps

- [Core Features](08_Core_Features.md) - Core system functionality
- [Advanced Features](09_Advanced_Features.md) - Advanced system features
- [User Guide](17_User_Guide.md) - User documentation
- [Developer Guide](18_Developer_Guide.md) - Development and customization
