# API Documentation

## 5.1 Authentication

### Authentication Methods
- **Session Authentication**: Laravel session-based
- **Token Authentication**: Laravel Sanctum tokens
- **API Key**: Custom API key authentication (future)

### Login Endpoint
```http
POST /api/auth/login
Content-Type: application/json

{
  "email": "user@example.com",
  "password": "password"
}
```

**Response:**
```json
{
  "success": true,
  "data": {
    "user": {
      "id": 1,
      "name": "John Doe",
      "email": "user@example.com",
      "roles": ["player"]
    },
    "token": "1|abcdef123456..."
  },
  "message": "Login successful"
}
```

### Logout Endpoint
```http
POST /api/auth/logout
Authorization: Bearer {token}
```

### Register Endpoint
```http
POST /api/auth/register
Content-Type: application/json

{
  "name": "John Doe",
  "email": "user@example.com",
  "password": "password",
  "password_confirmation": "password",
  "phone": "+254712345678"
}
```

## 5.2 Tournament Endpoints

### List Tournaments
```http
GET /api/tournaments
Authorization: Bearer {token}
```

**Query Parameters:**
- `status`: Filter by status (active, completed, cancelled)
- `type`: Filter by type (knockout, round-robin, etc.)
- `page`: Page number for pagination
- `per_page`: Items per page (default: 15)

**Response:**
```json
{
  "success": true,
  "data": [
    {
      "id": 1,
      "name": "Championship 2025",
      "status": "active",
      "start_date": "2025-02-01",
      "end_date": "2025-02-15",
      "total_participants": 16,
      "entry_fee": 1000
    }
  ],
  "meta": {
    "pagination": {
      "current_page": 1,
      "total": 5,
      "per_page": 15
    }
  }
}
```

### Create Tournament
```http
POST /api/tournaments
Authorization: Bearer {token}
Content-Type: application/json

{
  "name": "New Tournament",
  "description": "Tournament description",
  "type": "knockout",
  "start_date": "2025-02-01",
  "end_date": "2025-02-15",
  "max_participants": 16,
  "entry_fee": 1000
}
```

### Get Tournament
```http
GET /api/tournaments/{id}
Authorization: Bearer {token}
```

### Update Tournament
```http
PUT /api/tournaments/{id}
Authorization: Bearer {token}
Content-Type: application/json

{
  "name": "Updated Tournament Name",
  "description": "Updated description"
}
```

### Delete Tournament
```http
DELETE /api/tournaments/{id}
Authorization: Bearer {token}
```

## 5.3 Team Endpoints

### List Teams
```http
GET /api/teams
Authorization: Bearer {token}
```

### Create Team
```http
POST /api/teams
Authorization: Bearer {token}
Content-Type: application/json

{
  "name": "Team Name",
  "description": "Team description",
  "founded_year": 2020,
  "location": "Nairobi"
}
```

### Get Team
```http
GET /api/teams/{id}
Authorization: Bearer {token}
```

### Update Team
```http
PUT /api/teams/{id}
Authorization: Bearer {token}
Content-Type: application/json

{
  "name": "Updated Team Name",
  "description": "Updated description"
}
```

### Team Registration for Tournament
```http
POST /api/teams/{team_id}/register-tournament
Authorization: Bearer {token}
Content-Type: application/json

{
  "tournament_id": 1
}
```

## 5.4 Player Endpoints

### List Players
```http
GET /api/players
Authorization: Bearer {token}
```

### Create Player
```http
POST /api/players
Authorization: Bearer {token}
Content-Type: application/json

{
  "team_id": 1,
  "jersey_number": 10,
  "position": "forward",
  "height": 180,
  "weight": 75,
  "date_of_birth": "1995-01-01",
  "nationality": "Kenyan"
}
```

### Get Player
```http
GET /api/players/{id}
Authorization: Bearer {token}
```

### Update Player
```http
PUT /api/players/{id}
Authorization: Bearer {token}
Content-Type: application/json

{
  "jersey_number": 11,
  "position": "midfielder"
}
```

### Player Statistics
```http
GET /api/players/{id}/statistics
Authorization: Bearer {token}
```

## 5.5 Match Endpoints

### List Matches
```http
GET /api/matches
Authorization: Bearer {token}
```

**Query Parameters:**
- `tournament_id`: Filter by tournament
- `status`: Filter by status (scheduled, live, completed)
- `round`: Filter by round number

### Create Match
```http
POST /api/matches
Authorization: Bearer {token}
Content-Type: application/json

{
  "tournament_id": 1,
  "team1_id": 1,
  "team2_id": 2,
  "scheduled_at": "2025-02-01 15:00:00",
  "venue": "Stadium Name"
}
```

### Update Match Score
```http
PATCH /api/matches/{id}/score
Authorization: Bearer {token}
Content-Type: application/json

{
  "team1_score": 2,
  "team2_score": 1
}
```

### Complete Match
```http
POST /api/matches/{id}/complete
Authorization: Bearer {token}
Content-Type: application/json

{
  "team1_score": 2,
  "team2_score": 1,
  "winner_id": 1
}
```

## 5.6 Payment Endpoints

### Initiate Payment
```http
POST /api/payments/initiate
Authorization: Bearer {token}
Content-Type: application/json

{
  "amount": 1000,
  "phone_number": "+254712345678",
  "description": "Tournament Registration"
}
```

**Response:**
```json
{
  "success": true,
  "data": {
    "payment_id": 1,
    "checkout_request_id": "ws_CO_123456",
    "merchant_request_id": "ws_MR_123456",
    "customer_message": "Please enter your M-Pesa PIN to complete payment"
  }
}
```

### Check Payment Status
```http
GET /api/payments/{id}/status
Authorization: Bearer {token}
```

### Payment History
```http
GET /api/payments
Authorization: Bearer {token}
```

## 5.7 Admin Endpoints

### Dashboard Statistics
```http
GET /api/admin/dashboard
Authorization: Bearer {token}
```

### User Management
```http
GET /api/admin/users
POST /api/admin/users
GET /api/admin/users/{id}
PUT /api/admin/users/{id}
DELETE /api/admin/users/{id}
```

### Role Management
```http
GET /api/admin/roles
POST /api/admin/roles
PUT /api/admin/roles/{id}
DELETE /api/admin/roles/{id}
```

### Permission Management
```http
GET /api/admin/permissions
POST /api/admin/permissions
PUT /api/admin/permissions/{id}
DELETE /api/admin/permissions/{id}
```

## 5.8 Error Handling

### Error Response Format
```json
{
  "success": false,
  "error": {
    "code": "VALIDATION_ERROR",
    "message": "The given data was invalid",
    "details": {
      "name": ["The name field is required"],
      "email": ["The email must be a valid email address"]
    }
  }
}
```

### Common Error Codes
- **VALIDATION_ERROR**: Input validation failed
- **UNAUTHORIZED**: Authentication required
- **FORBIDDEN**: Insufficient permissions
- **NOT_FOUND**: Resource not found
- **CONFLICT**: Resource conflict
- **SERVER_ERROR**: Internal server error

### HTTP Status Codes
- **200**: Success
- **201**: Created
- **400**: Bad Request
- **401**: Unauthorized
- **403**: Forbidden
- **404**: Not Found
- **422**: Validation Error
- **500**: Internal Server Error

### Rate Limiting
```http
X-RateLimit-Limit: 100
X-RateLimit-Remaining: 99
X-RateLimit-Reset: 1640995200
```

---

## API Usage Examples

### Complete Tournament Creation Flow
```javascript
// 1. Create tournament
const tournament = await api.post('/api/tournaments', {
  name: 'Championship 2025',
  type: 'knockout',
  start_date: '2025-02-01',
  end_date: '2025-02-15',
  max_participants: 16
});

// 2. Register team
const team = await api.post('/api/teams', {
  name: 'Team Alpha',
  description: 'Championship team'
});

// 3. Register team for tournament
await api.post(`/api/teams/${team.data.id}/register-tournament`, {
  tournament_id: tournament.data.id
});

// 4. Add players
await api.post('/api/players', {
  team_id: team.data.id,
  jersey_number: 10,
  position: 'forward'
});
```

### Payment Processing Flow
```javascript
// 1. Initiate payment
const payment = await api.post('/api/payments/initiate', {
  amount: 1000,
  phone_number: '+254712345678',
  description: 'Tournament Registration'
});

// 2. Poll for payment status
const checkStatus = async () => {
  const status = await api.get(`/api/payments/${payment.data.payment_id}/status`);
  if (status.data.status === 'completed') {
    console.log('Payment successful!');
  } else if (status.data.status === 'failed') {
    console.log('Payment failed');
  } else {
    setTimeout(checkStatus, 5000); // Check again in 5 seconds
  }
};
```

---

## Next Steps

- [Frontend Components](06_Frontend_Components.md) - Frontend implementation
- [Business Logic](10_Business_Logic.md) - Service layer documentation
- [Security](11_Security.md) - Security implementation
- [API Reference](19_API_Reference.md) - Complete API reference
