<?php

namespace Database\Seeders;

use Illuminate\Database\Console\Seeds\WithoutModelEvents;
use Illuminate\Database\Seeder;
use App\Models\Tournament;
use Carbon\Carbon;

class TournamentSeeder extends Seeder
{
    /**
     * Run the database seeds.
     */
    public function run(): void
    {
        // Delete all existing tournaments first
        $this->command->info('🗑️  Clearing existing tournaments...');
        Tournament::query()->delete();
        
        // Create 5 editions of The Milimus Tournament
        // Each edition runs from October 1st to December 20th
        // 5th Edition is for current year (2025)
        
        $currentYear = 2025;
        $editions = [
            [
                'edition' => '1st',
                'year' => $currentYear - 4, // 2021
                'status' => 'completed',
            ],
            [
                'edition' => '2nd',
                'year' => $currentYear - 3, // 2022
                'status' => 'completed',
            ],
            [
                'edition' => '3rd',
                'year' => $currentYear - 2, // 2023
                'status' => 'completed',
            ],
            [
                'edition' => '4th',
                'year' => $currentYear - 1, // 2024
                'status' => 'completed',
            ],
            [
                'edition' => '5th',
                'year' => $currentYear, // 2025
                'status' => 'active',
            ],
        ];

        $this->command->info('');
        $this->command->info('🏆 Creating The Milimus Tournament Editions...');
        $this->command->info('');

        foreach ($editions as $edition) {
            $startDate = Carbon::create($edition['year'], 10, 1); // October 1st
            $endDate = Carbon::create($edition['year'], 12, 20);   // December 20th
            $registrationStart = $startDate->copy()->subMonths(2); // 2 months before
            $registrationEnd = $startDate->copy()->subDays(10);    // 10 days before start

            Tournament::create([
                'name' => "The Milimus Tournament - {$edition['edition']} Edition {$edition['year']}",
                'slug' => "milimus-tournament-{$edition['edition']}-edition-{$edition['year']}",
                'description' => "The {$edition['edition']} edition of The Milimus Tournament - Kenya's premier football championship featuring all categories of teams competing for glory and substantial prizes.",
                'type' => 'knockout',
                'format' => 'team',
                'start_date' => $startDate->format('Y-m-d'),
                'end_date' => $endDate->format('Y-m-d'),
                'registration_start' => $registrationStart->format('Y-m-d'),
                'registration_end' => $registrationEnd->format('Y-m-d'),
                'entry_fee' => 4500, // Default fee (will be overridden by category fees)
                'currency' => 'KES',
                
                // Tournament capacity (number of teams)
                'min_participants' => 8,    // Minimum 8 teams to start
                'max_participants' => 64,   // Maximum 64 teams allowed
                
                // Team size (players per team)
                'min_players_per_team' => 11,  // Each team needs minimum 11 players
                'max_players_per_team' => 23,  // Each team can have maximum 23 players
                
                'status' => $edition['status'],
                'registration_status' => $edition['status'] === 'active' ? 'open' : 'closed',
                'location' => 'Multiple Venues Across Kenya',
                'venue' => 'Various Stadiums',
                'organizer' => 'The Milimus Tournament Committee',
                'contact_email' => 'info@themilimus.co.ke',
                'contact_phone' => '+254700000000',
                'rules' => 'FIFA Standard Rules - Category-specific age and gender requirements apply',
                'prize_pool' => 1000000,
                'visibility' => 'public',
                'require_approval' => true,
                'enable_two_tier_approval' => false,
                'auto_accept_registrations' => false,
                
                // Category configuration - All categories enabled
                'enable_categories' => true,
                'available_categories' => ['men', 'women', 'youth_male', 'youth_female', 'children'],
                'category_fees' => [
                    'men' => 4500,
                    'women' => 3500,
                    'youth_male' => 0,      // Free
                    'youth_female' => 0,    // Free
                    'children' => 0,        // Free
                ],
            ]);

            $statusIcon = $edition['status'] === 'active' ? '🟢' : '✅';
            $this->command->info("   {$statusIcon} {$edition['edition']} Edition {$edition['year']} ({$edition['status']})");
        }

        $this->command->info('');
        $this->command->info('━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━');
        $this->command->info('✅ Tournament Seeding Complete');
        $this->command->info('━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━');
        $this->command->info('   📅 5 Editions Created (2021-2025)');
        $this->command->info('   🏆 All categories enabled for each edition');
        $this->command->info('   💰 Men: KES 4,500 | Women: KES 3,500 | Youth & Children: FREE');
        $this->command->info('   📆 Running Period: October 1 - December 20 each year');
        $this->command->info('   🟢 5th Edition 2025: ACTIVE & OPEN');
        $this->command->info('   ✅ Editions 1-4: COMPLETED');
        $this->command->info('━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━');
        $this->command->info('');
    }
}
