<?php

namespace Database\Seeders;

use Illuminate\Database\Seeder;
use App\Models\Permission;
use App\Models\Role;

class TournamentCategoryPermissionsSeeder extends Seeder
{
    public function run(): void
    {
        // Create permissions
        $permissions = [
            [
                'name' => 'tournaments.categories.view',
                'display_name' => 'View Tournament Categories',
                'description' => 'View tournament category configuration',
                'category' => 'tournaments',
                'is_system' => true,
                'sort_order' => 6,
            ],
            [
                'name' => 'tournaments.categories.manage',
                'display_name' => 'Manage Tournament Categories',
                'description' => 'Enable/disable and configure tournament categories',
                'category' => 'tournaments',
                'is_system' => true,
                'sort_order' => 7,
            ],
            [
                'name' => 'tournaments.categories.fees',
                'display_name' => 'Manage Category Fees',
                'description' => 'Set and modify category-specific registration fees',
                'category' => 'tournaments',
                'is_system' => true,
                'sort_order' => 8,
            ],
        ];

        foreach ($permissions as $permissionData) {
            Permission::updateOrCreate(
                ['name' => $permissionData['name']],
                $permissionData
            );
        }

        // Assign permissions to roles
        $adminRole = Role::where('name', 'admin')->first();
        $organizerRole = Role::where('name', 'organizer')->first();
        $tournamentManagerRole = Role::where('name', 'tournament_manager')->first();

        $categoryPermissions = Permission::whereIn('name', [
            'tournaments.categories.view',
            'tournaments.categories.manage',
            'tournaments.categories.fees',
        ])->get();

        if ($adminRole && $categoryPermissions->isNotEmpty()) {
            foreach ($categoryPermissions as $permission) {
                $adminRole->permissions()->syncWithoutDetaching($permission->id);
            }
            $this->command->info('✓ Admin role granted category permissions');
        }

        if ($organizerRole && $categoryPermissions->isNotEmpty()) {
            foreach ($categoryPermissions as $permission) {
                $organizerRole->permissions()->syncWithoutDetaching($permission->id);
            }
            $this->command->info('✓ Organizer role granted category permissions');
        }

        if ($tournamentManagerRole && $categoryPermissions->isNotEmpty()) {
            foreach ($categoryPermissions as $permission) {
                $tournamentManagerRole->permissions()->syncWithoutDetaching($permission->id);
            }
            $this->command->info('✓ Tournament Manager role granted category permissions');
        }

        $this->command->info('Tournament category permissions created and assigned successfully!');
    }
}

