<?php

namespace Database\Seeders;

use Illuminate\Database\Console\Seeds\WithoutModelEvents;
use Illuminate\Database\Seeder;
use App\Models\Role;
use App\Models\Permission;

class RoleSeeder extends Seeder
{
    /**
     * Run the database seeds.
     */
    public function run(): void
    {
        $roles = [
            [
                'name' => 'super_admin',
                'display_name' => 'Super Administrator',
                'description' => 'Full system access with all permissions',
                'permissions' => ['*'],
                'is_system' => true,
                'is_active' => true,
                'priority' => 100,
                'color' => 'bg-red-600',
                'icon' => 'shield-check',
            ],
            [
                'name' => 'admin',
                'display_name' => 'Administrator',
                'description' => 'Tournament system administrator',
                'permissions' => [
                    'tournaments.*',
                    'players.*',
                    'teams.*',
                    'matches.*',
                    'brackets.*',
                    'users.view',
                    'users.edit',
                    'roles.view',
                    'vlog.*',
                ],
                'is_system' => true,
                'is_active' => true,
                'priority' => 90,
                'color' => 'bg-blue-600',
                'icon' => 'shield',
            ],
            [
                'name' => 'organizer',
                'display_name' => 'Tournament Organizer',
                'description' => 'Can create and manage tournaments',
                'permissions' => [
                    'tournaments.create',
                    'tournaments.edit',
                    'tournaments.view',
                    'players.create',
                    'players.edit',
                    'players.view',
                    'teams.create',
                    'teams.edit',
                    'teams.view',
                    'matches.create',
                    'matches.edit',
                    'matches.view',
                    'vlog.view',
                    'vlog.create',
                    'vlog.edit',
                    'vlog.delete',
                    'vlog.manage',
                ],
                'is_system' => true,
                'is_active' => true,
                'priority' => 80,
                'color' => 'bg-green-600',
                'icon' => 'trophy',
            ],
            [
                'name' => 'referee',
                'display_name' => 'Referee',
                'description' => 'Can officiate matches',
                'permissions' => [
                    'matches.view',
                    'matches.edit',
                    'matches.start',
                    'matches.complete',
                    'players.view',
                    'teams.view',
                ],
                'is_system' => true,
                'is_active' => true,
                'priority' => 70,
                'color' => 'bg-yellow-600',
                'icon' => 'whistle',
            ],
            [
                'name' => 'player',
                'display_name' => 'Player',
                'description' => 'Tournament participant',
                'permissions' => [
                    'tournaments.view',
                    'matches.view',
                    'players.view',
                    'teams.view',
                ],
                'is_system' => true,
                'is_active' => true,
                'priority' => 50,
                'color' => 'bg-purple-600',
                'icon' => 'user',
            ],
            [
                'name' => 'spectator',
                'display_name' => 'Spectator',
                'description' => 'Can view public information',
                'permissions' => [
                    'tournaments.view',
                    'matches.view',
                    'players.view',
                    'teams.view',
                ],
                'is_system' => true,
                'is_active' => true,
                'priority' => 30,
                'color' => 'bg-gray-600',
                'icon' => 'eye',
            ],
        ];

        foreach ($roles as $roleData) {
            $role = Role::updateOrCreate(
                ['name' => $roleData['name']],
                $roleData
            );

            // Assign specific permissions via Permission model relationships
            if (isset($roleData['permissions']) && is_array($roleData['permissions'])) {
                $permissionNames = [];
                
                foreach ($roleData['permissions'] as $permissionPattern) {
                    if ($permissionPattern === '*') {
                        // Keep wildcard in JSON for super admin
                        continue;
                    } elseif (str_ends_with($permissionPattern, '.*')) {
                        // Handle wildcard patterns like 'tournaments.*'
                        $category = str_replace('.*', '', $permissionPattern);
                        $categoryPermissions = Permission::where('category', $category)
                            ->where('is_active', true)
                            ->pluck('name')
                            ->toArray();
                        $permissionNames = array_merge($permissionNames, $categoryPermissions);
                    } else {
                        // Direct permission name
                        $permissionNames[] = $permissionPattern;
                    }
                }

                // Sync permissions via relationships
                if (!empty($permissionNames)) {
                    $permissionIds = Permission::whereIn('name', $permissionNames)
                        ->where('is_active', true)
                        ->pluck('id')
                        ->toArray();
                    
                    $role->permissions()->sync($permissionIds);
                }
            }
        }

        $this->command->info('Roles seeded successfully!');
    }
}
