<?php

namespace Database\Seeders;

use Illuminate\Database\Console\Seeds\WithoutModelEvents;
use Illuminate\Database\Seeder;
use App\Models\Permission;
use App\Models\Role;

class PrizeAwardPermissionSeeder extends Seeder
{
    /**
     * Run the database seeds.
     */
    public function run(): void
    {
        // Create Prize Award permissions
        $permissions = [
            [
                'name' => 'prize_awards.view',
                'display_name' => 'View Prize Awards',
                'description' => 'View prize awards and tournament recognition',
                'category' => 'prize_awards',
                'is_system' => true,
                'is_active' => true,
                'sort_order' => 1,
            ],
            [
                'name' => 'prize_awards.create',
                'display_name' => 'Create Prize Awards',
                'description' => 'Create new prize awards for tournaments',
                'category' => 'prize_awards',
                'is_system' => true,
                'is_active' => true,
                'sort_order' => 2,
            ],
            [
                'name' => 'prize_awards.edit',
                'display_name' => 'Edit Prize Awards',
                'description' => 'Edit existing prize awards and toggle status',
                'category' => 'prize_awards',
                'is_system' => true,
                'is_active' => true,
                'sort_order' => 3,
            ],
            [
                'name' => 'prize_awards.delete',
                'display_name' => 'Delete Prize Awards',
                'description' => 'Delete prize awards and perform bulk actions',
                'category' => 'prize_awards',
                'is_system' => true,
                'is_active' => true,
                'sort_order' => 4,
            ],
        ];

        foreach ($permissions as $permissionData) {
            Permission::firstOrCreate(
                ['name' => $permissionData['name']],
                $permissionData
            );
        }

        $this->command->info('Prize award permissions created successfully!');

        // Assign permissions to roles
        $this->assignPermissionsToRoles();
    }

    private function assignPermissionsToRoles(): void
    {
        $adminRole = Role::firstOrCreate(['name' => 'admin']);
        $organizerRole = Role::firstOrCreate(['name' => 'organizer']);
        $teamManagerRole = Role::firstOrCreate(['name' => 'team_manager']);

        $prizeAwardPermissions = Permission::where('category', 'prize_awards')->get();

        // Admin gets all permissions
        foreach ($prizeAwardPermissions as $permission) {
            if (!$adminRole->permissions()->where('permission_id', $permission->id)->exists()) {
                $adminRole->permissions()->attach($permission->id);
            }
        }

        // Organizer gets view, create, and edit permissions
        $organizerPermissions = $prizeAwardPermissions->whereIn('name', ['prize_awards.view', 'prize_awards.create', 'prize_awards.edit']);
        foreach ($organizerPermissions as $permission) {
            if (!$organizerRole->permissions()->where('permission_id', $permission->id)->exists()) {
                $organizerRole->permissions()->attach($permission->id);
            }
        }

        // Team Manager gets only view permission
        $viewPermission = $prizeAwardPermissions->where('name', 'prize_awards.view')->first();
        if ($viewPermission && !$teamManagerRole->permissions()->where('permission_id', $viewPermission->id)->exists()) {
            $teamManagerRole->permissions()->attach($viewPermission->id);
        }

        $this->command->info('Prize award permissions assigned to roles successfully!');
    }
}
