<?php

namespace Database\Seeders;

use App\Models\Tournament;
use App\Models\Team;
use App\Models\GameMatch;
use App\Models\User;
use Illuminate\Database\Console\Seeds\WithoutModelEvents;
use Illuminate\Database\Seeder;
use Carbon\Carbon;

class MilimuCupFixturesSeeder extends Seeder
{
    public function run(): void
    {
        $tournament = Tournament::where('slug', 'milimu-cup-3rd-edition')->first();
        
        if (!$tournament) {
            $this->command->error('Milimu Cup Tournament not found. Please run MilimuCupTournamentSeeder first.');
            return;
        }

        $this->createFixtures($tournament);
        $this->command->info('Milimu Cup Fixtures seeded successfully!');
    }

    private function createFixtures(Tournament $tournament): void
    {
        // December 19, 2023
        $this->createMatch($tournament, 'Ingolomosio Fc', 'Shiners Fc', '2023-12-19 11:00:00', 'group_stage');
        $this->createMatch($tournament, 'Juventus Fc', 'Clevanics Fc', '2023-12-19 14:00:00', 'group_stage');
        $this->createMatch($tournament, 'Township Fc (Arsenal)', 'Salvation Fc', '2023-12-19 16:00:00', 'group_stage');

        // December 21, 2023
        $this->createMatch($tournament, 'Kambiri Fc', 'Young City Fc', '2023-12-21 11:00:00', 'group_stage');
        $this->createMatch($tournament, 'Shiners Fc', 'Green Forest fc (Makuche)', '2023-12-21 13:30:00', 'group_stage');
        $this->createMatch($tournament, 'Mitoto Fc', 'Clevanics Fc', '2023-12-21 16:00:00', 'group_stage');

        // December 22, 2023
        $this->createMatch($tournament, 'Salvation Fc', 'Mukhuyu Fc', '2023-12-22 13:00:00', 'group_stage');
        $this->createMatch($tournament, 'Young City Fc', 'Juventus B Fc', '2023-12-22 16:00:00', 'group_stage');

        // December 23, 2023
        $this->createMatch($tournament, 'Juventus Fc', 'Mitoto Fc', '2023-12-23 13:00:00', 'group_stage');
        $this->createMatch($tournament, 'Ingolomosio Fc', 'Green Forest fc (Makuche)', '2023-12-23 16:00:00', 'group_stage');

        // December 24, 2023
        $this->createMatch($tournament, 'Township Fc (Arsenal)', 'Mukhuyu Fc', '2023-12-24 13:00:00', 'group_stage');
        $this->createMatch($tournament, 'Kambiri Fc', 'Juventus B Fc', '2023-12-24 16:00:00', 'group_stage');

        // December 26, 2023 (Special Kids Day)
        $this->createMatch($tournament, 'Team A Kid\'s', 'Team B Kid\'s', '2023-12-26 10:00:00', 'kids_special');
        $this->createMatch($tournament, 'Team A Girl\'s', 'Team B Girl\'s', '2023-12-26 11:15:00', 'kids_special');
        $this->createMatch($tournament, '1A', '2C', '2023-12-26 15:45:00', 'knockout_stage');

        // December 27, 2023 (Knockout Stage)
        $this->createMatch($tournament, '1B', '2D', '2023-12-27 10:00:00', 'knockout_stage');
        $this->createMatch($tournament, '1C', '2A', '2023-12-27 12:30:00', 'knockout_stage');
        $this->createMatch($tournament, '1D', '2B', '2023-12-27 15:30:00', 'knockout_stage');

        // December 28, 2023 (Ladies Matches)
        $this->createMatch($tournament, 'Juventus Queens', 'Rising Starlets', '2023-12-28 13:00:00', 'ladies_special');
        $this->createMatch($tournament, 'The Inglets', 'The Angels', '2023-12-28 16:00:00', 'ladies_special');

        // December 29, 2023 (Semi-finals)
        $this->createMatch($tournament, 'Winner Match 16', 'Winner Match 17', '2023-12-29 11:30:00', 'semi_final');
        $this->createMatch($tournament, 'Winner Match 18', 'Winner Match 19', '2023-12-29 14:30:00', 'semi_final');

        // December 30, 2023 (Consolation Matches)
        $this->createMatch($tournament, 'Ladies Looser No. 3', 'Ladies Looser No. 4', '2023-12-30 13:00:00', 'consolation');
        $this->createMatch($tournament, 'Looser Game 20', 'Looser Game 21', '2023-12-30 15:30:00', 'consolation');

        // January 1, 2024 (Finals)
        $this->createMatch($tournament, 'Ladies Final Match', 'Ladies Final Match', '2024-01-01 10:00:00', 'ladies_final');
        $this->createMatch($tournament, 'Wazee Final', 'Wazee Final', '2024-01-01 12:30:00', 'entertainment');
        $this->createMatch($tournament, 'Mens Final Match', 'Mens Final Match', '2024-01-01 14:00:00', 'mens_final');
    }

    private function createMatch(Tournament $tournament, string $team1Name, string $team2Name, string $scheduledTime, string $matchType): void
    {
        $team1 = $tournament->teams()->where('name', $team1Name)->first();
        $team2 = $tournament->teams()->where('name', $team2Name)->first();
        $defaultUser = User::first();

        if (!$team1) {
            $team1 = Team::create([
                'tournament_id' => $tournament->id,
                'name' => $team1Name,
                'captain_id' => $defaultUser->id,
                'description' => "Special team for {$matchType}",
                'status' => 'registered',
                'payment_status' => 'completed',
                'payment_amount' => 0.00,
                'payment_method' => 'cheque',
                'payment_completed_at' => Carbon::create(2023, 12, 15),
                'max_players' => 20,
                'notes' => "Special match team",
                'notify_match_updates' => true,
                'notify_tournament_updates' => true,
                'stats' => [
                    'category' => $matchType,
                    'matches_played' => 0,
                    'wins' => 0,
                    'losses' => 0,
                    'draws' => 0,
                    'goals_for' => 0,
                    'goals_against' => 0,
                    'points' => 0
                ]
            ]);
        }

        if (!$team2) {
            $team2 = Team::create([
                'tournament_id' => $tournament->id,
                'name' => $team2Name,
                'captain_id' => $defaultUser->id,
                'description' => "Special team for {$matchType}",
                'status' => 'registered',
                'payment_status' => 'completed',
                'payment_amount' => 0.00,
                'payment_method' => 'cheque',
                'payment_completed_at' => Carbon::create(2023, 12, 15),
                'max_players' => 20,
                'notes' => "Special match team",
                'notify_match_updates' => true,
                'notify_tournament_updates' => true,
                'stats' => [
                    'category' => $matchType,
                    'matches_played' => 0,
                    'wins' => 0,
                    'losses' => 0,
                    'draws' => 0,
                    'goals_for' => 0,
                    'goals_against' => 0,
                    'points' => 0
                ]
            ]);
        }

        GameMatch::create([
            'tournament_id' => $tournament->id,
            'team1_id' => $team1->id,
            'team2_id' => $team2->id,
            'match_type' => $matchType,
            'round' => 1,
            'scheduled_time' => Carbon::parse($scheduledTime),
            'venue' => 'Inglomosio Primary School',
            'status' => 'scheduled',
            'format' => 'best_of_3',
            'duration' => 90,
            'notes' => $this->getNotesForMatchType($matchType),
            'metadata' => [
                'match_type' => $matchType,
                'special_event' => $this->isSpecialEvent($matchType),
                'entertainment_duration' => 45,
                'award_ceremony' => $matchType === 'mens_final' ? '5:45 P.M to 6:30 P.M' : null
            ]
        ]);
    }

    private function getRoundForMatchType(string $matchType): string
    {
        return match($matchType) {
            'group_stage' => 'Group Stage',
            'knockout_stage' => 'Round of 16',
            'semi_final' => 'Semi Final',
            'mens_final' => 'Final',
            'ladies_final' => 'Ladies Final',
            'kids_special' => 'Kids Special',
            'ladies_special' => 'Ladies Special',
            'entertainment' => 'Entertainment',
            'consolation' => 'Consolation',
            default => 'Other'
        };
    }

    private function getNotesForMatchType(string $matchType): string
    {
        return match($matchType) {
            'kids_special' => 'Kids below 13 years',
            'ladies_special' => 'Ladies teams match',
            'entertainment' => 'Eating & Dance Competition (MEN, WOMEN, CHILDREN)',
            'wazee_final' => 'Entertainment for elders',
            'mens_final' => 'Final match with award ceremony',
            default => 'Regular tournament match'
        };
    }

    private function isSpecialEvent(string $matchType): bool
    {
        return in_array($matchType, ['kids_special', 'ladies_special', 'entertainment', 'wazee_final']);
    }
} 