<?php

namespace Database\Seeders;

use Illuminate\Database\Console\Seeds\WithoutModelEvents;
use Illuminate\Database\Seeder;
use App\Models\Permission;
use App\Models\Role;

class FaqPermissionSeeder extends Seeder
{
    /**
     * Run the database seeds.
     */
    public function run(): void
    {
        // Create FAQ permissions
        $permissions = [
            [
                'name' => 'faqs.view',
                'display_name' => 'View FAQs',
                'description' => 'View frequently asked questions',
                'category' => 'faqs',
                'is_system' => true,
                'is_active' => true,
                'sort_order' => 1,
            ],
            [
                'name' => 'faqs.create',
                'display_name' => 'Create FAQs',
                'description' => 'Create new frequently asked questions',
                'category' => 'faqs',
                'is_system' => true,
                'is_active' => true,
                'sort_order' => 2,
            ],
            [
                'name' => 'faqs.edit',
                'display_name' => 'Edit FAQs',
                'description' => 'Edit existing frequently asked questions',
                'category' => 'faqs',
                'is_system' => true,
                'is_active' => true,
                'sort_order' => 3,
            ],
            [
                'name' => 'faqs.delete',
                'display_name' => 'Delete FAQs',
                'description' => 'Delete frequently asked questions',
                'category' => 'faqs',
                'is_system' => true,
                'is_active' => true,
                'sort_order' => 4,
            ],
        ];

        foreach ($permissions as $permissionData) {
            Permission::firstOrCreate(
                ['name' => $permissionData['name']],
                $permissionData
            );
        }

        $this->command->info('FAQ permissions created successfully!');

        // Assign permissions to roles
        $this->assignPermissionsToRoles();
    }

    /**
     * Assign FAQ permissions to appropriate roles
     */
    private function assignPermissionsToRoles(): void
    {
        // Get or create roles
        $adminRole = Role::firstOrCreate(['name' => 'admin']);
        $organizerRole = Role::firstOrCreate(['name' => 'organizer']);
        $teamManagerRole = Role::firstOrCreate(['name' => 'team_manager']);

        // Get FAQ permissions
        $faqPermissions = Permission::where('category', 'faqs')->get();

        // Admin gets all FAQ permissions
        foreach ($faqPermissions as $permission) {
            if (!$adminRole->permissions()->where('permission_id', $permission->id)->exists()) {
                $adminRole->permissions()->attach($permission->id);
            }
        }

        // Organizer gets view, create, and edit permissions
        $organizerPermissions = $faqPermissions->whereIn('name', ['faqs.view', 'faqs.create', 'faqs.edit']);
        foreach ($organizerPermissions as $permission) {
            if (!$organizerRole->permissions()->where('permission_id', $permission->id)->exists()) {
                $organizerRole->permissions()->attach($permission->id);
            }
        }

        // Team Manager gets only view permission
        $viewPermission = $faqPermissions->where('name', 'faqs.view')->first();
        if ($viewPermission && !$teamManagerRole->permissions()->where('permission_id', $viewPermission->id)->exists()) {
            $teamManagerRole->permissions()->attach($viewPermission->id);
        }

        $this->command->info('FAQ permissions assigned to roles successfully!');
        $this->command->info('Admin: All FAQ permissions');
        $this->command->info('Organizer: View, Create, Edit FAQ permissions');
        $this->command->info('Team Manager: View FAQ permission only');
    }
}