<?php

namespace Database\Seeders;

use Illuminate\Database\Seeder;
use App\Models\User;
use App\Models\Tournament;
use App\Models\Team;
use App\Models\Player;
use App\Models\Bracket;
use App\Models\GameMatch;
use App\Models\Role;
use App\Models\Payment;

class DummyDataSeeder extends Seeder
{
    /**
     * Run the database seeds.
     */
    public function run(): void
    {
        // Disable notifications during seeding to avoid queue issues
        \Illuminate\Support\Facades\Notification::fake();
        
        $this->command->info('🚀 Starting dummy data generation...');

        // Create Users
        $this->command->info('👥 Creating users...');
        $adminRole = Role::where('name', 'admin')->first();
        $organizerRole = Role::where('name', 'organizer')->first();
        $teamManagerRole = Role::where('name', 'team_manager')->first();

        // Create admin users
        $admins = User::factory()->count(3)->create();
        if ($adminRole) {
            foreach ($admins as $admin) {
                $admin->roles()->attach($adminRole->id);
            }
        }
        $this->command->info('✅ Created 3 admin users');

        // Create organizer users
        $organizers = User::factory()->count(5)->create();
        if ($organizerRole) {
            foreach ($organizers as $organizer) {
                $organizer->roles()->attach($organizerRole->id);
            }
        }
        $this->command->info('✅ Created 5 organizer users');

        // Create team manager users
        $teamManagers = User::factory()->count(10)->create();
        if ($teamManagerRole) {
            foreach ($teamManagers as $manager) {
                $manager->roles()->attach($teamManagerRole->id);
            }
        }
        $this->command->info('✅ Created 10 team manager users');

        // Create regular users
        $regularUsers = User::factory()->count(20)->create();
        $this->command->info('✅ Created 20 regular users');

        // Get tournaments created by TournamentSeeder
        $this->command->info('🏆 Getting tournaments from TournamentSeeder...');
        $tournaments = Tournament::all();
        $this->command->info('✅ Found ' . $tournaments->count() . ' tournaments');
        
        // Create Teams and Players with categories
        $this->command->info('⚽ Creating teams with categories and players...');
        $allTeams = collect();
        $categories = ['men', 'women', 'youth_male', 'youth_female', 'children'];
        
        foreach ($teamManagers->take(20) as $index => $manager) {
            // Create 1-2 teams per manager
            $teamCount = rand(1, 2);
            
            for ($i = 0; $i < $teamCount; $i++) {
                $category = $categories[array_rand($categories)];
                
                $team = Team::factory()->create([
                    'captain_id' => $manager->id,
                    'category' => $category,
                    'status' => 'active',
                ]);

                // Create 11-15 players per team
                $playerCount = rand(11, 15);
                Player::factory()->count($playerCount)->create([
                    'team_id' => $team->id,
                ]);

                $allTeams->push($team);
            }
        }
        
        $this->command->info('✅ Created ' . $allTeams->count() . ' teams with categories and players');

        // Register teams to tournaments and create payments
        $this->command->info('📝 Registering teams to tournaments with payments...');
        $paymentCount = 0;
        
        foreach ($tournaments as $tournament) {
            // Get teams that match tournament categories
            $eligibleTeams = $tournament->enable_categories
                ? $allTeams->filter(function ($team) use ($tournament) {
                    return in_array($team->category, $tournament->available_categories ?? []);
                })
                : $allTeams;

            // Register 4-12 teams per tournament
            $teamsToRegister = $eligibleTeams->random(min(rand(4, 12), $eligibleTeams->count()));
            
            foreach ($teamsToRegister as $team) {
                // Register team
                $tournament->teams()->attach($team->id, [
                    'status' => collect(['registered', 'active'])->random(),
                    'registered_at' => now()->subDays(rand(1, 30)),
                    'approval_status' => collect(['pending', 'first_approved', 'second_approved'])->random(),
                    'first_approved_at' => rand(0, 1) ? now()->subDays(rand(1, 20)) : null,
                    'second_approved_at' => rand(0, 1) ? now()->subDays(rand(1, 15)) : null,
                ]);

                // Create payment with category-specific fee
                $categoryFee = $tournament->getFeeForCategory($team->category);
                $paymentStatus = collect(['pending', 'completed', 'failed'])->random();
                
                Payment::create([
                    'tournament_id' => $tournament->id,
                    'team_id' => $team->id,
                    'user_id' => $team->captain_id,
                    'amount' => $categoryFee,
                    'status' => $paymentStatus,
                    'reference_number' => 'PAY_' . strtoupper(uniqid()),
                    'payment_method' => collect(['Mpesa', 'Bank Transfer', 'Cash'])->random(),
                    'currency' => $tournament->currency ?? 'KES',
                    'description' => "Tournament Registration Fee - {$tournament->name} ({$team->getCategoryName()} Category)",
                    'paid_at' => $paymentStatus === 'completed' ? now()->subDays(rand(1, 25)) : null,
                ]);
                
                $paymentCount++;
            }
            
            $this->command->info('✅ Registered ' . $teamsToRegister->count() . ' teams to: ' . $tournament->name);
        }

        // Create Brackets for tournaments
        $this->command->info('🎯 Creating tournament brackets...');
        foreach ($tournaments as $tournament) {
            Bracket::factory()->count(rand(1, 3))->create([
                'tournament_id' => $tournament->id,
            ]);
        }
        $this->command->info('✅ Created brackets for tournaments');

        // Create Game Matches
        $this->command->info('🎮 Creating game matches...');
        foreach ($tournaments as $tournament) {
            $tournamentTeams = $tournament->teams;
            if ($tournamentTeams->count() >= 2) {
                // Create 5-10 matches per tournament
                $matchCount = min(rand(5, 10), floor($tournamentTeams->count() / 2));
                
                for ($i = 0; $i < $matchCount; $i++) {
                    $twoTeams = $tournamentTeams->random(2);
                    
                    GameMatch::create([
                        'tournament_id' => $tournament->id,
                        'team1_id' => $twoTeams[0]->id,
                        'team2_id' => $twoTeams[1]->id,
                        'round' => rand(1, 4),
                        'match_number' => $i + 1,
                        'status' => collect(['scheduled', 'in_progress', 'completed'])->random(),
                        'scheduled_time' => now()->addDays(rand(1, 30)),
                        'player1_score' => rand(0, 5),
                        'player2_score' => rand(0, 5),
                    ]);
                }
                
                $this->command->info('✅ Created ' . $matchCount . ' matches for tournament: ' . $tournament->name);
            }
        }

        // Summary
        $this->command->newLine();
        $this->command->info('🎉 Dummy data generation completed!');
        $this->command->newLine();
        $this->command->info('📊 Summary:');
        $this->command->info('   - Users: ' . User::count());
        $this->command->info('   - Tournaments: ' . Tournament::count());
        $this->command->info('   - Teams: ' . Team::count());
        $this->command->info('   - Players: ' . Player::count());
        $this->command->info('   - Payments: ' . $paymentCount);
        $this->command->info('   - Brackets: ' . Bracket::count());
        $this->command->info('   - Matches: ' . GameMatch::count());
        $this->command->newLine();
        $this->command->info('💡 Category Breakdown:');
        foreach ($categories as $cat) {
            $count = Team::where('category', $cat)->count();
            $this->command->info("   - " . ucfirst(str_replace('_', ' ', $cat)) . ": {$count} teams");
        }
        $this->command->newLine();
    }
}
