<?php

namespace Database\Seeders;

use Illuminate\Database\Seeder;
use App\Models\Permission;
use App\Models\Role;

class ApprovalPermissionsSeeder extends Seeder
{
    /**
     * Run the database seeds.
     */
    public function run(): void
    {
        // Create approval permissions
        $permissions = [
            [
                'name' => 'tournaments.approve_participants',
                'display_name' => 'Approve Tournament Participants',
                'description' => 'Can approve tournament participants (both first and second approval)',
                'category' => 'tournaments',
                'is_system' => true,
                'is_active' => true,
                'sort_order' => 10,
            ],
            [
                'name' => 'tournaments.first_approve_participants',
                'display_name' => 'First Approve Tournament Participants',
                'description' => 'Can perform first approval of tournament participants',
                'category' => 'tournaments',
                'is_system' => true,
                'is_active' => true,
                'sort_order' => 11,
            ],
            [
                'name' => 'tournaments.second_approve_participants',
                'display_name' => 'Second Approve Tournament Participants',
                'description' => 'Can perform second approval of tournament participants',
                'category' => 'tournaments',
                'is_system' => true,
                'is_active' => true,
                'sort_order' => 12,
            ],
            [
                'name' => 'tournaments.eliminate_participants',
                'display_name' => 'Eliminate Tournament Participants',
                'description' => 'Can eliminate tournament participants',
                'category' => 'tournaments',
                'is_system' => true,
                'is_active' => true,
                'sort_order' => 13,
            ],
            [
                'name' => 'tournaments.withdraw_participants',
                'display_name' => 'Withdraw Tournament Participants',
                'description' => 'Can withdraw tournament participants',
                'category' => 'tournaments',
                'is_system' => true,
                'is_active' => true,
                'sort_order' => 14,
            ],
            [
                'name' => 'tournaments.manage_approval_settings',
                'display_name' => 'Manage Tournament Approval Settings',
                'description' => 'Can configure tournament approval settings and assign approvers',
                'category' => 'tournaments',
                'is_system' => true,
                'is_active' => true,
                'sort_order' => 15,
            ],
        ];

        foreach ($permissions as $permissionData) {
            Permission::updateOrCreate(
                ['name' => $permissionData['name']],
                $permissionData
            );
        }

        // Assign permissions to roles
        $this->assignPermissionsToRoles();
    }

    /**
     * Assign permissions to appropriate roles
     */
    private function assignPermissionsToRoles(): void
    {
        // Super Admin gets all permissions
        $superAdminRole = Role::where('name', 'super_admin')->first();
        if ($superAdminRole) {
            $superAdminRole->syncPermissions([
                'tournaments.approve_participants',
                'tournaments.first_approve_participants',
                'tournaments.second_approve_participants',
                'tournaments.eliminate_participants',
                'tournaments.withdraw_participants',
                'tournaments.manage_approval_settings',
            ]);
        }

        // Admin gets all tournament permissions
        $adminRole = Role::where('name', 'admin')->first();
        if ($adminRole) {
            $adminRole->syncPermissions([
                'tournaments.approve_participants',
                'tournaments.first_approve_participants',
                'tournaments.second_approve_participants',
                'tournaments.eliminate_participants',
                'tournaments.withdraw_participants',
                'tournaments.manage_approval_settings',
            ]);
        }

        // Create approver roles if they don't exist
        $firstApproverRole = Role::firstOrCreate(
            ['name' => 'first_approver'],
            [
                'display_name' => 'First Approver',
                'description' => 'Can perform first approval of tournament participants',
                'is_system' => true,
                'is_active' => true,
                'priority' => 50,
                'color' => 'bg-blue-500',
                'icon' => 'user-check',
            ]
        );

        $secondApproverRole = Role::firstOrCreate(
            ['name' => 'second_approver'],
            [
                'display_name' => 'Second Approver',
                'description' => 'Can perform second approval of tournament participants',
                'is_system' => true,
                'is_active' => true,
                'priority' => 60,
                'color' => 'bg-green-500',
                'icon' => 'shield-check',
            ]
        );

        // Assign specific permissions to approver roles
        $firstApproverRole->syncPermissions([
            'tournaments.first_approve_participants',
            'tournaments.eliminate_participants',
            'tournaments.withdraw_participants',
        ]);

        $secondApproverRole->syncPermissions([
            'tournaments.second_approve_participants',
            'tournaments.eliminate_participants',
            'tournaments.withdraw_participants',
        ]);
    }
}
