<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Schema;

return new class extends Migration
{
    /**
     * Run the migrations.
     */
    public function up(): void
    {
        Schema::create('brackets', function (Blueprint $table) {
            $table->bigIncrements('id');
            $table->unsignedBigInteger('tournament_id');
            $table->string('name');
            $table->enum('type', ['winners', 'losers', 'final', 'quarter-finals', 'semi-finals']);
            $table->integer('rounds');
            $table->json('structure')->nullable();
            $table->json('seeding')->nullable();
            $table->boolean('is_active')->default(true);
            $table->timestamps();

            $table->index(['tournament_id', 'type']);
        });

        Schema::create('cache', function (Blueprint $table) {
            $table->string('key')->primary();
            $table->mediumText('value');
            $table->integer('expiration');
        });

        Schema::create('cache_locks', function (Blueprint $table) {
            $table->string('key')->primary();
            $table->string('owner');
            $table->integer('expiration');
        });

        Schema::create('email_credentials', function (Blueprint $table) {
            $table->bigIncrements('id');
            $table->string('name');
            $table->string('email');
            $table->string('provider')->index();
            $table->string('smtp_host');
            $table->integer('smtp_port');
            $table->string('smtp_username');
            $table->text('smtp_password');
            $table->string('smtp_encryption')->default('tls');
            $table->boolean('is_active')->default(true);
            $table->text('instructions')->nullable();
            $table->json('metadata')->nullable();
            $table->timestamps();

            $table->index(['email', 'is_active']);
        });

        Schema::create('email_user', function (Blueprint $table) {
            $table->bigIncrements('id');
            $table->unsignedBigInteger('email_credential_id');
            $table->unsignedBigInteger('user_id');
            $table->boolean('can_send')->default(true);
            $table->boolean('can_receive')->default(true);
            $table->json('permissions')->nullable();
            $table->timestamp('granted_at')->useCurrent();
            $table->timestamp('expires_at')->nullable();
            $table->timestamps();

            $table->unique(['email_credential_id', 'user_id']);
            $table->index(['user_id', 'can_receive']);
            $table->index(['user_id', 'can_send']);
        });

        Schema::create('failed_jobs', function (Blueprint $table) {
            $table->bigIncrements('id');
            $table->string('uuid')->unique();
            $table->text('connection');
            $table->text('queue');
            $table->longText('payload');
            $table->longText('exception');
            $table->timestamp('failed_at')->useCurrent();
        });

        Schema::create('faqs', function (Blueprint $table) {
            $table->bigIncrements('id');
            $table->string('category')->default('general');
            $table->string('question');
            $table->text('answer');
            $table->integer('sort_order')->default(0);
            $table->boolean('is_active')->default(true);
            $table->timestamps();
        });

        Schema::create('formations', function (Blueprint $table) {
            $table->bigIncrements('id');
            $table->string('name');
            $table->string('display_name');
            $table->text('description')->nullable();
            $table->json('positions');
            $table->string('category')->default('standard');
            $table->boolean('is_popular')->default(false);
            $table->boolean('is_active')->default(true);
            $table->string('country')->nullable();
            $table->timestamps();
        });

        Schema::create('frontend_fixtures', function (Blueprint $table) {
            $table->bigIncrements('id');
            $table->date('date');
            $table->time('time');
            $table->json('team_a');
            $table->json('team_b');
            $table->string('venue');
            $table->enum('status', ['upcoming', 'live', 'completed'])->default('upcoming');
            $table->string('result')->nullable();
            $table->integer('sort_order')->default(0);
            $table->boolean('active')->default(true);
            $table->timestamps();
        });

        Schema::create('frontend_gallery_images', function (Blueprint $table) {
            $table->bigIncrements('id');
            $table->string('src');
            $table->string('alt');
            $table->string('category');
            $table->string('edition')->nullable();
            $table->integer('height')->nullable();
            $table->string('title')->nullable();
            $table->text('description')->nullable();
            $table->string('photographer')->nullable();
            $table->date('date_taken')->nullable();
            $table->enum('status', ['draft', 'published'])->default('published');
            $table->integer('sort_order')->default(0);
            $table->boolean('featured')->default(false);
            $table->timestamps();
        });

        Schema::create('frontend_hero_slides', function (Blueprint $table) {
            $table->bigIncrements('id');
            $table->string('title');
            $table->text('subtitle');
            $table->string('bg_image');
            $table->string('cta_primary')->nullable();
            $table->string('cta_secondary')->nullable();
            $table->string('cta_primary_url')->nullable();
            $table->string('cta_secondary_url')->nullable();
            $table->integer('sort_order')->default(0);
            $table->enum('status', ['draft', 'published'])->default('published');
            $table->boolean('active')->default(true);
            $table->timestamps();
        });

        Schema::create('frontend_news_articles', function (Blueprint $table) {
            $table->bigIncrements('id');
            $table->string('title');
            $table->text('excerpt')->nullable();
            $table->longText('content');
            $table->string('category');
            $table->string('author');
            $table->string('author_image')->nullable();
            $table->date('date');
            $table->string('read_time')->nullable();
            $table->string('image')->nullable();
            $table->json('tags')->nullable();
            $table->integer('views')->default(0);
            $table->integer('likes')->default(0);
            $table->boolean('featured')->default(false);
            $table->enum('status', ['draft', 'published'])->default('published');
            $table->dateTime('published_at')->nullable();
            $table->string('slug')->unique();
            $table->text('meta_description')->nullable();
            $table->text('meta_keywords')->nullable();
            $table->timestamps();
        });

        Schema::create('frontend_sponsors', function (Blueprint $table) {
            $table->bigIncrements('id');
            $table->string('name');
            $table->string('category');
            $table->string('logo')->nullable();
            $table->text('description')->nullable();
            $table->enum('tier', ['platinum', 'gold', 'silver', 'bronze'])->default('bronze');
            $table->string('website_url')->nullable();
            $table->string('contact_email')->nullable();
            $table->string('contact_phone')->nullable();
            $table->integer('sort_order')->default(0);
            $table->enum('status', ['draft', 'published'])->default('published');
            $table->boolean('active')->default(true);
            $table->timestamps();
        });

        Schema::create('frontend_testimonials', function (Blueprint $table) {
            $table->bigIncrements('id');
            $table->string('name');
            $table->string('role');
            $table->string('team')->nullable();
            $table->string('image')->nullable();
            $table->integer('rating')->default(5);
            $table->text('quote');
            $table->string('achievement')->nullable();
            $table->integer('year')->nullable();
            $table->integer('sort_order')->default(0);
            $table->enum('status', ['draft', 'published'])->default('published');
            $table->boolean('active')->default(true);
            $table->boolean('featured')->default(false);
            $table->timestamps();
        });

        Schema::create('frontend_tournament_achievements', function (Blueprint $table) {
            $table->bigIncrements('id');
            $table->integer('year');
            $table->string('title');
            $table->text('description');
            $table->string('participants')->nullable();
            $table->integer('sort_order')->default(0);
            $table->boolean('active')->default(true);
            $table->timestamps();
        });

        Schema::create('frontend_tournament_highlights', function (Blueprint $table) {
            $table->bigIncrements('id');
            $table->string('icon');
            $table->string('title');
            $table->text('description');
            $table->string('value');
            $table->string('color')->nullable();
            $table->integer('sort_order')->default(0);
            $table->boolean('active')->default(true);
            $table->boolean('status')->default(true);
            $table->timestamps();
        });

        Schema::create('job_batches', function (Blueprint $table) {
            $table->string('id')->primary();
            $table->string('name');
            $table->integer('total_jobs');
            $table->integer('pending_jobs');
            $table->integer('failed_jobs');
            $table->longText('failed_job_ids');
            $table->mediumText('options')->nullable();
            $table->integer('cancelled_at')->nullable();
            $table->integer('created_at');
            $table->integer('finished_at')->nullable();
        });

        Schema::create('jobs', function (Blueprint $table) {
            $table->bigIncrements('id');
            $table->string('queue')->index();
            $table->longText('payload');
            $table->unsignedTinyInteger('attempts');
            $table->unsignedInteger('reserved_at')->nullable();
            $table->unsignedInteger('available_at');
            $table->unsignedInteger('created_at');
        });

        Schema::create('matches', function (Blueprint $table) {
            $table->bigIncrements('id');
            $table->string('name')->nullable();
            $table->string('slug')->nullable();
            $table->string('match_code')->nullable();
            $table->enum('match_type', ['individual', 'team', 'mixed', 'group_stage', 'knockout_stage', 'semi_final', 'final', 'ladies_final', 'kids_special', 'ladies_special', 'entertainment', 'consolation', 'mens_final', 'wazee_final'])->default('individual');
            $table->enum('format', ['best_of_1', 'best_of_3', 'best_of_5', 'best_of_7', 'first_to_win', 'time_limit'])->default('best_of_1');
            $table->integer('best_of_games')->default(1);
            $table->integer('time_limit_minutes')->nullable();
            $table->json('rules')->nullable();
            $table->json('settings')->nullable();
            $table->text('description')->nullable();
            $table->text('notes')->nullable();
            $table->unsignedBigInteger('tournament_id');
            $table->unsignedBigInteger('bracket_id')->nullable()->index('matches_bracket_id_foreign');
            $table->integer('round');
            $table->integer('match_number')->nullable();
            $table->unsignedBigInteger('player1_id')->nullable();
            $table->unsignedBigInteger('player2_id')->nullable()->index('matches_player2_id_foreign');
            $table->unsignedBigInteger('team1_id')->nullable();
            $table->unsignedBigInteger('team2_id')->nullable()->index('matches_team2_id_foreign');
            $table->integer('player1_score')->default(0);
            $table->integer('player2_score')->default(0);
            $table->json('game_scores')->nullable();
            $table->json('round_scores')->nullable();
            $table->integer('player1_games_won')->default(0);
            $table->integer('player2_games_won')->default(0);
            $table->integer('team1_games_won')->default(0);
            $table->integer('team2_games_won')->default(0);
            $table->decimal('player1_rating_change')->default(0);
            $table->decimal('player2_rating_change')->default(0);
            $table->unsignedBigInteger('winner_id')->nullable();
            $table->unsignedBigInteger('winning_team_id')->nullable()->index('matches_winning_team_id_foreign');
            $table->enum('status', ['scheduled', 'in_progress', 'completed', 'cancelled'])->default('scheduled');
            $table->enum('progress_status', ['not_started', 'warmup', 'in_progress', 'paused', 'resumed', 'completed', 'cancelled', 'forfeited'])->default('not_started');
            $table->enum('result_type', ['normal', 'forfeit', 'disqualification', 'technical_win', 'draw', 'no_contest'])->default('normal');
            $table->text('result_notes')->nullable();
            $table->enum('verification_status', ['pending', 'verified', 'disputed', 'under_review'])->default('pending');
            $table->timestamp('verified_at')->nullable();
            $table->dateTime('scheduled_time')->nullable();
            $table->string('venue')->nullable();
            $table->integer('duration')->nullable();
            $table->integer('estimated_duration_minutes')->nullable();
            $table->integer('actual_duration_minutes')->nullable();
            $table->dateTime('started_at')->nullable();
            $table->timestamp('warmup_started_at')->nullable();
            $table->timestamp('warmup_ended_at')->nullable();
            $table->timestamp('paused_at')->nullable();
            $table->timestamp('resumed_at')->nullable();
            $table->integer('pause_duration_minutes')->default(0);
            $table->dateTime('completed_at')->nullable();
            $table->timestamps();
            $table->unsignedBigInteger('verified_by')->nullable()->index('matches_verified_by_foreign');
            $table->boolean('is_streamed')->default(false);
            $table->string('stream_url')->nullable();
            $table->string('twitch_url')->nullable();
            $table->string('youtube_url')->nullable();
            $table->string('stream_key')->nullable();
            $table->string('broadcaster_notes')->nullable();
            $table->boolean('allow_spectators')->default(true);
            $table->integer('spectator_count')->default(0);
            $table->integer('peak_spectator_count')->default(0);
            $table->json('spectator_analytics')->nullable();
            $table->json('match_statistics')->nullable();
            $table->json('performance_metrics')->nullable();
            $table->json('highlight_moments')->nullable();
            $table->decimal('match_quality_score', 5)->nullable();
            $table->integer('total_actions')->default(0);
            $table->integer('average_action_per_minute')->default(0);
            $table->string('server_location')->nullable();
            $table->string('game_server')->nullable();
            $table->string('connection_quality')->nullable();
            $table->json('admin_notes')->nullable();
            $table->json('referee_notes')->nullable();
            $table->unsignedBigInteger('referee_id')->nullable();
            $table->unsignedBigInteger('created_by')->nullable()->index('matches_created_by_foreign');
            $table->unsignedBigInteger('updated_by')->nullable()->index('matches_updated_by_foreign');
            $table->json('metadata')->nullable();
            $table->timestamp('last_activity_at')->nullable();
            $table->timestamp('archived_at')->nullable();

            $table->index(['created_at', 'updated_at']);
            $table->index(['is_streamed', 'allow_spectators']);
            $table->index(['match_type', 'format', 'progress_status']);
            $table->index(['player1_id', 'player2_id']);
            $table->index(['referee_id', 'verified_by']);
            $table->index(['scheduled_time', 'started_at', 'completed_at']);
            $table->index(['slug', 'match_code']);
            $table->index(['status', 'scheduled_time']);
            $table->index(['team1_id', 'team2_id']);
            $table->index(['tournament_id', 'round', 'match_number']);
            $table->index(['tournament_id', 'status', 'scheduled_time']);
            $table->index(['winner_id', 'winning_team_id']);
        });

        Schema::create('media', function (Blueprint $table) {
            $table->bigIncrements('id');
            $table->string('name');
            $table->string('file_name');
            $table->string('mime_type')->index();
            $table->string('extension');
            $table->unsignedBigInteger('size');
            $table->string('path');
            $table->string('disk')->default('public');
            $table->string('collection_name')->nullable()->index();
            $table->json('metadata')->nullable();
            $table->json('custom_properties')->nullable();
            $table->string('alt_text')->nullable();
            $table->text('description')->nullable();
            $table->unsignedBigInteger('user_id')->nullable()->index('media_user_id_foreign');
            $table->string('model_type')->nullable();
            $table->unsignedBigInteger('model_id')->nullable();
            $table->integer('order_column')->nullable();
            $table->timestamps();

            $table->index(['model_type', 'model_id']);
        });

        Schema::create('notification_settings', function (Blueprint $table) {
            $table->bigIncrements('id');
            $table->string('notification_type')->unique()->comment('Notification class identifier');
            $table->string('category')->index()->comment('Notification category (team, payment, etc.)');
            $table->boolean('enabled')->default(true)->comment('System-wide enabled status');
            $table->text('description')->nullable()->comment('Human-readable description');
            $table->boolean('is_critical')->default(false)->comment('Future use: mark as critical');
            $table->json('channels')->nullable()->comment('Available channels: mail, database');
            $table->timestamps();
        });

        Schema::create('notifications', function (Blueprint $table) {
            $table->char('id', 36)->primary();
            $table->string('type');
            $table->string('notifiable_type');
            $table->unsignedBigInteger('notifiable_id');
            $table->text('data');
            $table->timestamp('read_at')->nullable();
            $table->timestamps();

            $table->index(['notifiable_type', 'notifiable_id']);
        });

        Schema::create('password_reset_tokens', function (Blueprint $table) {
            $table->string('email')->primary();
            $table->string('token');
            $table->timestamp('created_at')->nullable();
        });

        Schema::create('payments', function (Blueprint $table) {
            $table->bigIncrements('id');
            $table->unsignedBigInteger('user_id');
            $table->unsignedBigInteger('team_id')->nullable();
            $table->unsignedBigInteger('tournament_id')->nullable()->index('payments_tournament_id_foreign');
            $table->string('payment_method');
            $table->decimal('amount', 10);
            $table->string('currency', 3)->default('KES');
            $table->string('reference_number')->unique();
            $table->string('checkout_request_id')->nullable();
            $table->string('merchant_request_id')->nullable();
            $table->enum('status', ['pending', 'processing', 'completed', 'failed', 'cancelled'])->default('pending');
            $table->json('payment_data')->nullable();
            $table->text('description')->nullable();
            $table->timestamp('paid_at')->nullable();
            $table->string('failure_reason')->nullable();
            $table->timestamps();

            $table->index(['team_id', 'status']);
            $table->index(['user_id', 'status']);
        });

        Schema::create('permission_role', function (Blueprint $table) {
            $table->bigIncrements('id');
            $table->unsignedBigInteger('permission_id')->index();
            $table->unsignedBigInteger('role_id')->index();
            $table->timestamps();

            $table->unique(['permission_id', 'role_id']);
        });

        Schema::create('permissions', function (Blueprint $table) {
            $table->bigIncrements('id');
            $table->string('name')->unique();
            $table->string('display_name')->nullable();
            $table->text('description')->nullable();
            $table->string('category', 50)->default('general');
            $table->boolean('is_system')->default(false);
            $table->boolean('is_active')->default(true);
            $table->integer('sort_order')->default(0)->index();
            $table->json('metadata')->nullable();
            $table->timestamps();

            $table->index(['category', 'is_active']);
            $table->index(['name', 'is_active']);
        });

        Schema::create('players', function (Blueprint $table) {
            $table->bigIncrements('id');
            $table->string('digital_id')->nullable()->unique();
            $table->string('id_number')->nullable();
            $table->string('birth_certificate_number')->nullable();
            $table->string('passport_photo')->nullable();
            $table->string('birth_cert_photo')->nullable();
            $table->enum('approval_status', ['pending', 'under_review', 'approved', 'rejected'])->default('pending');
            $table->unsignedBigInteger('user_id');
            $table->unsignedBigInteger('tournament_id');
            $table->text('national_id_front')->nullable();
            $table->text('national_id_back')->nullable();
            $table->string('display_name');
            $table->text('bio')->nullable();
            $table->string('avatar')->nullable();
            $table->string('country')->nullable();
            $table->string('city')->nullable();
            $table->date('date_of_birth')->nullable();
            $table->enum('gender', ['male', 'female', 'other', 'prefer_not_to_say'])->nullable();
            $table->string('phone')->nullable();
            $table->string('discord_username')->nullable();
            $table->string('steam_id')->nullable();
            $table->string('twitch_username')->nullable();
            $table->string('youtube_channel')->nullable();
            $table->string('twitter_handle')->nullable();
            $table->string('instagram_handle')->nullable();
            $table->string('website')->nullable();
            $table->enum('player_type', ['amateur', 'semi_pro', 'professional', 'legend'])->default('amateur');
            $table->enum('experience_level', ['beginner', 'intermediate', 'advanced', 'expert'])->default('beginner');
            $table->string('primary_game')->nullable();
            $table->enum('football_position', ['Goalkeeper', 'Defender', 'Midfielder', 'Forward', 'Striker', 'Winger', 'Captain', 'Vice-Captain'])->nullable();
            $table->string('jersey_number')->nullable();
            $table->boolean('is_team_captain')->default(false);
            $table->json('game_preferences')->nullable();
            $table->decimal('global_rating')->nullable()->default(1000)->index();
            $table->integer('global_rank')->nullable();
            $table->decimal('tournament_rating')->nullable()->default(1000)->index();
            $table->integer('tournament_rank')->nullable();
            $table->integer('seeding_position')->nullable();
            $table->enum('status', ['registered', 'active', 'eliminated', 'withdrawn', 'suspended', 'banned'])->default('registered');
            $table->enum('verification_status', ['unverified', 'pending', 'verified', 'rejected'])->default('unverified')->index();
            $table->timestamp('verified_at')->nullable();
            $table->string('otp_code', 6)->nullable();
            $table->timestamp('otp_expires_at')->nullable();
            $table->boolean('otp_verified')->nullable()->default(false);
            $table->timestamp('registered_at')->nullable();
            $table->timestamp('last_active_at')->nullable()->index();
            $table->integer('tournament_wins')->nullable()->default(0);
            $table->integer('tournament_losses')->nullable()->default(0);
            $table->integer('tournament_draws')->nullable()->default(0);
            $table->decimal('tournament_win_rate', 5)->nullable()->default(0);
            $table->integer('tournament_matches_played')->nullable()->default(0);
            $table->integer('tournament_matches_won')->nullable()->default(0);
            $table->integer('tournament_current_streak')->nullable()->default(0);
            $table->integer('tournament_longest_streak')->nullable()->default(0);
            $table->json('match_history')->nullable();
            $table->json('performance_metrics')->nullable();
            $table->json('achievements')->nullable();
            $table->json('tournament_stats')->nullable();
            $table->json('rankings_history')->nullable();
            $table->decimal('total_earnings', 10)->nullable()->default(0);
            $table->integer('tournaments_won')->nullable()->default(0);
            $table->integer('prizes_won')->nullable()->default(0);
            $table->json('prize_history')->nullable();
            $table->json('notification_preferences')->nullable();
            $table->json('privacy_settings')->nullable();
            $table->boolean('is_public_profile')->nullable()->default(true);
            $table->boolean('allow_messages')->nullable()->default(true);
            $table->boolean('show_contact_info')->nullable()->default(false);
            $table->json('metadata')->nullable();
            $table->timestamps();
            $table->unsignedBigInteger('team_id')->nullable();

            $table->index(['country', 'city']);
            $table->index(['player_type', 'experience_level']);
            $table->index(['team_id', 'is_team_captain']);
            $table->index(['tournament_id', 'status']);
            $table->unique(['user_id', 'tournament_id']);
            $table->unique(['team_id', 'jersey_number'], 'unique_team_jersey');
        });

        Schema::create('prize_awards', function (Blueprint $table) {
            $table->bigIncrements('id');
            $table->unsignedBigInteger('tournament_id');
            $table->string('category');
            $table->string('position');
            $table->string('title');
            $table->text('description')->nullable();
            $table->decimal('prize_value', 10)->nullable();
            $table->string('prize_type');
            $table->text('additional_benefits')->nullable();
            $table->boolean('is_active')->default(true);
            $table->integer('sort_order')->default(0);
            $table->timestamps();

            $table->index(['is_active', 'sort_order']);
            $table->index(['tournament_id', 'category']);
            $table->index(['tournament_id', 'position']);
        });

        Schema::create('role_user', function (Blueprint $table) {
            $table->bigIncrements('id');
            $table->unsignedBigInteger('role_id');
            $table->unsignedBigInteger('user_id');
            $table->boolean('is_primary')->default(false);
            $table->timestamp('assigned_at')->useCurrent();
            $table->timestamp('expires_at')->nullable()->index();
            $table->json('metadata')->nullable();
            $table->timestamps();

            $table->unique(['role_id', 'user_id']);
            $table->index(['user_id', 'is_primary']);
        });

        Schema::create('roles', function (Blueprint $table) {
            $table->bigIncrements('id');
            $table->string('name')->unique();
            $table->string('display_name')->nullable();
            $table->text('description')->nullable();
            $table->json('permissions')->nullable();
            $table->boolean('is_system')->default(false);
            $table->boolean('is_active')->default(true);
            $table->integer('priority')->default(0)->index();
            $table->string('color')->nullable();
            $table->string('icon')->nullable();
            $table->json('metadata')->nullable();
            $table->timestamps();

            $table->index(['name', 'is_active']);
        });

        Schema::create('sessions', function (Blueprint $table) {
            $table->string('id')->primary();
            $table->unsignedBigInteger('user_id')->nullable()->index();
            $table->string('ip_address', 45)->nullable();
            $table->text('user_agent')->nullable();
            $table->longText('payload');
            $table->integer('last_activity')->index();
        });

        Schema::create('team_formations', function (Blueprint $table) {
            $table->bigIncrements('id');
            $table->unsignedBigInteger('team_id');
            $table->unsignedBigInteger('formation_id')->index('team_formations_formation_id_foreign');
            $table->string('name');
            $table->text('description')->nullable();
            $table->json('player_positions');
            $table->json('formation_data');
            $table->boolean('is_default')->default(false);
            $table->boolean('is_active')->default(true);
            $table->timestamps();

            $table->unique(['team_id', 'is_default'], 'unique_team_default_formation');
        });

        Schema::create('team_members', function (Blueprint $table) {
            $table->bigIncrements('id');
            $table->unsignedBigInteger('team_id');
            $table->unsignedBigInteger('user_id')->index('team_members_user_id_foreign');
            $table->enum('role', ['captain', 'member'])->default('member');
            $table->timestamps();

            $table->unique(['team_id', 'user_id']);
        });

        Schema::create('team_officials', function (Blueprint $table) {
            $table->bigIncrements('id');
            $table->unsignedBigInteger('team_id');
            $table->string('name');
            $table->string('email');
            $table->string('phone');
            $table->enum('role', ['Coach', 'Assistant Coach', 'Manager', 'Physio', 'Doctor', 'Other']);
            $table->text('qualifications')->nullable();
            $table->string('otp_code', 6)->nullable();
            $table->timestamp('otp_expires_at')->nullable();
            $table->boolean('otp_verified')->default(false);
            $table->timestamps();

            $table->unique(['team_id', 'email']);
        });

        Schema::create('team_tournament', function (Blueprint $table) {
            $table->bigIncrements('id');
            $table->unsignedBigInteger('team_id');
            $table->unsignedBigInteger('tournament_id');
            $table->enum('status', ['registered', 'active', 'eliminated', 'withdrawn'])->default('registered');
            $table->enum('approval_status', ['pending', 'first_approved', 'second_approved', 'rejected'])->default('pending');
            $table->unsignedBigInteger('first_approved_by')->nullable()->index('team_tournament_first_approved_by_foreign');
            $table->timestamp('first_approved_at')->nullable();
            $table->unsignedBigInteger('second_approved_by')->nullable()->index('team_tournament_second_approved_by_foreign');
            $table->timestamp('second_approved_at')->nullable();
            $table->text('first_approval_notes')->nullable();
            $table->text('second_approval_notes')->nullable();
            $table->text('rejection_reason')->nullable();
            $table->unsignedBigInteger('rejected_by')->nullable()->index('team_tournament_rejected_by_foreign');
            $table->timestamp('rejected_at')->nullable();
            $table->timestamp('registered_at')->useCurrent();
            $table->timestamp('approved_at')->nullable();
            $table->text('registration_notes')->nullable();
            $table->json('tournament_specific_data')->nullable();
            $table->timestamps();

            $table->unique(['team_id', 'tournament_id']);
            $table->index(['tournament_id', 'approval_status']);
            $table->index(['tournament_id', 'status']);
        });

        Schema::create('teams', function (Blueprint $table) {
            $table->bigIncrements('id');
            $table->integer('tournament_id')->nullable();
            $table->string('name');
            $table->text('description')->nullable();
            $table->integer('captain_id')->nullable();
            $table->string('country')->nullable();
            $table->string('county')->nullable();
            $table->string('location')->nullable();
            $table->string('logo')->nullable();
            $table->integer('max_players')->nullable();
            $table->text('notes')->nullable();
            $table->boolean('notify_match_updates')->default(true);
            $table->boolean('notify_tournament_updates')->default(true);
            $table->timestamp('payment_completed_at')->nullable();
            $table->json('stats')->nullable();
            $table->enum('status', ['active', 'eliminated', 'withdrawn', 'registered', 'pending', 'approved', 'rejected', 'completed'])->default('active')->index('teams_tournament_id_status_index');
            $table->enum('category', ['men', 'women', 'youth_male', 'youth_female', 'children'])->default('men')->index()->comment('Team category: men, women, youth_male, youth_female, children');
            $table->string('payment_status')->default('pending');
            $table->string('transaction_id')->nullable();
            $table->decimal('payment_amount', 10)->nullable();
            $table->string('payment_method')->nullable();
            $table->timestamps();
        });

        Schema::create('tournaments', function (Blueprint $table) {
            $table->bigIncrements('id');
            $table->string('name');
            $table->string('slug')->nullable()->index();
            $table->text('description')->nullable();
            $table->string('location')->nullable();
            $table->string('venue')->nullable();
            $table->string('organizer')->nullable();
            $table->string('contact_email')->nullable();
            $table->string('contact_phone')->nullable();
            $table->string('website')->nullable();
            $table->dateTime('registration_start')->nullable();
            $table->dateTime('registration_end')->nullable();
            $table->enum('type', ['single_elimination', 'double_elimination', 'round_robin', 'group_stage_knockout', 'swiss_system', 'league', 'friendly', 'exhibition', 'knockout', 'elimination', 'round-robin', 'group-knockout', 'double-elimination'])->index();
            $table->enum('status', ['draft', 'active', 'completed', 'cancelled'])->default('draft');
            $table->dateTime('start_date');
            $table->dateTime('end_date');
            $table->integer('max_teams')->nullable();
            $table->integer('min_teams')->nullable()->default(2);
            $table->integer('max_participants')->nullable();
            $table->integer('min_participants')->nullable()->default(2);
            $table->boolean('auto_accept_registrations')->default(true);
            $table->enum('registration_status', ['open', 'closed', 'full'])->default('open');
            $table->string('game_title')->nullable();
            $table->string('game_version')->nullable();
            $table->text('format')->nullable();
            $table->integer('team_size')->nullable();
            $table->integer('rounds')->nullable();
            $table->integer('matches_per_round')->nullable();
            $table->json('bracket_settings')->nullable();
            $table->string('stream_url')->nullable();
            $table->string('youtube_url')->nullable();
            $table->string('twitch_url')->nullable();
            $table->boolean('is_streamed')->default(false);
            $table->boolean('is_featured')->default(false);
            $table->decimal('entry_fee')->nullable()->default(0);
            $table->decimal('prize_pool', 10)->nullable()->default(0);
            $table->json('prize_distribution')->nullable();
            $table->string('currency', 3)->default('USD');
            $table->json('sponsor_info')->nullable();
            $table->json('rewards')->nullable();
            $table->integer('total_participants')->default(0);
            $table->integer('total_matches')->default(0);
            $table->integer('completed_matches')->default(0);
            $table->decimal('average_match_duration')->nullable();
            $table->json('tournament_stats')->nullable();
            $table->string('banner_image')->nullable();
            $table->string('logo_image')->nullable();
            $table->json('social_links')->nullable();
            $table->text('announcement')->nullable();
            $table->enum('visibility', ['public', 'private', 'unlisted'])->default('public');
            $table->boolean('allow_spectators')->default(true);
            $table->boolean('require_approval')->default(false);
            $table->boolean('enable_two_tier_approval')->default(false);
            $table->boolean('enable_categories')->default(false)->comment('Enable multiple categories with different fees');
            $table->json('category_fees')->nullable()->comment('Fees for each category: {"men": 1000, "women": 800, ...}');
            $table->json('available_categories')->nullable()->comment('Which categories are available for this tournament');
            $table->unsignedBigInteger('first_approver_id')->nullable()->index('tournaments_first_approver_id_foreign');
            $table->unsignedBigInteger('second_approver_id')->nullable()->index('tournaments_second_approver_id_foreign');
            $table->json('approval_settings')->nullable();
            $table->json('admin_notes')->nullable();
            $table->json('metadata')->nullable();
            $table->timestamp('published_at')->nullable();
            $table->timestamp('completed_at')->nullable();
            $table->timestamp('cancelled_at')->nullable();
            $table->json('rules')->nullable();
            $table->json('settings')->nullable();
            $table->boolean('is_active')->default(true);
            $table->timestamps();

            $table->unique(['slug']);
            $table->index(['status', 'start_date']);
        });

        Schema::create('user_notification_preferences', function (Blueprint $table) {
            $table->bigIncrements('id');
            $table->unsignedBigInteger('user_id');
            $table->string('notification_type')->comment('Notification type or category identifier');
            $table->enum('preference_type', ['category', 'individual'])->comment('Whether this preference is for a category or individual notification');
            $table->boolean('enabled')->default(true)->comment('User preference: enabled or disabled');
            $table->timestamps();

            $table->index(['user_id', 'preference_type']);
            $table->unique(['user_id', 'notification_type', 'preference_type'], 'user_notif_pref_unique');
        });

        Schema::create('users', function (Blueprint $table) {
            $table->bigIncrements('id');
            $table->string('name');
            $table->string('email')->unique();
            $table->timestamp('email_verified_at')->nullable();
            $table->string('password');
            $table->string('phone')->nullable();
            $table->boolean('profile_complete')->default(false);
            $table->enum('payment_status', ['pending', 'partial', 'completed', 'not_required'])->default('pending');
            $table->enum('user_type', ['individual', 'team', 'player', 'official', 'spectator', 'team_manager'])->default('individual');
            $table->string('otp_code')->nullable();
            $table->timestamp('otp_expires_at')->nullable();
            $table->boolean('must_change_password')->default(false);
            $table->timestamp('terms_accepted_at')->nullable();
            $table->string('remember_token')->nullable();
            $table->timestamps();
            $table->string('id_number')->nullable()->comment('National ID Number');
            $table->string('passport_number')->nullable()->comment('Passport Number');
            $table->date('date_of_birth')->nullable()->comment('Date of Birth');
            $table->enum('gender', ['male', 'female', 'other'])->nullable()->comment('Gender');
            $table->string('nationality')->nullable()->comment('Nationality');
            $table->string('county')->nullable()->comment('County');
            $table->string('sub_county')->nullable()->comment('Sub County');
            $table->string('ward')->nullable()->comment('Ward');
            $table->text('address')->nullable()->comment('Physical Address');
            $table->string('postal_code')->nullable()->comment('Postal Code');
            $table->string('occupation')->nullable()->comment('Occupation');
            $table->string('employer')->nullable()->comment('Employer');
            $table->string('emergency_contact_name')->nullable()->comment('Emergency Contact Name');
            $table->string('emergency_contact_phone')->nullable()->comment('Emergency Contact Phone');
            $table->string('emergency_contact_relationship')->nullable()->comment('Emergency Contact Relationship');
            $table->string('profile_image')->nullable()->comment('Profile Image Path');
            $table->text('bio')->nullable()->comment('Biography');
            $table->json('kyc_documents')->nullable()->comment('KYC Documents (JSON)');
            $table->enum('kyc_status', ['pending', 'verified', 'rejected', 'not_submitted'])->default('not_submitted')->comment('KYC Verification Status');
            $table->timestamp('kyc_verified_at')->nullable()->comment('KYC Verification Date');
            $table->text('kyc_notes')->nullable()->comment('KYC Verification Notes');
        });

        Schema::create('vlog_videos', function (Blueprint $table) {
            $table->bigIncrements('id');
            $table->string('title');
            $table->text('description')->nullable();
            $table->enum('platform', ['youtube', 'instagram', 'facebook', 'tiktok', 'twitter', 'linkedin', 'vimeo']);
            $table->string('video_url');
            $table->text('embed_url')->nullable();
            $table->string('thumbnail_url')->nullable();
            $table->integer('duration')->nullable();
            $table->integer('views')->default(0);
            $table->integer('likes')->default(0);
            $table->string('category')->default('matches');
            $table->json('tags')->nullable();
            $table->boolean('featured')->default(false);
            $table->enum('status', ['draft', 'published', 'archived'])->default('draft');
            $table->timestamp('published_at')->nullable();
            $table->integer('sort_order')->default(0);
            $table->timestamps();
            $table->softDeletes();

            $table->index(['category', 'status']);
            $table->index(['featured', 'status']);
            $table->index(['platform', 'status']);
            $table->index(['status', 'published_at']);
        });

        Schema::table('brackets', function (Blueprint $table) {
            $table->foreign(['tournament_id'])->references(['id'])->on('tournaments')->onUpdate('restrict')->onDelete('cascade');
        });

        Schema::table('email_user', function (Blueprint $table) {
            $table->foreign(['email_credential_id'])->references(['id'])->on('email_credentials')->onUpdate('restrict')->onDelete('cascade');
            $table->foreign(['user_id'])->references(['id'])->on('users')->onUpdate('restrict')->onDelete('cascade');
        });

        Schema::table('matches', function (Blueprint $table) {
            $table->foreign(['bracket_id'])->references(['id'])->on('brackets')->onUpdate('restrict')->onDelete('cascade');
            $table->foreign(['created_by'])->references(['id'])->on('users')->onUpdate('restrict')->onDelete('restrict');
            $table->foreign(['player1_id'])->references(['id'])->on('players')->onUpdate('restrict')->onDelete('cascade');
            $table->foreign(['player2_id'])->references(['id'])->on('players')->onUpdate('restrict')->onDelete('cascade');
            $table->foreign(['referee_id'])->references(['id'])->on('users')->onUpdate('restrict')->onDelete('restrict');
            $table->foreign(['team1_id'])->references(['id'])->on('teams')->onUpdate('restrict')->onDelete('cascade');
            $table->foreign(['team2_id'])->references(['id'])->on('teams')->onUpdate('restrict')->onDelete('cascade');
            $table->foreign(['tournament_id'])->references(['id'])->on('tournaments')->onUpdate('restrict')->onDelete('cascade');
            $table->foreign(['updated_by'])->references(['id'])->on('users')->onUpdate('restrict')->onDelete('restrict');
            $table->foreign(['verified_by'])->references(['id'])->on('users')->onUpdate('restrict')->onDelete('restrict');
            $table->foreign(['winner_id'])->references(['id'])->on('players')->onUpdate('restrict')->onDelete('cascade');
            $table->foreign(['winning_team_id'])->references(['id'])->on('teams')->onUpdate('restrict')->onDelete('cascade');
        });

        Schema::table('media', function (Blueprint $table) {
            $table->foreign(['user_id'])->references(['id'])->on('users')->onUpdate('restrict')->onDelete('set null');
        });

        Schema::table('payments', function (Blueprint $table) {
            $table->foreign(['team_id'])->references(['id'])->on('teams')->onUpdate('restrict')->onDelete('cascade');
            $table->foreign(['tournament_id'])->references(['id'])->on('tournaments')->onUpdate('restrict')->onDelete('cascade');
            $table->foreign(['user_id'])->references(['id'])->on('users')->onUpdate('restrict')->onDelete('cascade');
        });

        Schema::table('permission_role', function (Blueprint $table) {
            $table->foreign(['permission_id'])->references(['id'])->on('permissions')->onUpdate('restrict')->onDelete('cascade');
            $table->foreign(['role_id'])->references(['id'])->on('roles')->onUpdate('restrict')->onDelete('cascade');
        });

        Schema::table('players', function (Blueprint $table) {
            $table->foreign(['team_id'])->references(['id'])->on('teams')->onUpdate('restrict')->onDelete('set null');
            $table->foreign(['tournament_id'])->references(['id'])->on('tournaments')->onUpdate('restrict')->onDelete('cascade');
            $table->foreign(['user_id'])->references(['id'])->on('users')->onUpdate('restrict')->onDelete('cascade');
        });

        Schema::table('prize_awards', function (Blueprint $table) {
            $table->foreign(['tournament_id'])->references(['id'])->on('tournaments')->onUpdate('restrict')->onDelete('cascade');
        });

        Schema::table('role_user', function (Blueprint $table) {
            $table->foreign(['role_id'])->references(['id'])->on('roles')->onUpdate('restrict')->onDelete('cascade');
            $table->foreign(['user_id'])->references(['id'])->on('users')->onUpdate('restrict')->onDelete('cascade');
        });

        Schema::table('team_formations', function (Blueprint $table) {
            $table->foreign(['formation_id'])->references(['id'])->on('formations')->onUpdate('restrict')->onDelete('cascade');
            $table->foreign(['team_id'])->references(['id'])->on('teams')->onUpdate('restrict')->onDelete('cascade');
        });

        Schema::table('team_members', function (Blueprint $table) {
            $table->foreign(['team_id'])->references(['id'])->on('teams')->onUpdate('restrict')->onDelete('cascade');
            $table->foreign(['user_id'])->references(['id'])->on('users')->onUpdate('restrict')->onDelete('cascade');
        });

        Schema::table('team_officials', function (Blueprint $table) {
            $table->foreign(['team_id'])->references(['id'])->on('teams')->onUpdate('restrict')->onDelete('cascade');
        });

        Schema::table('team_tournament', function (Blueprint $table) {
            $table->foreign(['first_approved_by'])->references(['id'])->on('users')->onUpdate('restrict')->onDelete('set null');
            $table->foreign(['rejected_by'])->references(['id'])->on('users')->onUpdate('restrict')->onDelete('set null');
            $table->foreign(['second_approved_by'])->references(['id'])->on('users')->onUpdate('restrict')->onDelete('set null');
            $table->foreign(['team_id'])->references(['id'])->on('teams')->onUpdate('restrict')->onDelete('cascade');
            $table->foreign(['tournament_id'])->references(['id'])->on('tournaments')->onUpdate('restrict')->onDelete('cascade');
        });

        Schema::table('tournaments', function (Blueprint $table) {
            $table->foreign(['first_approver_id'])->references(['id'])->on('users')->onUpdate('restrict')->onDelete('set null');
            $table->foreign(['second_approver_id'])->references(['id'])->on('users')->onUpdate('restrict')->onDelete('set null');
        });

        Schema::table('user_notification_preferences', function (Blueprint $table) {
            $table->foreign(['user_id'])->references(['id'])->on('users')->onUpdate('restrict')->onDelete('cascade');
        });
    }

    /**
     * Reverse the migrations.
     */
    public function down(): void
    {
        Schema::table('user_notification_preferences', function (Blueprint $table) {
            $table->dropForeign('user_notification_preferences_user_id_foreign');
        });

        Schema::table('tournaments', function (Blueprint $table) {
            $table->dropForeign('tournaments_first_approver_id_foreign');
            $table->dropForeign('tournaments_second_approver_id_foreign');
        });

        Schema::table('team_tournament', function (Blueprint $table) {
            $table->dropForeign('team_tournament_first_approved_by_foreign');
            $table->dropForeign('team_tournament_rejected_by_foreign');
            $table->dropForeign('team_tournament_second_approved_by_foreign');
            $table->dropForeign('team_tournament_team_id_foreign');
            $table->dropForeign('team_tournament_tournament_id_foreign');
        });

        Schema::table('team_officials', function (Blueprint $table) {
            $table->dropForeign('team_officials_team_id_foreign');
        });

        Schema::table('team_members', function (Blueprint $table) {
            $table->dropForeign('team_members_team_id_foreign');
            $table->dropForeign('team_members_user_id_foreign');
        });

        Schema::table('team_formations', function (Blueprint $table) {
            $table->dropForeign('team_formations_formation_id_foreign');
            $table->dropForeign('team_formations_team_id_foreign');
        });

        Schema::table('role_user', function (Blueprint $table) {
            $table->dropForeign('role_user_role_id_foreign');
            $table->dropForeign('role_user_user_id_foreign');
        });

        Schema::table('prize_awards', function (Blueprint $table) {
            $table->dropForeign('prize_awards_tournament_id_foreign');
        });

        Schema::table('players', function (Blueprint $table) {
            $table->dropForeign('players_team_id_foreign');
            $table->dropForeign('players_tournament_id_foreign');
            $table->dropForeign('players_user_id_foreign');
        });

        Schema::table('permission_role', function (Blueprint $table) {
            $table->dropForeign('permission_role_permission_id_foreign');
            $table->dropForeign('permission_role_role_id_foreign');
        });

        Schema::table('payments', function (Blueprint $table) {
            $table->dropForeign('payments_team_id_foreign');
            $table->dropForeign('payments_tournament_id_foreign');
            $table->dropForeign('payments_user_id_foreign');
        });

        Schema::table('media', function (Blueprint $table) {
            $table->dropForeign('media_user_id_foreign');
        });

        Schema::table('matches', function (Blueprint $table) {
            $table->dropForeign('matches_bracket_id_foreign');
            $table->dropForeign('matches_created_by_foreign');
            $table->dropForeign('matches_player1_id_foreign');
            $table->dropForeign('matches_player2_id_foreign');
            $table->dropForeign('matches_referee_id_foreign');
            $table->dropForeign('matches_team1_id_foreign');
            $table->dropForeign('matches_team2_id_foreign');
            $table->dropForeign('matches_tournament_id_foreign');
            $table->dropForeign('matches_updated_by_foreign');
            $table->dropForeign('matches_verified_by_foreign');
            $table->dropForeign('matches_winner_id_foreign');
            $table->dropForeign('matches_winning_team_id_foreign');
        });

        Schema::table('email_user', function (Blueprint $table) {
            $table->dropForeign('email_user_email_credential_id_foreign');
            $table->dropForeign('email_user_user_id_foreign');
        });

        Schema::table('brackets', function (Blueprint $table) {
            $table->dropForeign('brackets_tournament_id_foreign');
        });

        Schema::dropIfExists('vlog_videos');

        Schema::dropIfExists('users');

        Schema::dropIfExists('user_notification_preferences');

        Schema::dropIfExists('tournaments');

        Schema::dropIfExists('teams');

        Schema::dropIfExists('team_tournament');

        Schema::dropIfExists('team_officials');

        Schema::dropIfExists('team_members');

        Schema::dropIfExists('team_formations');

        Schema::dropIfExists('sessions');

        Schema::dropIfExists('roles');

        Schema::dropIfExists('role_user');

        Schema::dropIfExists('prize_awards');

        Schema::dropIfExists('players');

        Schema::dropIfExists('permissions');

        Schema::dropIfExists('permission_role');

        Schema::dropIfExists('payments');

        Schema::dropIfExists('password_reset_tokens');

        Schema::dropIfExists('notifications');

        Schema::dropIfExists('notification_settings');

        Schema::dropIfExists('media');

        Schema::dropIfExists('matches');

        Schema::dropIfExists('jobs');

        Schema::dropIfExists('job_batches');

        Schema::dropIfExists('frontend_tournament_highlights');

        Schema::dropIfExists('frontend_tournament_achievements');

        Schema::dropIfExists('frontend_testimonials');

        Schema::dropIfExists('frontend_sponsors');

        Schema::dropIfExists('frontend_news_articles');

        Schema::dropIfExists('frontend_hero_slides');

        Schema::dropIfExists('frontend_gallery_images');

        Schema::dropIfExists('frontend_fixtures');

        Schema::dropIfExists('formations');

        Schema::dropIfExists('faqs');

        Schema::dropIfExists('failed_jobs');

        Schema::dropIfExists('email_user');

        Schema::dropIfExists('email_credentials');

        Schema::dropIfExists('cache_locks');

        Schema::dropIfExists('cache');

        Schema::dropIfExists('brackets');
    }
};
