<?php

namespace Database\Factories;

use App\Models\Tournament;
use App\Models\User;
use Illuminate\Database\Eloquent\Factories\Factory;

/**
 * @extends \Illuminate\Database\Eloquent\Factories\Factory<\App\Models\Player>
 */
class PlayerFactory extends Factory
{
    /**
     * Define the model's default state.
     *
     * @return array<string, mixed>
     */
    public function definition(): array
    {
        $playerTypes = ['amateur', 'semi_pro', 'professional', 'legend'];
        $experienceLevels = ['beginner', 'intermediate', 'advanced', 'expert'];
        $statuses = ['registered', 'active', 'eliminated', 'withdrawn', 'suspended', 'banned'];
        $verificationStatuses = ['unverified', 'pending', 'verified', 'rejected'];
        $genders = ['male', 'female', 'other', 'prefer_not_to_say'];
        
        $matchesPlayed = $this->faker->numberBetween(0, 100);
        $matchesWon = $this->faker->numberBetween(0, $matchesPlayed);
        $winRate = $matchesPlayed > 0 ? round(($matchesWon / $matchesPlayed) * 100, 2) : 0;
        
        $globalRating = $this->faker->randomFloat(2, 800, 2500);
        $tournamentRating = $this->faker->randomFloat(2, 800, 2500);
        
        // Western Kenya - Luhya region cities and towns
        $westernKenyaCities = [
            // Kakamega County
            'Kakamega', 'Mumias', 'Butere', 'Khwisero', 'Shinyalu', 'Lurambi', 'Ikolomani', 'Lugari',
            'Malava', 'Matungu', 'Navakholo', 'Likuyani',
            // Bungoma County
            'Bungoma', 'Webuye', 'Kimilili', 'Chwele', 'Sirisia', 'Bumula', 'Kanduyi', 'Tongaren', 'Malakisi',
            // Vihiga County
            'Vihiga', 'Mbale', 'Luanda', 'Emuhaya', 'Hamisi', 'Sabatia',
            // Busia County
            'Busia', 'Malaba', 'Nambale', 'Butula', 'Funyula', 'Teso'
        ];
        
        // Kenyan phone number format
        $phonePrefix = $this->faker->randomElement(['0701', '0702', '0703', '0704', '0705', '0706', '0707', '0708', '0709', '0710', '0711', '0712', '0713', '0714', '0715', '0720', '0721', '0722', '0723', '0724', '0725', '0726', '0727', '0728', '0729', '0740', '0741', '0742', '0743', '0745', '0746', '0748', '0757', '0758', '0759', '0768', '0769', '0790', '0791', '0792', '0793', '0794', '0795', '0796', '0797', '0798', '0799']);
        $phoneNumber = $phonePrefix . $this->faker->numberBetween(100000, 999999);
        
        return [
            'user_id' => User::factory(),
            'tournament_id' => Tournament::factory(),
            'display_name' => $this->faker->unique()->userName(),
            'bio' => $this->faker->paragraph(),
            'avatar' => $this->faker->imageUrl(200, 200, 'people'),
            'country' => 'Kenya',
            'city' => $westernKenyaCities[array_rand($westernKenyaCities)],
            'date_of_birth' => $this->faker->dateTimeBetween('-50 years', '-16 years'),
            'gender' => $this->faker->randomElement($genders),
            'phone' => $phoneNumber,
            'discord_username' => $this->faker->optional(0.7)->userName() . '#' . $this->faker->numberBetween(1000, 9999),
            'steam_id' => $this->faker->optional(0.6)->numerify('STEAM_0:1:########'),
            'twitch_username' => $this->faker->optional(0.5)->userName(),
            'youtube_channel' => $this->faker->optional(0.3)->url('youtube.com'),
            'twitter_handle' => $this->faker->optional(0.4)->userName(),
            'instagram_handle' => $this->faker->optional(0.4)->userName(),
            'website' => $this->faker->optional(0.2)->url(),
            'player_type' => $this->faker->randomElement($playerTypes),
            'experience_level' => $this->faker->randomElement($experienceLevels),
            'primary_game' => $this->faker->randomElement(['League of Legends', 'Dota 2', 'CS:GO', 'Valorant', 'Overwatch', 'Fortnite', 'PUBG', 'Rocket League']),
            'game_preferences' => $this->faker->randomElements(['FPS', 'MOBA', 'Battle Royale', 'Racing', 'Fighting', 'Strategy'], $this->faker->numberBetween(1, 3)),
            'global_rating' => $globalRating,
            'global_rank' => $this->faker->optional(0.8)->numberBetween(1, 10000),
            'tournament_rating' => $tournamentRating,
            'tournament_rank' => $this->faker->optional(0.8)->numberBetween(1, 1000),
            'seeding_position' => $this->faker->optional(0.6)->numberBetween(1, 64),
            'status' => $this->faker->randomElement($statuses),
            'verification_status' => $this->faker->randomElement($verificationStatuses),
            'verified_at' => $this->faker->optional(0.7)->dateTimeBetween('-1 year', 'now'),
            'registered_at' => $this->faker->dateTimeBetween('-2 years', 'now'),
            'last_active_at' => $this->faker->dateTimeBetween('-1 month', 'now'),
            'tournament_wins' => $this->faker->numberBetween(0, 50),
            'tournament_losses' => $this->faker->numberBetween(0, 50),
            'tournament_draws' => $this->faker->numberBetween(0, 10),
            'tournament_win_rate' => $winRate,
            'tournament_matches_played' => $matchesPlayed,
            'tournament_matches_won' => $matchesWon,
            'tournament_current_streak' => $this->faker->numberBetween(0, 10),
            'tournament_longest_streak' => $this->faker->numberBetween(0, 15),
            'match_history' => $this->generateMatchHistory(),
            'performance_metrics' => $this->generatePerformanceMetrics(),
            'achievements' => $this->generateAchievements(),
            'tournament_stats' => $this->generateTournamentStats(),
            'rankings_history' => $this->generateRankingsHistory(),
            'total_earnings' => $this->faker->randomFloat(2, 0, 50000),
            'tournaments_won' => $this->faker->numberBetween(0, 20),
            'prizes_won' => $this->faker->numberBetween(0, 50),
            'prize_history' => $this->generatePrizeHistory(),
            'notification_preferences' => [
                'email_notifications' => $this->faker->boolean(),
                'push_notifications' => $this->faker->boolean(),
                'match_reminders' => $this->faker->boolean(),
                'tournament_updates' => $this->faker->boolean(),
                'achievement_notifications' => $this->faker->boolean(),
            ],
            'privacy_settings' => [
                'show_profile' => $this->faker->boolean(80),
                'show_stats' => $this->faker->boolean(70),
                'show_contact' => $this->faker->boolean(30),
                'allow_messages' => $this->faker->boolean(60),
            ],
            'is_public_profile' => $this->faker->boolean(80),
            'allow_messages' => $this->faker->boolean(60),
            'show_contact_info' => $this->faker->boolean(30),
            'metadata' => [
                'favorite_team' => $this->faker->optional()->randomElement(['AFC Leopards', 'Kakamega Homeboyz', 'Western Stima', 'Vihiga United', 'Nzoia Sugar', 'Mumias Sugar', 'Bungoma Super Stars']),
                'favorite_player' => $this->faker->optional()->name(),
                'preferred_playstyle' => $this->faker->optional()->randomElement(['Aggressive', 'Defensive', 'Balanced', 'Support']),
                'timezone' => 'Africa/Nairobi',
                'language' => $this->faker->randomElement(['en', 'sw', 'luy']), // English, Swahili, and Luhya
            ],
        ];
    }

    /**
     * Generate sample match history
     */
    private function generateMatchHistory(): array
    {
        $history = [];
        $count = $this->faker->numberBetween(0, 20);
        
        for ($i = 0; $i < $count; $i++) {
            $history[] = [
                'match_id' => $this->faker->numberBetween(1, 1000),
                'opponent' => $this->faker->numberBetween(1, 100),
                'result' => $this->faker->randomElement(['win', 'loss']),
                'score' => [$this->faker->numberBetween(0, 21), $this->faker->numberBetween(0, 21)],
                'played_at' => $this->faker->dateTimeBetween('-6 months', 'now')->format('Y-m-d H:i:s'),
            ];
        }
        
        return $history;
    }

    /**
     * Generate performance metrics
     */
    private function generatePerformanceMetrics(): array
    {
        return [
            'average_match_duration' => $this->faker->numberBetween(15, 120),
            'best_win_streak' => $this->faker->numberBetween(0, 15),
            'worst_loss_streak' => $this->faker->numberBetween(0, 10),
            'total_playtime_hours' => $this->faker->numberBetween(10, 1000),
            'favorite_map' => $this->faker->randomElement(['Summoner\'s Rift', 'Dust 2', 'Ascent', 'Nuke', 'Mirage']),
            'preferred_role' => $this->faker->randomElement(['Carry', 'Support', 'Mid', 'Jungle', 'Top', 'ADC']),
            'accuracy_percentage' => $this->faker->randomFloat(2, 60, 95),
            'reaction_time_ms' => $this->faker->numberBetween(150, 300),
        ];
    }

    /**
     * Generate achievements
     */
    private function generateAchievements(): array
    {
        $achievements = [];
        $possibleAchievements = [
            'First Win', 'Win Streak', 'Tournament Champion', 'Perfect Match', 
            'Comeback King', 'Underdog Victory', 'Speed Demon', 'Accuracy Master',
            'Team Player', 'Solo Carry', 'Tournament MVP', 'Rising Star'
        ];
        
        $count = $this->faker->numberBetween(0, 8);
        $selectedAchievements = $this->faker->randomElements($possibleAchievements, $count);
        
        foreach ($selectedAchievements as $achievement) {
            $achievements[] = [
                'name' => $achievement,
                'earned_at' => $this->faker->dateTimeBetween('-1 year', 'now')->format('Y-m-d H:i:s'),
                'data' => [
                    'description' => $this->faker->sentence(),
                    'difficulty' => $this->faker->randomElement(['Easy', 'Medium', 'Hard', 'Legendary']),
                ],
            ];
        }
        
        return $achievements;
    }

    /**
     * Generate tournament statistics
     */
    private function generateTournamentStats(): array
    {
        return [
            'tournaments_joined' => $this->faker->numberBetween(1, 50),
            'tournaments_won' => $this->faker->numberBetween(0, 20),
            'best_finish' => $this->faker->randomElement(['1st', '2nd', '3rd', '4th', 'Top 8', 'Top 16']),
            'average_finish' => $this->faker->randomFloat(1, 1, 32),
            'total_prize_money' => $this->faker->randomFloat(2, 0, 50000),
            'biggest_prize' => $this->faker->randomFloat(2, 0, 10000),
            'favorite_tournament_type' => $this->faker->randomElement(['Single Elimination', 'Double Elimination', 'Round Robin', 'Swiss']),
        ];
    }

    /**
     * Generate rankings history
     */
    private function generateRankingsHistory(): array
    {
        $history = [];
        $months = 12;
        
        for ($i = 0; $i < $months; $i++) {
            $history[] = [
                'date' => now()->subMonths($months - $i)->format('Y-m-d'),
                'global_rating' => $this->faker->randomFloat(2, 800, 2500),
                'global_rank' => $this->faker->numberBetween(1, 10000),
                'tournament_rating' => $this->faker->randomFloat(2, 800, 2500),
                'tournament_rank' => $this->faker->numberBetween(1, 1000),
            ];
        }
        
        return $history;
    }

    /**
     * Generate prize history
     */
    private function generatePrizeHistory(): array
    {
        $history = [];
        $count = $this->faker->numberBetween(0, 10);
        
        for ($i = 0; $i < $count; $i++) {
            $history[] = [
                'tournament_name' => $this->faker->words(3, true) . ' Tournament',
                'position' => $this->faker->randomElement(['1st', '2nd', '3rd', '4th', 'Top 8']),
                'prize_amount' => $this->faker->randomFloat(2, 100, 10000),
                'date' => $this->faker->dateTimeBetween('-1 year', 'now')->format('Y-m-d'),
            ];
        }
        
        return $history;
    }

    /**
     * Indicate that the player is a professional
     */
    public function professional(): static
    {
        return $this->state(fn (array $attributes) => [
            'player_type' => 'professional',
            'experience_level' => 'expert',
            'global_rating' => $this->faker->randomFloat(2, 2000, 2500),
            'tournament_rating' => $this->faker->randomFloat(2, 2000, 2500),
            'verification_status' => 'verified',
        ]);
    }

    /**
     * Indicate that the player is verified
     */
    public function verified(): static
    {
        return $this->state(fn (array $attributes) => [
            'verification_status' => 'verified',
            'verified_at' => $this->faker->dateTimeBetween('-1 year', 'now'),
        ]);
    }

    /**
     * Indicate that the player is active
     */
    public function active(): static
    {
        return $this->state(fn (array $attributes) => [
            'status' => 'active',
            'last_active_at' => $this->faker->dateTimeBetween('-1 week', 'now'),
        ]);
    }

    /**
     * Indicate that the player is a legend
     */
    public function legend(): static
    {
        return $this->state(fn (array $attributes) => [
            'player_type' => 'legend',
            'experience_level' => 'expert',
            'global_rating' => $this->faker->randomFloat(2, 2200, 2500),
            'tournament_rating' => $this->faker->randomFloat(2, 2200, 2500),
            'verification_status' => 'verified',
            'tournaments_won' => $this->faker->numberBetween(5, 20),
            'total_earnings' => $this->faker->randomFloat(2, 10000, 50000),
        ]);
    }
}
