#!/usr/bin/env php
<?php

/**
 * Queue Worker Cron Job (No exec() required)
 * 
 * This script processes the Laravel queue WITHOUT using exec()
 * Perfect for shared hosting where exec() is disabled
 * 
 * cPanel Command:
 * /usr/bin/php /home/USERNAME/public_html/cron.php
 * 
 * Frequency: Every minute (* * * * *)
 */

// Define project root
define('LARAVEL_START', microtime(true));

// Bootstrap Laravel
require __DIR__.'/vendor/autoload.php';
$app = require_once __DIR__.'/bootstrap/app.php';

// Boot the application
$kernel = $app->make(Illuminate\Contracts\Console\Kernel::class);
$kernel->bootstrap();

// Log file
$logFile = __DIR__ . '/storage/logs/queue-cron.log';

// Function to log messages
function logQueue($message) {
    global $logFile;
    $timestamp = date('Y-m-d H:i:s');
    $logEntry = "[{$timestamp}] {$message}" . PHP_EOL;
    file_put_contents($logFile, $logEntry, FILE_APPEND);
}

try {
    // Process queue directly using Artisan facade
    $startTime = microtime(true);
    
    // Run queue:work command
    Illuminate\Support\Facades\Artisan::call('queue:work', [
        '--queue' => 'high,default,low',
        '--stop-when-empty' => true,
        '--timeout' => 120,
        '--tries' => 3,
    ]);
    
    $executionTime = round(microtime(true) - $startTime, 2);
    
    // Get queue status
    $jobsCount = \Illuminate\Support\Facades\DB::table('jobs')->count();
    $failedCount = \Illuminate\Support\Facades\DB::table('failed_jobs')->count();
    
    // Log only if there's activity or errors
    if ($jobsCount > 0 || $failedCount > 10 || $executionTime > 5) {
        logQueue("Queue: {$jobsCount} pending, {$failedCount} failed, {$executionTime}s");
        
        if ($failedCount > 10) {
            logQueue("WARNING: {$failedCount} failed jobs - run: php artisan queue:retry all");
        }
    }
    
} catch (\Exception $e) {
    logQueue("ERROR: " . $e->getMessage());
}

// Clean up old log entries (keep last 200 lines only)
if (file_exists($logFile)) {
    $lines = file($logFile);
    if (count($lines) > 200) {
        file_put_contents($logFile, implode('', array_slice($lines, -200)));
    }
}

exit(0);

