<?php

namespace App\Services;

use App\Models\Tournament;
use App\Models\Bracket;
use App\Models\GameMatch;
use App\Models\Team;
use App\Models\Player;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class TournamentBracketService
{
    protected Tournament $tournament;
    protected array $participants;
    protected ?string $format;
    protected array $settings;

    public function __construct(Tournament $tournament)
    {
        $this->tournament = $tournament;
        $this->format = $tournament->format ?? 'team';
        $this->settings = $tournament->bracket_settings ?? [];
    }

    /**
     * Generate bracket based on tournament type
     */
    public function generateBracket(array $participants, string $bracketType = 'winners'): Bracket
    {
        $this->participants = $participants;
        
        // Create bracket
        $bracket = Bracket::create([
            'tournament_id' => $this->tournament->id,
            'name' => $this->getBracketName($bracketType),
            'type' => $bracketType,
            'rounds' => 0,
            'structure' => [],
            'seeding' => [],
            'is_active' => true,
        ]);

        // Apply intelligent seeding
        $seededParticipants = $this->applySeeding($participants);

        // Generate matches based on tournament type
        switch ($this->tournament->type) {
            case 'knockout':
            case 'elimination':
                $this->generateSingleElimination($bracket, $seededParticipants);
                break;
            case 'double-elimination':
                $this->generateDoubleElimination($bracket, $seededParticipants);
                break;
            case 'round-robin':
                $this->generateRoundRobin($bracket, $seededParticipants);
                break;
            case 'group-knockout':
                $this->generateGroupKnockout($bracket, $seededParticipants);
                break;
            case 'swiss_system':
                $this->generateSwissSystem($bracket, $seededParticipants);
                break;
            default:
                $this->generateSingleElimination($bracket, $seededParticipants);
        }

        return $bracket->fresh(['gameMatches']);
    }

    /**
     * Apply intelligent seeding based on rankings, ratings, or random
     */
    protected function applySeeding(array $participants): array
    {
        $seedingMethod = $this->settings['seeding_method'] ?? 'random';
        
        switch ($seedingMethod) {
            case 'ranking':
                return $this->seedByRanking($participants);
            case 'rating':
                return $this->seedByRating($participants);
            case 'balanced':
                return $this->seedBalanced($participants);
            case 'random':
            default:
                return $this->seedRandom($participants);
        }
    }

    /**
     * Seed participants by their ranking
     */
    protected function seedByRanking(array $participants): array
    {
        // Sort by ranking (assuming participants have ranking data)
        usort($participants, function ($a, $b) {
            $rankA = $this->getParticipantRanking($a);
            $rankB = $this->getParticipantRanking($b);
            return $rankA <=> $rankB;
        });

        return $participants;
    }

    /**
     * Seed participants by their rating/skill level
     */
    protected function seedByRating(array $participants): array
    {
        // Sort by rating (assuming participants have rating data)
        usort($participants, function ($a, $b) {
            $ratingA = $this->getParticipantRating($a);
            $ratingB = $this->getParticipantRating($b);
            return $ratingB <=> $ratingA; // Higher rating first
        });

        return $participants;
    }

    /**
     * Balanced seeding to avoid early matchups of top seeds
     */
    protected function seedBalanced(array $participants): array
    {
        $count = count($participants);
        $seeded = array_fill(0, $count, null);
        
        // Place top seeds in strategic positions
        $topSeeds = array_slice($participants, 0, min(8, $count));
        $positions = $this->getBalancedPositions($count);
        
        foreach ($topSeeds as $index => $participant) {
            $seeded[$positions[$index]] = $participant;
        }
        
        // Fill remaining positions with other participants
        $remaining = array_slice($participants, count($topSeeds));
        $remainingIndex = 0;
        
        for ($i = 0; $i < $count; $i++) {
            if ($seeded[$i] === null) {
                $seeded[$i] = $remaining[$remainingIndex] ?? null;
                $remainingIndex++;
            }
        }
        
        return array_filter($seeded);
    }

    /**
     * Random seeding
     */
    protected function seedRandom(array $participants): array
    {
        shuffle($participants);
        return $participants;
    }

    /**
     * Generate single elimination bracket
     */
    protected function generateSingleElimination(Bracket $bracket, array $participants): void
    {
        $participantCount = count($participants);
        $rounds = ceil(log($participantCount, 2));
        $totalMatches = $participantCount - 1;
        
        // Calculate byes for first round
        $powerOfTwo = pow(2, $rounds);
        $byes = $powerOfTwo - $participantCount;
        
        $this->createSingleEliminationMatches($bracket, $participants, $rounds, $byes);
        
        $bracket->update([
            'rounds' => $rounds,
            'structure' => [
                'type' => 'single_elimination',
                'total_participants' => $participantCount,
                'total_matches' => $totalMatches,
                'rounds' => $rounds,
                'byes' => $byes,
                'power_of_two' => $powerOfTwo,
            ],
            'seeding' => [
                'method' => $this->settings['seeding_method'] ?? 'random',
                'participants' => $participants,
            ],
        ]);
    }

    /**
     * Generate double elimination bracket
     */
    protected function generateDoubleElimination(Bracket $bracket, array $participants): void
    {
        $participantCount = count($participants);
        $rounds = ceil(log($participantCount, 2)) * 2 - 1;
        
        // Create winners bracket
        $this->createSingleEliminationMatches($bracket, $participants, ceil(log($participantCount, 2)), 0, 'winners');
        
        // Create losers bracket (simplified - would need more complex logic for full implementation)
        $this->createLosersBracketMatches($bracket, $participantCount);
        
        $bracket->update([
            'rounds' => $rounds,
            'structure' => [
                'type' => 'double_elimination',
                'total_participants' => $participantCount,
                'rounds' => $rounds,
                'winners_bracket_rounds' => ceil(log($participantCount, 2)),
                'losers_bracket_rounds' => ceil(log($participantCount, 2)) - 1,
            ],
        ]);
    }

    /**
     * Generate round robin tournament
     */
    protected function generateRoundRobin(Bracket $bracket, array $participants): void
    {
        $participantCount = count($participants);
        $rounds = $participantCount - 1;
        $matchesPerRound = intval($participantCount / 2);
        
        $this->createRoundRobinMatches($bracket, $participants, $rounds, $matchesPerRound);
        
        $bracket->update([
            'rounds' => $rounds,
            'structure' => [
                'type' => 'round_robin',
                'total_participants' => $participantCount,
                'rounds' => $rounds,
                'matches_per_round' => $matchesPerRound,
                'total_matches' => $rounds * $matchesPerRound,
            ],
        ]);
    }

    /**
     * Generate group stage + knockout tournament
     */
    protected function generateGroupKnockout(Bracket $bracket, array $participants): void
    {
        $participantCount = count($participants);
        $groups = $this->settings['groups'] ?? 4;
        $participantsPerGroup = ceil($participantCount / $groups);
        
        // Create group stage matches
        $this->createGroupStageMatches($bracket, $participants, $groups, $participantsPerGroup);
        
        $bracket->update([
            'rounds' => 0, // Will be updated when knockout stage is created
            'structure' => [
                'type' => 'group_knockout',
                'total_participants' => $participantCount,
                'groups' => $groups,
                'participants_per_group' => $participantsPerGroup,
                'group_stage_completed' => false,
            ],
        ]);
    }

    /**
     * Generate Swiss system tournament
     */
    protected function generateSwissSystem(Bracket $bracket, array $participants): void
    {
        $participantCount = count($participants);
        $rounds = $this->settings['swiss_rounds'] ?? ceil(log($participantCount, 2));
        
        // Swiss system starts with random pairings
        $this->createSwissRoundMatches($bracket, $participants, 1);
        
        $bracket->update([
            'rounds' => $rounds,
            'structure' => [
                'type' => 'swiss_system',
                'total_participants' => $participantCount,
                'rounds' => $rounds,
                'current_round' => 1,
                'pairing_method' => 'random', // First round is random
            ],
        ]);
    }

    /**
     * Create matches for single elimination
     */
    protected function createSingleEliminationMatches(Bracket $bracket, array $participants, int $rounds, int $byes, string $bracketType = 'winners'): void
    {
        $participantCount = count($participants);
        $powerOfTwo = pow(2, $rounds);
        
        // First round matches
        $firstRoundMatches = intval($powerOfTwo / 2);
        $matchNumber = 1;
        
        for ($i = 0; $i < $firstRoundMatches; $i++) {
            $participant1 = $participants[$i * 2] ?? null;
            $participant2 = $participants[$i * 2 + 1] ?? null;
            
            // Skip if both participants are null (bye situation)
            if (!$participant1 && !$participant2) {
                continue;
            }
            
            $matchData = [
                'tournament_id' => $this->tournament->id,
                'bracket_id' => $bracket->id,
                'round' => 1,
                'match_number' => $matchNumber,
                'status' => 'scheduled',
                'match_type' => $this->format === 'team' ? 'team' : 'individual',
            ];
            
            if ($participant1) {
                if ($this->format === 'team') {
                    $matchData['team1_id'] = $participant1['id'];
                } else {
                    $matchData['player1_id'] = $participant1['id'];
                }
            }
            
            if ($participant2) {
                if ($this->format === 'team') {
                    $matchData['team2_id'] = $participant2['id'];
                } else {
                    $matchData['player2_id'] = $participant2['id'];
                }
            }
            
            GameMatch::create($matchData);
            $matchNumber++;
        }
    }

    /**
     * Create matches for round robin
     */
    protected function createRoundRobinMatches(Bracket $bracket, array $participants, int $rounds, int $matchesPerRound): void
    {
        $participantCount = count($participants);
        
        for ($round = 1; $round <= $rounds; $round++) {
            $matchNumber = 1;
            
            // Round robin pairing algorithm
            $pairs = $this->generateRoundRobinPairs($participants, $round);
            
            foreach ($pairs as $pair) {
                $matchData = [
                    'tournament_id' => $this->tournament->id,
                    'bracket_id' => $bracket->id,
                    'round' => $round,
                    'match_number' => $matchNumber,
                    'status' => 'scheduled',
                    'match_type' => $this->format === 'team' ? 'team' : 'individual',
                ];
                
                if ($this->format === 'team') {
                    $matchData['team1_id'] = $pair[0]['id'];
                    $matchData['team2_id'] = $pair[1]['id'];
                } else {
                    $matchData['player1_id'] = $pair[0]['id'];
                    $matchData['player2_id'] = $pair[1]['id'];
                }
                
                GameMatch::create($matchData);
                $matchNumber++;
            }
        }
    }

    /**
     * Create group stage matches
     */
    protected function createGroupStageMatches(Bracket $bracket, array $participants, int $groups, int $participantsPerGroup): void
    {
        // Distribute participants into groups
        $groupedParticipants = array_chunk($participants, $participantsPerGroup);
        
        foreach ($groupedParticipants as $groupIndex => $groupParticipants) {
            $groupNumber = $groupIndex + 1;
            $groupRounds = count($groupParticipants) - 1;
            
            // Create round robin matches for each group
            for ($round = 1; $round <= $groupRounds; $round++) {
                $pairs = $this->generateRoundRobinPairs($groupParticipants, $round);
                
                foreach ($pairs as $pairIndex => $pair) {
                    $matchData = [
                        'tournament_id' => $this->tournament->id,
                        'bracket_id' => $bracket->id,
                        'round' => $round,
                        'match_number' => $pairIndex + 1,
                        'status' => 'scheduled',
                        'match_type' => $this->format === 'team' ? 'team' : 'individual',
                        'group_number' => $groupNumber,
                    ];
                    
                    if ($this->format === 'team') {
                        $matchData['team1_id'] = $pair[0]['id'];
                        $matchData['team2_id'] = $pair[1]['id'];
                    } else {
                        $matchData['player1_id'] = $pair[0]['id'];
                        $matchData['player2_id'] = $pair[1]['id'];
                    }
                    
                    GameMatch::create($matchData);
                }
            }
        }
    }

    /**
     * Create Swiss system round matches
     */
    protected function createSwissRoundMatches(Bracket $bracket, array $participants, int $round): void
    {
        $participantCount = count($participants);
        $matchesPerRound = intval($participantCount / 2);
        
        // For first round, use random pairing
        if ($round === 1) {
            shuffle($participants);
        } else {
            // For subsequent rounds, pair by score (would need to implement scoring system)
            $participants = $this->pairByScore($participants);
        }
        
        for ($i = 0; $i < $matchesPerRound; $i++) {
            $participant1 = $participants[$i * 2];
            $participant2 = $participants[$i * 2 + 1];
            
            $matchData = [
                'tournament_id' => $this->tournament->id,
                'bracket_id' => $bracket->id,
                'round' => $round,
                'match_number' => $i + 1,
                'status' => 'scheduled',
                'match_type' => $this->format === 'team' ? 'team' : 'individual',
            ];
            
            if ($this->format === 'team') {
                $matchData['team1_id'] = $participant1['id'];
                $matchData['team2_id'] = $participant2['id'];
            } else {
                $matchData['player1_id'] = $participant1['id'];
                $matchData['player2_id'] = $participant2['id'];
            }
            
            GameMatch::create($matchData);
        }
    }

    /**
     * Generate round robin pairs for a specific round
     */
    protected function generateRoundRobinPairs(array $participants, int $round): array
    {
        $count = count($participants);
        $pairs = [];
        
        // Round robin algorithm
        for ($i = 0; $i < $count / 2; $i++) {
            $pairs[] = [
                $participants[$i],
                $participants[$count - 1 - $i]
            ];
        }
        
        // Rotate participants for next round (except first participant)
        if ($round > 1) {
            $first = array_shift($participants);
            $last = array_pop($participants);
            array_unshift($participants, $last);
            array_unshift($participants, $first);
        }
        
        return $pairs;
    }

    /**
     * Create losers bracket matches (simplified implementation)
     */
    protected function createLosersBracketMatches(Bracket $bracket, int $participantCount): void
    {
        // This is a simplified implementation
        // Full double elimination would require more complex logic
        $losersRounds = ceil(log($participantCount, 2)) - 1;
        
        for ($round = 1; $round <= $losersRounds; $round++) {
            $matchesInRound = pow(2, $losersRounds - $round);
            
            for ($match = 1; $match <= $matchesInRound; $match++) {
                GameMatch::create([
                    'tournament_id' => $this->tournament->id,
                    'bracket_id' => $bracket->id,
                    'round' => $round + 100, // Offset to distinguish from winners bracket
                    'match_number' => $match,
                    'status' => 'scheduled',
                    'match_type' => $this->format === 'team' ? 'team' : 'individual',
                    'bracket_type' => 'losers',
                ]);
            }
        }
    }

    /**
     * Advance tournament to next round
     */
    public function advanceRound(Bracket $bracket, int $currentRound): array
    {
        $completedMatches = $bracket->gameMatches()
            ->where('round', $currentRound)
            ->where('status', 'completed')
            ->get();

        if ($completedMatches->isEmpty()) {
            throw new \Exception('No completed matches in current round');
        }

        $nextRound = $currentRound + 1;
        $winners = $this->extractWinners($completedMatches);
        
        // Create next round matches
        $this->createNextRoundMatches($bracket, $winners, $nextRound);
        
        return [
            'round' => $nextRound,
            'matches_created' => count($winners) / 2,
            'winners' => $winners,
        ];
    }

    /**
     * Extract winners from completed matches
     */
    protected function extractWinners(Collection $matches): array
    {
        $winners = [];
        
        foreach ($matches as $match) {
            if ($match->winner_id) {
                $winners[] = ['id' => $match->winner_id, 'type' => 'player'];
            } elseif ($match->winning_team_id) {
                $winners[] = ['id' => $match->winning_team_id, 'type' => 'team'];
            }
        }
        
        return $winners;
    }

    /**
     * Create matches for next round
     */
    protected function createNextRoundMatches(Bracket $bracket, array $winners, int $round): void
    {
        $matchNumber = 1;
        
        for ($i = 0; $i < count($winners); $i += 2) {
            $winner1 = $winners[$i];
            $winner2 = $winners[$i + 1] ?? null;
            
            $matchData = [
                'tournament_id' => $this->tournament->id,
                'bracket_id' => $bracket->id,
                'round' => $round,
                'match_number' => $matchNumber,
                'status' => 'scheduled',
                'match_type' => $this->format === 'team' ? 'team' : 'individual',
            ];
            
            if ($winner1['type'] === 'team') {
                $matchData['team1_id'] = $winner1['id'];
                if ($winner2) {
                    $matchData['team2_id'] = $winner2['id'];
                }
            } else {
                $matchData['player1_id'] = $winner1['id'];
                if ($winner2) {
                    $matchData['player2_id'] = $winner2['id'];
                }
            }
            
            GameMatch::create($matchData);
            $matchNumber++;
        }
    }

    /**
     * Get balanced positions for seeding
     */
    protected function getBalancedPositions(int $count): array
    {
        $positions = [];
        $powerOfTwo = pow(2, ceil(log($count, 2)));
        
        // Top seed positions in balanced bracket
        $topPositions = [0, $powerOfTwo - 1, $powerOfTwo / 2 - 1, $powerOfTwo / 2];
        
        // Add more positions for larger tournaments
        if ($count > 4) {
            $topPositions[] = $powerOfTwo / 4 - 1;
            $topPositions[] = $powerOfTwo / 4;
            $topPositions[] = 3 * $powerOfTwo / 4 - 1;
            $topPositions[] = 3 * $powerOfTwo / 4;
        }
        
        return array_slice($topPositions, 0, min(8, $count));
    }

    /**
     * Get participant ranking
     */
    protected function getParticipantRanking(array $participant): int
    {
        return $participant['ranking'] ?? 999;
    }

    /**
     * Get participant rating
     */
    protected function getParticipantRating(array $participant): int
    {
        return $participant['rating'] ?? 1000;
    }

    /**
     * Get bracket name
     */
    protected function getBracketName(string $type): string
    {
        return match($type) {
            'winners' => 'Winners Bracket',
            'losers' => 'Losers Bracket',
            'final' => 'Final Bracket',
            default => ucfirst($type) . ' Bracket',
        };
    }

    /**
     * Pair participants by score (for Swiss system)
     */
    protected function pairByScore(array $participants): array
    {
        // Sort by score (would need to implement scoring system)
        usort($participants, function ($a, $b) {
            $scoreA = $this->getParticipantScore($a);
            $scoreB = $this->getParticipantScore($b);
            return $scoreB <=> $scoreA;
        });
        
        return $participants;
    }

    /**
     * Get participant score
     */
    protected function getParticipantScore(array $participant): int
    {
        return $participant['score'] ?? 0;
    }
}
